/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include "world.h"
#include "spell.h"
#include <fstream>
#include <sstream>

spell_info_container* spell_info_container::handle = NULL;

spell_info_container* spell_info_container::get_handle() {
	if(!handle)
		handle = new spell_info_container();
	return handle;
}

bool spell::in_range(selectable *caster, sf::Vector2f pos, world &w) const
{
	mushroom *nearest_mushroom = w.nearest_mushroom(pos, SELECT_MUSHROOM_RANGE);
	mushroom *nearest_mushroom_growing = w.nearest_mushroom(pos, SELECT_MUSHROOM_RANGE, true);
	command_center *nearest_cc = w.nearest_cc(pos, SELECT_CC_RANGE);
	sf::Vector2f caster_center = caster->get_center();
	
	switch(spell_infos->at(t).target)
	{
		case spell_info::mushroom:
			if(nearest_mushroom 
					&& dist(nearest_mushroom->get_center(), caster_center) >= get_range(caster))
				return false;
			break;
		case spell_info::mushroom_growing:
			if(nearest_mushroom_growing
					&& dist(nearest_mushroom_growing->get_center(), caster_center) >= get_range(caster))
				return false;
			break;
		case spell_info::command_center:
			if(nearest_cc
					&& dist(nearest_cc->get_center(), caster_center) >= get_range(caster))
				return false;
			break;
		default:
			break;
	}
	if(need_target() && dist(pos, caster_center) >= get_range(caster))
		return false;
	return true;	
}


bool spell::can_cast(selectable *caster, sf::Vector2f pos, world &w) const
{
	sf::Vector2f caster_center = caster->get_center();

	mushroom *nearest_mushroom = w.nearest_mushroom(pos, SELECT_MUSHROOM_RANGE);
	mushroom *nearest_mushroom_growing = w.nearest_mushroom(pos, SELECT_MUSHROOM_RANGE, true);
	command_center *nearest_cc = w.nearest_cc(pos, SELECT_CC_RANGE);
	ant *nearest_ant = w.nearest_ant(pos, SELECT_ANT_RANGE);

	if(get_cur_cooldown() > 0)
		return false;
	
	if(!in_range(caster, pos, w))
		return false;

	if(pos.y < get_y_limit())
		return false;

	if(w.get_player().get_level() < get_req_level())
		return false;
	
	if(caster->get_energy() < get_energy_cost(caster))
		return false;

	switch(spell_infos->at(t).target)
	{
		case spell_info::ground:
				//no checking required
			break;
		case spell_info::mushroom:
			if(!nearest_mushroom 
					|| dist(nearest_mushroom->get_center(), caster_center) >= get_range(caster))
				return false;
			break;
		case spell_info::mushroom_growing:
			if(!nearest_mushroom_growing
					|| dist(nearest_mushroom_growing->get_center(), caster_center) >= get_range(caster))
				return false;
			break;
		case spell_info::command_center:
			if(!nearest_cc
					|| dist(nearest_cc->get_center(), caster_center) >= get_range(caster))
				return false;
			break;
		case spell_info::wall:
			if(w.get_map().get_tile(sf::Vector2i(pos.x / TILE_W, pos.y / TILE_H)).t != tile::wall)
				return false;
			break;
		case spell_info::ant:
			if(!nearest_ant)
				return false;
			break;
		case spell_info::none:
			break;
		default:
			std::cout << "Invalid spell target type" << std::endl;
			break;
	}
	
	if(need_target() && get_check_los()) {
		switch(spell_infos->at(t).target)
		{
			case spell_info::ground:
				if(!w.get_map().los(caster_center, pos))
				return false;
			break;
			case spell_info::mushroom:
				if(!w.get_map().los(caster_center, nearest_mushroom->get_center()))
				return false;
			break;
			case spell_info::mushroom_growing:	
				if(!w.get_map().los(caster_center, nearest_mushroom_growing->get_center()))
				return false;
			break;
			case spell_info::command_center:
				if(!w.get_map().los(caster_center, nearest_cc->get_center()))
				return false;
			break;
			case spell_info::ant:
				if(!w.get_map().los(caster_center, nearest_ant->get_center()))
				return false;
			break;
			case spell_info::wall:
				//too lazy to get the center of a tile
				if(!w.get_map().los(caster_center, pos))
				return false;
			break;
			default:
			break;
		}
	}
	

	return true;

}

spell::spell() : 
	id(-1), t(spell_count), cooldown(0), 
	spell_infos(spell_info_container::get_handle())
{
}
spell::spell(spell_type _t) : 
	id(-1), t(_t), cooldown(0), 
	spell_infos(spell_info_container::get_handle())
{
}

std::string spell::get_animation() const
{
	return spell_infos->at(t).animation;
}


std::vector<std::string> spell::get_desc(selectable *caster, bool show_hotkey) const
{
	std::vector<std::string> text;

	std::stringstream ss(std::stringstream::out);
	ss << spell_infos->at(t).name;
	
	text.push_back(ss.str());
	ss.str("");

	ss << spell_infos->at(t).desc;

	text.push_back(ss.str());
	ss.str("");

	if(get_req_level()>0) {
		ss << "Level requirement " << get_req_level();
		text.push_back(ss.str());
		ss.str("");
	}
	if(get_energy_cost(caster)>0) {
		ss << "Energy cost: " << get_energy_cost(caster);
		text.push_back(ss.str());
		ss.str("");
	}
	if(show_hotkey) {
		ss << "Hotkey: " << get_hotkey(); 
		text.push_back(ss.str());
		ss.str("");
	}

	return text;
}

void spell::update()
{
	cast_clock.step();
}

void spell::update_cooldown(const float &cd)
{
	if (cd > get_cur_cooldown())
	{
		cooldown = cd;
		cast_clock.reset();
	}
}

float spell::get_cur_cooldown() const 
{ 
	return std::max(0.0f, cooldown - cast_clock.get_time()); 
}

float spell::get_cooldown_phase() const
{
	if(cooldown == 0) 
		return 0.0f; 
	else 
		return get_cur_cooldown() / cooldown; 
}

float spell::get_s_cd() const 
{
	return spell_infos->at(t).s_cd;
}

float spell::get_g_cd() const 
{
	return spell_infos->at(t).g_cd;
}
		
int spell::get_energy_cost(selectable *caster) const
{
	command_center * caster_cc = dynamic_cast<command_center*>(caster);
	if(get_spell_type() == cc_grow_mushroom) {
		return (0.5+(caster_cc->mushrooms_growd)*0.5)
		 * spell_infos->at(t).e_cost;
	}
	if(get_spell_type() >= cc_fertilize_str && get_spell_type() <= cc_fertilize_dur) {
		return (caster_cc->get_tree().size() * spell_infos->at(t).e_cost);
	}
	return spell_infos->at(t).e_cost;
}

float spell::get_range(selectable *caster) const 
{
	mushroom *c_mushroom = dynamic_cast<mushroom*>(caster);
	if(c_mushroom) {
		return c_mushroom->get_spell_range() * spell_infos->at(t).range;
	}
	return spell_infos->at(t).range;
}

bool spell::get_check_los() const
{
	return spell_infos->at(t).check_los;
}

bool spell::need_target() const
{
	return spell_infos->at(t).target != spell_info::none;
}

char spell::get_hotkey() const
{
	return spell_infos->at(t).hotkey;
}

int spell::get_req_level() const
{
	return spell_infos->at(t).req_level;
}

int spell::get_y_limit() const
{
	return spell_infos->at(t).y_limit;
}

bool spell::available(selectable* caster, world& w)
{
	bool no_cd = get_cur_cooldown() <= 0;
	bool enough_energy = caster->get_energy() >= get_energy_cost(caster);
	bool enough_level = w.get_player().get_level() >= get_req_level();
	return no_cd && enough_energy && enough_level;
}

