/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include "object_map.h"
#include <cmath>
#include <iostream>
#include "mushroom.h"
#include "commandcenter.h"
#include "creature.h"
#include "trap.h"
#include <algorithm>
#include "map.h"

object_map::object_map(const int _w, const int _h) : w(_w), h(_h)
{
	std::vector<std::list<object*> > tmp;
	tmp.resize(w);
	map.assign(h, tmp);
}

void object_map::add(object *obj)
{
	if(obj->get_w() == 0 || obj->get_h() == 0)
		throw std::invalid_argument("object_map::add: obj_w or obj_h == 0");

	int left_x = obj->get_x() / OBJMAP_TILE_W;
	int right_x = (obj->get_x() + obj->get_w() - 1) / OBJMAP_TILE_W;
	int up_y = obj->get_y() / OBJMAP_TILE_H;
	int down_y = (obj->get_y() + obj->get_h() - 1) / OBJMAP_TILE_H;

	for(int y = up_y; y <= down_y; y++) {
		for(int x = left_x; x <= right_x; x++) {
			if(x < 0 || x >= w || y < 0 || y >= h)
				continue;
			map[y][x].push_back(obj);
		}
	}
}

void object_map::remove(object *obj)
{
	if(obj->get_w() == 0 || obj->get_h() == 0)
		throw std::invalid_argument("object_map::add: obj_w or obj_h == 0");

	int left_x = obj->get_x() / OBJMAP_TILE_W;
	int right_x = (obj->get_x() + obj->get_w() - 1) / OBJMAP_TILE_W;
	int up_y = obj->get_y() / OBJMAP_TILE_H;
	int down_y = (obj->get_y() + obj->get_h() - 1) / OBJMAP_TILE_H;

	for(int y = up_y; y <= down_y; y++) {
		for(int x = left_x; x <= right_x; x++) {
			if(x < 0 || x >= w || y < 0 || y >= h)
				continue;
			map[y][x].remove(obj);
		}
	}
}

std::list<object*> object_map::get_objects(const object *obj)
{
	std::list<object*> ret;
	if(obj->get_w() == 0 || obj->get_h() == 0)
		throw std::invalid_argument("object_map::add: obj_w or obj_h == 0");

	int left_x = obj->get_x() / OBJMAP_TILE_W;
	int right_x = (obj->get_x() + obj->get_w() - 1) / OBJMAP_TILE_W;
	int up_y = obj->get_y() / OBJMAP_TILE_H;
	int down_y = (obj->get_y() + obj->get_h() - 1) / OBJMAP_TILE_H;


	for(int y = up_y; y <= down_y; y++) {
		for(int x = left_x; x <= right_x; x++) {
			if(x < 0 || x >= w || y < 0 || y >= h)
				continue;
			std::list<object*> l = map[y][x];
			ret.insert(ret.end(), l.begin(), l.end());
		}
	}
	return ret;
}

bool object_map::collision(const object *obj)
{
	std::list<object*> check = get_objects(obj);
	std::list<object*>::const_iterator iter;

	for(iter = check.begin(); iter != check.end(); iter++) {
		if(obj->collision(*(*iter)))
			return true;

	}
	return false;
}

bool object_map::out_of_bounds(sf::Vector2f p)
{
	return (p.x < 0 || p.x >= OBJMAP_TILE_W * w || p.y < 0 || p.y >= OBJMAP_TILE_H * h);
}

sf::Vector2f object_map::ant_target_location()
{
	int area_w = 4, area_h = 4;
	int areas_x = ceil((double)w/area_w);
	int areas_y = ceil((double)h/area_h);

	for(int area_x = areas_x-1; area_x > -1; area_x--) {
		int best_y = -1;
		int best_targets = 1000;
		int area_y;
		//don't try to go up there!
		for(area_y = 2; area_y < areas_y; area_y++) {
			int cur_targets = ant_targets_in_area(area_x *area_w, area_y*area_h, area_w, area_h);
			if(cur_targets != 0 && cur_targets < best_targets) {
				best_targets = cur_targets;
				best_y = area_y;
			}
		}
		if(best_y != -1) {
			return sf::Vector2f( (area_x * area_w + area_w/2) * OBJMAP_TILE_W,
							(best_y * area_h + area_h/2) * OBJMAP_TILE_H);
		}
	}
	return sf::Vector2f(-1.0f, -1.0f);
}

int object_map::ant_targets_in_area(int x, int y, int area_w, int area_h) 
{
	int ret = 0;
	for (int y2 = 0; y2<area_h; y2++) {
		for(int x2 = 0; x2<area_w; x2++) {
			if(x+x2 < 0 || x+x2 >= w || y+y2 < 0 || y+y2 >= h)
				continue;
			std::list<object*> l = map[y+y2][x+x2];
			std::list<object*>::iterator it; 
			for (it = l.begin(); it != l.end(); it++) {
				mushroom *m = dynamic_cast<mushroom*>(*it);
				command_center *cc = dynamic_cast<command_center*>(*it);
				if(m && !m->is_invisible())
					ret++;
				if(cc && !cc->is_invisible())
					ret++;
			}
		}
	}
	return ret;
}

int object_map::objs_on_vector(sf::Vector2f start, sf::Vector2f end, object* dis)
{
	int ret = 0;
	sf::Vector2f dir = end - start;
	sf::Vector2f pos = start;

	int sx = sgn(dir.x);
	int sy = sgn(dir.y);

	while(coor_to_tile(pos) != coor_to_tile(end)) {
		std::list<object*> &objects = get_tile(coor_to_tile(pos));
		std::list<object*>::iterator it;
		for(it = objects.begin(); it != objects.end(); it++) {
			if(!((*it)==dis || dynamic_cast<creature*>(*it) || dynamic_cast<trap*>(*it)))
				ret++;
		}
		
		if(out_of_bounds(pos)) {
			return ret;
		}

		double bx = nearest_next_border(pos.x, sx, OBJMAP_TILE_W);
		double by = nearest_next_border(pos.y, sy, OBJMAP_TILE_H);

		double dx = bx - pos.x;
		double dy = by - pos.y;

		if(sy && sx) {
			if(fabs(dx / dy) > fabs(dir.x / dir.y)) {
				pos.y = by;
				pos.x += dir.x / dir.y * dy;
			} else {
				pos.x = bx;
				pos.y += dir.y / dir.x * dx;
			}
		} else {
			if(sy == 0)
				pos.x = bx;
			else if(sx == 0)
				pos.y = by;
		}
	}
	return ret;
}

sf::Vector2i object_map::coor_to_tile(sf::Vector2f pos)
{
	if(pos.x < 0) 
		pos.x = 0;
	if(pos.y < 0)
		pos.y = 0;
	if(pos.y >= h * OBJMAP_TILE_H)
		pos.y = h * OBJMAP_TILE_H - 1;
	if(pos.x >= w * OBJMAP_TILE_W)
		pos.x = w * OBJMAP_TILE_W - 1;

	return sf::Vector2i(pos.x / TILE_W, pos.y / TILE_H);
}

double object_map::nearest_next_border(double x, int sign, double w)
{
	if(sign == 0)
		return x;
	if(fmod(x, w) < 0.000001)
		return x + sign * w;
	return x - fmod(x, w) + ((sign > 0) ? w : -1.0f);
}
