/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include "mushroom.h"
#include "settings.h"
#include "creature.h"
#include "world.h"
#include <sstream>
#include <algorithm>
#ifndef NO_GRAPHICS
#include "game.h"
#endif

bool operator==(const mushroom::mushroom_type &a, const mushroom::mushroom_attr &b)
{
	return (int)a == (int)b;
}
bool operator==(const mushroom::mushroom_attr &a, const mushroom::mushroom_type &b)
{
	return (int)a == (int)b;
}

void mushroom::calculate_abilities()
{
	if(attributes[type] + attribute_modifier > 100)
	{
		attribute_modifier = 100 - attributes[type];
	}
	
	double s = get_attribute(str);
	double m = get_attribute(mana);
	double d = get_attribute(dur);
	max_hp = 20 + d*0.3;
	hp_regen = d*0.008;
	max_energy = 30 + m;
	energy_regen = m*0.05;
	spell_range = 80+s*0.9;
	switch (type)
	{
		case m_stem:
		{
		}
		break;
		case m_attack:
		{
			atk_skill.set_atk_boost(s / 2);
			atk_skill.set_range_boost(s / 2);
		}
		break;
		case m_spore:
		{
			max_energy = 35 + m*2;
			energy_regen = m*0.075;
		}
		break;
		case m_tank:
		{
			atk_skill.set_atk_boost(s / 3);
			hp_regen = d*0.012;
			max_hp = 30 + d*0.8;
		}
		break;
		default:
		break;
	}
	calculate_color();
}

mushroom::mushroom(animation_loader *l) :
	selectable(l?&l->get_animation("mushroom"):0), type(m_stem), 
	atk_skill(attack::none_attack), shoot_laser(false), attribute_modifier(5),
	energy_regen(0), hp_regen(0), spell_range(0), near_grow(false), saved(false)
{
	attributes[0] = 0;
	attributes[1] = 0;
	attributes[2] = 0;

	set_w(38);
	set_h(38);
			
	set_mode("small");
#ifndef NO_GRAPHICS
	if(anim)
		anim->set_dynamic_color(sf::Color::Black);
#endif
	hp = 1;
	max_hp = 1;
	energy = 0.01;
	max_energy = 0.01;

	grow(m_stem);
}

mushroom::~mushroom()
{

}

bool mushroom::to_removed()
{
	if(dead_timer.get_time() >= 1.0)
		return true;
	else
		return false;
}

bool mushroom::is_saved()
{
	if(saved || !is_dead()) {
		return true;
	} else {
		if(growing)
			growing = false;
		else
			set_mode("die", false);
		saved = true;
		return false;
	}
}

void mushroom::grow(mushroom_type t)
{
	spells.clear();
	if(anim)
		anim->manual();
	grow_timer.reset();
	growing = true;
	near_grow = false;

	grow_type = t;
	switch(t) {
		case m_stem:
			grow_time = 7.0f;
			calculate_abilities();
			break;
		case m_attack:
			grow_time = 20.0f;
			break;
		case m_spore:
			grow_time = 10.0f;
			break;
		case m_tank:
			grow_time = 10.0f;
			break;
		default:
			break;
	}
}
		
void mushroom::disconnect(mushroom *m)
{
	std::list<mushroom*>::iterator it;
	for(std::list<mushroom*>::iterator it = connections.begin(); it != connections.end(); it++) {
		if(*it == m) {
			connections.erase(it);
			it--;
		}
	}	
}

bool mushroom::is_connected(mushroom *m)
{
	std::list<mushroom*>::iterator it;
	for(it = connections.begin(); it != connections.end(); it++) {
		if(*it == m) {
			return true;
		}
	}	
	return false;
}
void mushroom::connect(mushroom *m)
{ 
	if(!is_connected(m))
		connections.push_back(m); 
}
void mushroom::upgrade(mushroom_type t)
{
	type = t;
	if(anim)
		anim->automatic();
	set_mode("stand");

	switch (type) {
		case m_stem:
		{
			spells.push_back(spell(spell::mushroom_connect));
			spells.push_back(spell(spell::mushroom_upg_to_attack));
			spells.push_back(spell(spell::mushroom_upg_to_spore));
			spells.push_back(spell(spell::mushroom_upg_to_tank));
		}
		break;
		case m_attack:
		{
			atk_skill = attack_skill(attack::mushroom_attack);
			atk_skill.reset_timer();

			spells.clear();
			spells.push_back(spell(spell::mushroom_connect));
			spells.push_back(spell(spell::mushroom_attack_wall));
			spells.push_back(spell(spell::mushroom_target_ant));
			spells.push_back(spell(spell::mushroom_paralyse));
		}  
		break;
		case m_spore:
		{
			spells.clear();
			spells.push_back(spell(spell::mushroom_connect));
			spells.push_back(spell(spell::mushroom_multiply));
			spells.push_back(spell(spell::mushroom_give_energy));
		}
		break;
		case m_tank:
		{
			atk_skill = attack_skill(attack::tank_attack);
			atk_skill.reset_timer();

			spells.clear();
			spells.push_back(spell(spell::mushroom_connect));
			spells.push_back(spell(spell::mushroom_heal));
			spells.push_back(spell(spell::mushroom_taunt));
		}
		break;
		default:
		break;
	}
	calc_spell_ids();
	int old_max_hp = max_hp;
	calculate_abilities();
	if(type != m_stem) 
		add_hp(max_hp - old_max_hp);
}

void mushroom::pause()
{
	selectable::pause();
	grow_timer.pause();
}

#ifndef NO_GRAPHICS
void mushroom::draw(sf::RenderTarget &target)
{
	object::draw(target);
	
	Gui::drawFilledRectangle(target, Gui::Rect(get_x(), get_y() - 10, get_w(), 3), hp / max_hp, sf::Color::Red);
	
	if(growing) {
		Gui::drawFilledRectangle(target, Gui::Rect(get_x(), get_y() - 5, get_w(), 3),
				get_grow_timer() / get_grow_time(), sf::Color::Green);
	} else {
		Gui::drawFilledRectangle(target, Gui::Rect(get_x(), get_y() - 5, get_w(), 3),
				get_energy() / get_max_energy(), sf::Color::Blue);
	}
}
#endif

bool mushroom::collision(const object &o) const
{
	const creature *c;
	if((c = dynamic_cast<const creature*>(&o))) {
		if(c->get_type() == creature::griffin)
			return false;
	}

	return object::collision(o);
}

void mushroom::set_attributes(int _str, int _mana, int _dur)
{
	attributes[str] = _str;
	attributes[mana] = _mana;
	attributes[dur] = _dur;

	calculate_abilities();
}

int mushroom::get_attribute(mushroom_attr attr)
{
	if (type == attr) {
		return attributes[attr] + attribute_modifier;
	} else {
		return attributes[attr];
	}
}

void mushroom::change_modifier(int dm)
{
	attribute_modifier += dm;
	attribute_modifier = std::max(attribute_modifier, 0);

	calculate_abilities();
}

std::string mushroom::type_string()
{
	if(growing)
		return "Growing";
	switch (type)
	{
		case m_stem:
			return "Type: Stem";
		break;
		case m_attack:
			return "Type: Attack";
		break;
		case m_tank:
			return "Type: Tank";
		break;
		case m_spore:
			return "Type: Spore";
		break;
	}
	return "";
}

std::string mushroom::str_string()
{
	std::stringstream ss(std::stringstream::out);
	ss << "Strength: ";
	if(type != m_attack)
		ss << attributes[str];
	else
		ss << attributes[str] << " + " << attribute_modifier << " = " << attributes[str]+attribute_modifier;
	return ss.str();
}

std::string  mushroom::mana_string()
{
	std::stringstream ss(std::stringstream::out);
	ss << "Mana: ";
	if(type != m_spore)
		ss << attributes[mana];
	else
		ss << attributes[mana] << " + " << attribute_modifier << " = " << attributes[mana]+attribute_modifier;
	return ss.str();
}

std::string  mushroom::dur_string()
{
	std::stringstream ss(std::stringstream::out);
	ss << "Durabi: ";
	if(type != m_tank)
		ss << attributes[dur];
	else
		ss << attributes[dur] << " + " << attribute_modifier << " = " << attributes[dur]+attribute_modifier;
	return ss.str();
}

void mushroom::calculate_color()
{
	if(type != m_stem) {
		float colors[3] = {0.1f,0.1f,0.1f};	//0.1
		colors[type] += 0.3;				//0.4
		float attrib = ((float)(attributes[type]+attribute_modifier))/100;
		colors[0] += 0.1 * attrib;			//0.5
		colors[1] += 0.1 * attrib;
		colors[2] += 0.1 * attrib;
		colors[type] += 0.5 * attrib;		//1.0
#ifndef NO_GRAPHICS
		if(anim)
			anim->set_dynamic_color(sf::Color(colors[0]*255, colors[1]*255, colors[2]*255));
#endif
	}
}

void mushroom::update(world &w)
{
	selectable::update(w);

	if(is_dead()) {
		dead_timer.step();
		return;
	}

	atk_skill.update(false);
	if(growing) {
		grow_timer.unpause();
		grow_timer.step();
		if(anim) {
			anim->set_frame((float)anim->get_frame_count() * grow_timer.get_time() / grow_time);
		}
	
		//stem gets hp while growing
		if(grow_type == m_stem) {
			double grow_length = grow_time / DT;
			//1 hp is given at constructor
			add_hp((max_hp-1)/grow_length);
		}

		if(grow_timer.get_time() > grow_time) {
			growing = false;
			w.handler.grow(this, grow_type);
			upgrade(grow_type);
		} else if(grow_timer.get_time() + 0.3f > grow_time) {
			if(!near_grow)
				w.handler.near_grow(this, grow_type);
			near_grow = true;
		}
	} else {
		add_energy(energy_regen * DT);
		add_hp(hp_regen * DT);

		if(type == m_attack || type == m_tank) {
			attack_enemy(w);
		}
	}
}

void mushroom::wind_damage(double rel_num, int blow_count)
{
	const double wind_dmg = 0.25;
	double dmg = wind_dmg * DT / pow(16, -rel_num);
	if(type == m_tank)
		dmg *= 0.5;
	blow_count = std::min(blow_count, 3);
	add_hp(-dmg * (((double)blow_count*0.7)));
}

#ifndef NO_GRAPHICS

void mushroom::draw_attack_anim(sf::RenderTarget &target, particle_engine &e)
{
	if(atk_skill.get_type() == attack::tank_attack) {
		if(atk_skill.has_attacked() &&
				atk_skill.get_timer() * 2 < atk_skill.get_reload_time()) {
			const int waves = 2;
			double d = atk_skill.get_timer() * 3 / atk_skill.get_reload_time();
			for(int a = 0; a < waves; a++) {
				double w = 1.2 * d - 0.2 * double(a) / (waves - 1);
				if(0 < w && w < 1) {
					sf::Shape cr = sf::Shape::Circle(get_center(), (atk_skill.get_range() - 30) * w * 1.1,
							sf::Color(), 1.0f, sf::Color::Blue);
					cr.EnableFill(false);
					target.Draw(cr);
				}
			}
		}
	} else if(atk_skill.get_type() == attack::mushroom_attack && shoot_laser) {
		shoot_laser = false;
		laser las;
		las.shoot(get_center(), laser_target, e);
	}
}

#endif

void mushroom::attack_enemy(world &w)
{
	if(!atk_skill.can_attack())
		return;

	bool multi = atk_skill.is_multi();
	attack atk;
	bool attacked = false;

	ant *target_ant = w.nearest_ant(get_center(), atk_skill.get_range());

	if(preferred_target || multi) {
		for(std::vector<ant*>::iterator it = w.get_ants().begin(); it != w.get_ants().end(); it++) {
			if(dist(get_center(), (*it)->get_center()) < atk_skill.get_range()) {
				if(!multi && *it == preferred_target) {
					target_ant = dynamic_cast<ant*>(*it);
				} else if(multi) { 
					target_ant = dynamic_cast<ant*>(*it);
					if(!attacked)
						atk = atk_skill.do_attack();
					double delay = dist(get_pos(), target_ant->get_pos()) / atk_skill.get_range() / 1.1;
					atk.set_delay(delay);
					w.handler.hit(this, target_ant, atk);
					target_ant->damage(w, atk);
				}	
			}
		}
	}

	if(target_ant && !multi) {
		attack atk = atk_skill.do_attack();
		double delay = dist(get_pos(), target_ant->get_pos()) / (atk_skill.get_range() * 1.4);
		atk.set_delay(delay);
		w.handler.hit(this, target_ant, atk);
		target_ant->damage(w, atk);
		shoot_laser = true;
		laser_target = target_ant->get_center();
	}
}

void mushroom::stop_upgrade()
{
	grow_time = 0;
	growing = false;
}


