/***************************************************************************
*   Copyright (C) 2009-2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#ifndef _GUI_H
#define _GUI_H
#include <SFML/System.hpp>
#include <SFML/Graphics.hpp>
#include <SFML/Window.hpp>
#include <string>
#include <iostream>
#include <vector>

namespace Gui {
	using sf::Uint8;

	enum MouseButton {
		BUTTON1,
		BUTTON2,
		BUTTON3,
		BUTTON4,
		BUTTON5
	};


	MouseButton MouseConv(sf::Mouse::Button);

	unsigned int TextWidth(const std::string, const sf::Font&, float size = 30);
	unsigned int TextHeight(const std::string, const sf::Font&);

	typedef sf::Event::KeyEvent Key;
	typedef unsigned int ID;
	const ID ID_NONE = 0;

	class Coor;
	class Rect;
	std::ostream & operator<<(std::ostream & str, Gui::Rect r);

	class Rect {
		public:
			Rect() : x(0), y(0), w(0), h(0) { }
			Rect(int _x, int _y) : x(_x), y(_y), w(0), h(0) { }
			Rect(int _x, int _y, int _w, int _h) : x(_x), y(_y), w(std::max(_w, 0)), h(std::max(_h, 0)) { }
			Rect(const Coor &c) { *this = c; }
			const Rect & operator=(const Coor &c);
			const Rect & operator=(const Rect &c);
			sf::IntRect GetSFMLRect() const { 
				return sf::IntRect(x, y, x+w, y+h);
			}

			int x, y;
			int w, h;
	};

	void drawFilledRectangle(sf::RenderTarget &target, const Rect &r, double p, sf::Color col = sf::Color());

	class Coor {
		public:
			Coor() : x(0), y(0) { }
			Coor(const Rect & r) : x(r.x), y(r.y) { }
			Coor(int _x, int _y) : x(_x), y(_y) { }
			const Coor operator+(const Coor &c) const {
				return Coor(x + c.x, y + c.y);
			}
			sf::Vector2f sfVector() const { return sf::Vector2f(x, y); }
			bool InRect(const Rect &r) const {
				if(x >= r.x && x <= r.x + r.w && y >= r.y && y <= r.y + r.h)
					return true;
				else
					return false;
			}
			int x, y;
	};


	class ActionListener
	{
		public:
			virtual ~ActionListener() = 0;
			virtual void KeyDown(ID srcid, Key k) { }
			virtual void ScrollSelect(ID srcid, int selid) { }
			virtual void ScrollChSelect(ID srcid, int selid) { }
			virtual void SetText(ID srcid, std::string text) { }
			virtual void ButtonClick(ID srcid) { }
			virtual void MouseOver(ID srcid) { }
			virtual void MouseClick(ID srcid, int x, int y, MouseButton m) { }
			virtual void Draw(ID srcid, sf::RenderTarget &) { }
		private:

	};

	class Container;

	/*
	 * Describe a tooltip
	 */ 
	struct Tooltip 
	{
		std::vector<std::string> str;
		sf::Color color;
		sf::Color bgcolor;
	};

	class Object : public sf::Drawable
	{
		public:
			/*
			 * Set and get object's dimensions
			 * NOTE: {Get,Set}Rel{X,Y}, the versions without rel 
			 * collide with drawable's methods
			 */
			void SetRect(const Rect & _rect) { rect = _rect; ResizeAction(); }
			void SetCoor(const Coor & _coor) { rect = _coor; }
			void SetRelX(int _x) { rect.x = _x; }
			void SetRelY(int _y) { rect.y = _y; }
			void SetW(int _w) { rect.w = _w; }
			void SetH(int _h) { rect.h = _h; }
			const Rect GetRect() const { return rect; }
			const int GetRelX() const { return rect.x; }
			const int GetRelY() const { return rect.y; }
			const int GetW() const { return rect.w; }
			const int GetH() const { return rect.h; }


			/*
			 * Get object's unique id
			 */
			const ID GetID() { return id; }

			/*
			 * Get object's absolute coordinates (add parent's relative coords
			 * to objects relative coords)
			 */
			const Coor GetAbsCoor() const;

			const Object & operator=(const Object &o);

			Object(Rect r) : 
					skip_focus(false), rect(r), hide(false), focused(false), 
					drawbg(false), bwidth(0), id(++id_counter),	font(font_default),
					listener(NULL), parent(NULL), resize_handling(false), enabled(true)
			{ 
				count++; 
			}

			virtual ~Object();

			/*
			 * Actions that objects can get
			 */
			virtual void MouseDownAction(int x, int y, MouseButton m) { }
			virtual void MouseUpAction(int x, int y, MouseButton m) { }
			virtual void MouseMotionAction(int x, int y, int dx, int dy, const sf::Input&);
			virtual void KeyDownAction(Key k);
			virtual void KeyUpAction(Key k) { }
			virtual void FocusAction() { focused = true; }
			virtual void UnFocusAction() { focused = false; }
			virtual void TextAction(sf::Uint32 k) { }

			/*
			 * Resize a object, actually calls ResizeHandle to avoid endless loops
			 */
			void ResizeAction();

			/* 
			 * Set focus to this object
			 */
			void Focus();
			/*
			 * Raise the object to the top of parent container (is drawn on everything else)
			 */
			void Raise();

			bool IsFocused() { return focused; }

			void SetColor(sf::Color _color) {  color = _color; }
			sf::Color GetColor() { return color; }
			void SetBorderColor(sf::Color _color) {  bcolor = _color; }
			sf::Color GetBorderColor() { return bcolor; }
			void SetFontColor(sf::Color _color) {  font_color = _color; }
			sf::Color GetFontColor() { return font_color; }

			int GetBorderWidth() { return bwidth; }
			void SetBorderWidth(int _bwidth) { bwidth = _bwidth; }

			/*
			 * Expand object so that it's center stays the same
			 */
			void Expand(int dx, int dy) const;

			/*
			 * Get ActionListener recursively (if the object doesn't have one, ask its 
			 * parent which in turn asks its parent and so on)
			 */
			ActionListener *GetListener() const;

			/*
			 * Set object's action listener
			 */
			void SetActionListener(ActionListener *_listener) { listener = _listener; }
			/*
			 * Set parent to a container, use parent's Add method instead
			 */
			void SetParent(Container *_parent) { parent = _parent; }

			/*
			 * Set object's font
			 */
			void SetFont(sf::Font &_font) { font = _font; }
			static void SetDefaultFont(sf::Font &_font_default) { font_default = _font_default; }

			/*
			 * Should the object be skipped when giving focus to the element
			 * For example mages probably don't want focus but buttons want
			 */ 
			bool skip_focus;
			/*
			 * Should we draw object's background
			 */
			void DrawBG(bool b) { drawbg = b; }

			/* 
			 * Is the object drawn or being hidden?
			 */
			void Hide() { hide = true; }
			void Show() { hide = false; }
			void SetHidden(bool b) { hide = b; }
			bool IsHidden() { return hide; }

			/*
			 * Handle some logic things
			 */
			virtual void HandleLogic();

			bool IsEnabled() const { return enabled; }
			void SetEnabled(bool value) { enabled = value; }

			/*
			 * Is the object under the mouse?
			 */
			bool IsUnderMouse() const;

		protected:

			/*
			 * Function for drawing background
			 */
			void DrawBackground(sf::RenderTarget &target, const Coor coor = Coor()) const;

			/*
			 * Actual resize handling for elements
			 */
			virtual void ResizeHandler() { }
			/*
			 * Draw borders around the element
			 */
			void DrawBorder(sf::RenderTarget &target, const sf::Color _color, int border) const;
			void DrawBorder(sf::RenderTarget &target, const sf::Color _color) const {
				DrawBorder(target, _color, bwidth);
			}
			void DrawBorder(sf::RenderTarget &target) const {
				DrawBorder(target, bcolor);
			}

			virtual void Render(sf::RenderTarget&) const {}

			/*
			 * Get mouse position by asking recursively
			 */
			virtual Coor GetMousePos() const;

			virtual void SetTip(Tooltip &t);
			virtual void UnsetTip();

			/*
			 * Object's dimensions
			 */
			mutable Rect rect;

			/*
			 * Is the object hidden
			 */
			bool hide;
			/*
			 * Is the object currently focused
			 */
			bool focused;
			/*
			 * Should background be drawn
			 */
			mutable bool drawbg;
			/*
			 * Color settings
			 */
			sf::Color color;
			sf::Color bcolor;
			sf::Color font_color;

			/*
			 * Border width
			 */
			int bwidth;
			/*
			 * Object's unique id
			 */ 
			ID id;

			sf::Font &font;
			ActionListener *listener;
			Container *parent;

			/*
			 * Default font which is used when an object is constructed
			 */
			static sf::Font &font_default;
		private:
			/*
			 * id count to get unique ids
			 */
			static ID id_counter;
			/*
			 * How many elements do there actually exists
			 */
			static int count;

			/*
			 * Are we handling resize currently (this way we can notice
			 * potential endless loops)
			 */
			bool resize_handling;

			bool enabled;
	};

	/*
	 * Set the objects default font
	 */
	void SetFont(sf::Font &_font_default);

	typedef std::vector<Object*>::iterator object_iter;
	typedef std::vector<Object*>::reverse_iterator object_iter_r;

	typedef std::vector<Object*>::const_iterator object_c_iter;
	typedef std::vector<Object*>::const_reverse_iterator object_c_iter_r;

	class TextList;

	/*
	 * Container is a base class for things which contain objects
	 */
	class Container : public Object
	{
			public:
					/*
					 * Add an object to a container
					 */
					virtual Container *Add(Object *o);
					virtual Container *Add(Object *o, int x, int y);

					virtual Container *AddCenter(Object *o);

					const std::vector<Object*> & GetChilds() const { return childs; }

					Container(Rect r, sf::Color _color = sf::Color()) : 
							Object(r), mousex(0), mousey(0), tip(NULL), 
							lock(ID_NONE), focus(ID_NONE)
					{ color = _color; } 
					virtual ~Container();

					/*
					 * Mouse events that should be called for the top-level element
					 */
					void MouseDown(int x, int y, MouseButton m);
					void MouseUp(int x, int y, MouseButton m);
					void MouseMotion(int x, int y, const sf::Input&);

					virtual Coor GetMousePos() const;

					virtual void MouseDownAction(int x, int y, MouseButton m);
					virtual void MouseUpAction(int x, int y, MouseButton m);
					virtual void MouseMotionAction(int x, int y, int dx, int dy, const sf::Input&);
					virtual void KeyDownAction(Key k);
					virtual void FocusAction();
					virtual void UnFocusAction();
					virtual void TextAction(sf::Uint32 k);
					void ResizeHandler();

					/*
					 * Resize the container so that all object fit inside
					 */
					void Zip();

					/*
					 * Get object with ID id, find it recursively
					 */
					Object *GetObject(ID id);

					/*
					 * Should the container be focused as every other object
					 */
					virtual bool FocusAsObject() { return false; }
					/*
					 * Move focus to next or previous object
					 */
					virtual ID FocusPrev(bool focusparent = true);
					virtual ID FocusNext(bool focusparent = true);
					ID GetFocus() { return focus; }

					/*
					 * Remove all objects and delete them
					 */
					virtual void Clear();
					/*
					 * Remove all objects without deleting them
					 */
					virtual void ClearWithoutDestroy();
					/*
					 * Remove one element
					 */

					// FIXME: can these crash the program with focusit?
					void DestroyID(ID id);

					void DeleteID(ID id);
					/*
					 * Set focus to a certain object
					 */
					void SetFocus(ID id);
					/*
					 * Called when the focus is moved away
					 */
					void UnFocus();
					void FocusNone() {
						focus = ID_NONE;
					}
					/*
					 * Raise object, see objects Raise
					 */
					void RaiseID(ID id);

					/*
					 * Lock to one object so that no events go to others
					 */
					void Lock(ID id) { 
						lock = id;
					}
					/*
					 * Handle some logic things
					 */
					virtual void HandleLogic();

					void SetTip(Tooltip &t);
					void UnsetTip();
			protected:
					int mousex, mousey;
					virtual void Render(sf::RenderTarget &target) const;
					void RenderCoor(sf::RenderTarget &target, const Coor coor = Coor(0,0)) const;

					TextList *tip;

					std::vector<Object*> childs;
					ID lock;
					ID focus;
					/*
					 * Iterator to the focused object
					 */
					object_iter focusit;
	};


	class TextLabel : public Object
	{
			public:
					TextLabel(Rect r = Rect(), std::string _label = "") : Object(r), label(_label), size(30) { ResizeAction(); skip_focus = true; }
					TextLabel(std::string _label) : Object(Rect()), label(_label), size(30) { ResizeAction(); skip_focus = true; }

					void SetLabel(std::string _label) { label = _label; ResizeAction(); }
					void SetText(std::string t) { SetLabel(t); }
					void SetSize(float _size) { size = _size; }
			protected:
					virtual void Render(sf::RenderTarget &target) const { RenderCoor(target); }
					void RenderCoor(sf::RenderTarget &target, const Coor c = Coor(0,0)) const;
					std::string label;
					float size;

					virtual void ResizeHandler();
	};

	class Button : public TextLabel
	{
			public:
					Button(Rect r = Rect(), std::string _label = ""); 
					Button(std::string _label); 
					virtual void MouseDownAction(int x, int y, MouseButton m); 

					virtual void KeyDownAction(Key k);
					void SetHilightUnderMouse(bool b) 
					{
						hilight_under_mouse = b;
					}

					void ResizeHandler();
			protected:
					void Render(sf::RenderTarget &target) const;
			private:
					bool hilight_under_mouse;
					Coor start_size;
	};

	class Image : public Object
	{
			public:
					Image(Rect = Rect());
					Image(const sf::Image &, Rect = Rect());
					void SetImage(const sf::Image &_img) { img.SetImage(_img); ResizeAction(); }
			protected:
					void RenderCoor(sf::RenderTarget &target, Coor c = Coor()) const;
					void Render(sf::RenderTarget &target) const;
					sf::Sprite img;

					void ResizeHandler();
	};

	class ImgButton : public Image
	{
		public:
			ImgButton(Rect r = Rect());
			ImgButton(const sf::Image &img, Rect r = Rect());
			virtual ~ImgButton();
			virtual void MouseDownAction(int x, int y, MouseButton m);
			virtual void KeyDownAction(Key k);
			virtual void HandleLogic();
			void SetToolTip(std::vector<std::string> str, sf::Color color = sf::Color(), 
					sf::Color bgcolor = sf::Color());
			void ResizeHandler() { }
		protected:
			sf::Clock under_mouse;

			bool tooltip_on, tooltip_shown;
			Tooltip tooltip;

			virtual void Render(sf::RenderTarget &target) const;
	};


	class HorizontalList : public Container
	{
		public:
			HorizontalList(Rect r = Rect(), int _pad = 10);
			Container *Add(Object *o);
			Container *AddCenter(Object *o);

			void Clear() { width = 0; Container::Clear(); }
		private:
			int pad;
			int width;
			
			void ResizeHandler();
	};
	
	class VerticalList : public Container
	{
		public:
			VerticalList(Rect r = Rect(), int _pad = 10);
			Container *Add(Object *o);
			Container *AddCenter(Object *o);

			void Clear() { height = 0; Container::Clear(); }
		private:
			int pad;
			int height;
			
			void ResizeHandler();
	};

	class TextList : public Object
	{
		public:
			TextList(Rect r = Rect(), std::string _text = "", int _pad = 5) ;

			// Remember to set text again after setting a new font
			void SetText(std::string _text);
			void SetText(std::vector<std::string> _text);
			void MouseDownAction(int x, int y, MouseButton m);
			void KeyDownAction(Key k);
			virtual void ScrollDown();
			virtual void ScrollUp();
			void SetMaxTexts(int max) { max_texts = max; }
			void SetSize(float _size) { size = _size; }
		protected:
			void Render(sf::RenderTarget &target) const;
			std::vector<std::string> WordWrap(std::string text, const sf::Font &font, const unsigned int maxw);
			virtual void ResizeHandler();

			std::vector<std::string> text;
			int pad;
			int scroll;
			int max_texts;
			float size;
	};

	class ScrollSelection : public Container
	{
		public:
			ScrollSelection(Rect r, sf::Image &select, sf::Color color = sf::Color(), int _pad = 5);

			Container *Add(Object *o);

			void MouseDownAction(int x, int y, MouseButton m);
			void KeyDownAction(Key k);

			int GetIndex(int x, int y);
			void ScrollDown();
			void ScrollUp();
			ID FocusNext(bool) { return ID_NONE; }
			ID FocusPrev(bool) { return ID_NONE; }
			virtual void Select();
			int GetSelected() { return selected; }
			
			virtual bool FocusAsObject() { return true; }
			void Clear() { height = 0; Container::Clear(); }
		protected:
			void Render(sf::RenderTarget &target) const;
			int pad;
			int height;
			int scroll;
			int selected;
			sf::Sprite select;
			sf::Clock click_timer;

			int GetHeight(int a, int b);
			int MaxTexts(int a, int d);
	};

	class TextSelection : public ScrollSelection
	{
		public:
			TextSelection(Rect r, sf::Image &select, std::vector<std::string> _text, sf::Color color = sf::Color(), int _pad = 5) 
				: ScrollSelection(r, select, color, _pad) { SetText(_text); }
			void SetText(std::vector<std::string> &text);
			void Select();
	};

	class Window : public Container
	{
		public:
			Window(Rect = Rect(), sf::Color color = sf::Color(), int bar = 10, sf::Color barcolor = sf::Color());
			void MouseDownAction(int x, int y, MouseButton m);
			void MouseUpAction(int x, int y, MouseButton m);
			void MouseMotionAction(int x, int y, int dx, int dy, const sf::Input&);
		protected:
			void Render(sf::RenderTarget &target) const;

			int bar;
			sf::Color barcolor;
			bool grabbed;
	};

	class TextField : public Object
	{
		public:
			TextField(Rect = Rect(), sf::Color color = sf::Color(0xFF, 0xFF, 0xFF));
			TextField(int w, sf::Color color = sf::Color(0xFF, 0xFF, 0xFF));
			virtual void TextAction(sf::Uint32 k);
			virtual void KeyDownAction(Key k);

			void SetText(std::string t) {
				text = t;
				cursor = scroll = 0;
				CalcDrawWidth();
			}
			std::string GetText() { 
				return text;
			}
		protected:
			void Render(sf::RenderTarget &target) const;
			std::string text;
			int cursor;
			int scroll;
			int drawwidth;

			void CalcScroll();
			void CalcDrawWidth();
	};

	class DrawArea : public Object
	{
		public:
			DrawArea(Rect r) : Object(r) { }
		private:
			void Render(sf::RenderTarget &target) const
			{
				ActionListener *l = GetListener();
				if(l)
					l->Draw(id, target);

			}

	};

	class Bar : public Object
	{
		public:
			Bar(Rect r = Rect(), sf::Color col = sf::Color(), double _p = 1.0f) : Object(r), p(_p)
			{
				SetColor(col);
			}
			void SetPercentage(double _p) { p = _p; }
		protected:
			void Render(sf::RenderTarget &target) const;
		private:
			double p;
	};

}

std::ostream & operator<<(std::ostream & str, Gui::Rect r);
#endif
