/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include <iostream>
#include <sstream>
#include <cmath>
#include "game.h"
#include "butterfly.h"

game::game(creature::creature_type ct, soundplayer &sp, musicplayer &mp, bool rep, record r) :
	g_world(&anim_loader, *this, ct), sound_player(sp), music_player(mp),
	selected_box(NULL), rec(r), frame_count(0), replay_mode(rep), paused(false),
	quit_pressed(false), cur_mode(select_mode), cur_spell(), ghost_mushroom(&anim_loader),
	ghost_cc(&anim_loader)
{
	if(!replay_mode)
		rec.creature_type = ct;

	music_player.set_playlist(creature::names[ct]);

	ghost_mushroom.get_anim()->set_mode("stand");
	ghost_cc.get_anim()->set_mode("stand");

	window = new Gui::Container(Gui::Rect(0, 0, SCREEN_W, SCREEN_H));
	window->SetActionListener(this);

	sidebar = new Gui::Container(Gui::Rect(0, 0, SIDEBAR_W, SCREEN_H), sf::Color(140, 140, 140));

	draw_area = new Gui::Container(Gui::Rect(SIDEBAR_W, 0, SCREEN_W - SIDEBAR_W, SCREEN_H));
	window->Add(draw_area);

	// (16,14,118,86)
	//Gui::VerticalList *l1 = new Gui::VerticalList(Gui::Rect(20, 18, 110, 78));
	menu_img.LoadFromFile("gfx/menu_button.png");
	menu_button = new Gui::ImgButton(menu_img, Gui::Rect(13, 12, 118, 67));
	menu_button->SetColor(sf::Color::Blue);
	menu_button->SetBorderColor(sf::Color::Blue);
	menu_button->SetFontColor(sf::Color::Blue);

	sidebar->Add(menu_button);

	vol_up.LoadFromFile("gfx/more.png");
	vol_down.LoadFromFile("gfx/less.png");

	// 20, 91, 66, 176                                  80
	sound_vol = new volume_widget(Gui::Rect(23, 91, 40, 100), "sound", music_player, sound_player,
		vol_up, vol_down);
	sidebar->Add(sound_vol);

	music_vol = new volume_widget(Gui::Rect(83, 91, 40, 100), "music", music_player, sound_player,
		vol_up, vol_down);
	sidebar->Add(music_vol);

//	l1->Add(menu_button);

//	sidebar->Add(l1);

	menu = new Gui::Container(Gui::Rect(166, 75, 312, 340), sf::Color(40, 40, 150, 210));
	menu->Hide();
	menu->DrawBG(true);

	Gui::VerticalList *l2 = new Gui::VerticalList(Gui::Rect(0, 0, 300, 300), 30);

	logo.LoadFromFile("gfx/logo.png");

	l2->AddCenter(new Gui::Image(logo));

	continue_button = new Gui::Button("Continue");
	set_button(continue_button);
	l2->AddCenter(continue_button);
	quit_button = new Gui::Button("Quit");
	set_button(quit_button);
	l2->AddCenter(quit_button);

	menu->AddCenter(l2);
	window->AddCenter(menu);

//	general_info = new Gui::Container(Gui::Rect(1,100, SIDEBAR_W-2, 220));
//	general_info->SetBorderColor(sf::Color::White);
//	general_info->SetBorderWidth(2);
	level_label = new Gui::TextLabel(Gui::Coor(15, 200), "Level: 1");
	level_label->SetFontColor(sf::Color(100, 150, 150));
	exp_label = new Gui::TextLabel(Gui::Rect(55, 243));
	exp_label->SetSize(12);
	exp_label->SetFontColor(sf::Color::Blue);
	sidebar->Add(level_label)->
		Add(exp_label);
	// 20 234
	// 132 251
	exp_bar = new Gui::Bar(Gui::Rect(20, 234, 112, 17), sf::Color::Cyan);
//	exp_bar = new Gui::Bar(Gui::Rect(20, 50, SIDEBAR_W - 10, 15), sf::Color::Cyan);
	exp_bar->SetBorderColor(sf::Color::Black);
	exp_bar->SetBorderWidth(2);

	sidebar->Add(exp_bar);

//	general_info->Add(level_label);

//	sidebar->Add(general_info);
	selected_box = new sbox(Gui::Rect(14, 282, SIDEBAR_W-16-12, 350),
			spell_hotkey.get_mode() == spell_hotkey_handler::hotkey);
	sidebar->Add(selected_box);
	selected_box->update(g_world);

	sidebar_img.LoadFromFile("gfx/sidebar.png");
	sidebar->Add(new Gui::Image(sidebar_img));

	window->Add(sidebar);
	sidebar->Raise();

	popup_win = new popup(Gui::Rect(0, 0, 300, 220), popup::ok);
	popup_win->Hide();

	window->AddCenter(popup_win);

	ghost_mushroom.upgrade(mushroom::m_stem);

	num_key_selections.resize(9, NULL);

	butterfly_emitter *e = new butterfly_emitter(&anim_loader);
	butterfly_group *g = new butterfly_group(e);
	p_engine.add(g);
}

game::~game()
{
	delete window;
}

void game::set_button(Gui::Button *b)
{
	b->Expand(30, 10);
	b->DrawBG(true);
	b->SetHilightUnderMouse(true);
	b->SetColor(sf::Color(0, 0x25, 65, 200));
	b->SetBorderColor(sf::Color(20, 20, 80));
	b->SetFontColor(sf::Color(0, 0x80, 0x50));
}

void game::draw(sf::RenderTarget &target)
{
	sf::View v = target.GetDefaultView();
	v.Move(-SIDEBAR_W, 0);
	target.SetView(v);

	// draw things
	g_world.draw(target, p_engine);
	p_engine.draw(target);

	if(!paused && cur_mode == cast_spell_mode) {
		if(cur_spell.get_spell_type() == spell::cc_grow_mushroom
						|| cur_spell.get_spell_type() == spell::mushroom_multiply
						|| cur_spell.get_spell_type() == spell::unicorn_spawn_mushroom) {
			draw_ghost_mushroom(target);
		}
		if(cur_spell.get_spell_type() == spell::minotaur_build_cc) {
			draw_ghost_cc(target);
		}
		if(cur_spell.get_spell_type() == spell::griffin_explosive_trap ||
						cur_spell.get_spell_type() == spell::griffin_paralyse_trap) {
			draw_ghost_trap(target);
		}

		int range = cur_spell.get_range(g_world.get_selected());
		sf::Vector2f c = g_world.get_selected()->get_center();

		sf::Shape cr = sf::Shape::Circle(c.x, c.y, range, sf::Color(0xFF, 0x00, 0x00, 0x10), 1.0f, sf::Color::Red);
		cr.EnableFill(true);

		target.Draw(cr);

		if(!paused && cur_mode == cast_spell_mode
			&& (cur_spell.get_spell_type() == spell::mushroom_heal
				|| cur_spell.get_spell_type() == spell::mushroom_give_energy)) {
			mushroom *c_mushroom = dynamic_cast<mushroom*>(g_world.get_selected());
			std::list<mushroom*>::iterator it;
			if(c_mushroom) {
				for(it = c_mushroom->get_connections().begin(); it != c_mushroom->get_connections().end(); it++) {
					(*it)->draw_border(target, sf::Color::Cyan);
				}
			}
		}
	}
	target.SetView(target.GetDefaultView());
}

void game::draw_ghost_mushroom(sf::RenderTarget & target)
{
	sf::Vector2f pos = draw_area->GetMousePos().sfVector();
	pos = pos - sf::Vector2f(SIDEBAR_W, 0);
	bool in_range = true;
	if(dist(g_world.get_selected()->get_center(), pos) > cur_spell.get_range(g_world.get_selected())) {
		in_range = false;
	}

	bool too_up = pos.y < cur_spell.get_y_limit();

	pos = pos - sf::Vector2f(ghost_mushroom.get_w() / 2, ghost_mushroom.get_h() / 2);
	ghost_mushroom.set_pos(pos);

	object *caster = g_world.get_selected();

	bool collide = g_world.collision(ghost_mushroom);

	bool los = g_world.get_map().los(caster->get_center(), ghost_mushroom.get_center());

	bool fail;
//	std::cout << collide << " " << !in_range << " " <<!los<<std::endl;
	fail = collide || !in_range || too_up || !los;

	sf::Color color;
	if(fail)
		color = sf::Color(0xFF, 0x80, 0x80, 0x70);
	else
		color = sf::Color(0xFF, 0xFF, 0xFF, 0x50);
	ghost_mushroom.draw_color(target, color);
	ghost_mushroom.draw_border(target, color);
}

void game::draw_ghost_cc(sf::RenderTarget & target)
{
	sf::Vector2f pos = draw_area->GetMousePos().sfVector();
	pos = pos - sf::Vector2f(SIDEBAR_W, 0);
	bool in_range = true;
	if(dist(g_world.get_selected()->get_center(), pos) > cur_spell.get_range(g_world.get_selected())) {
		in_range = false;
	}

	bool too_up = pos.y < cur_spell.get_y_limit();

	pos = pos - sf::Vector2f(ghost_cc.get_w() / 2, ghost_cc.get_h() / 2);
	ghost_cc.set_pos(pos);

	object *caster = g_world.get_selected();

	bool collide = g_world.collision(ghost_cc);

	bool los = g_world.get_map().los(caster->get_center(), ghost_cc.get_center());

	bool fail;
//	std::cout << collide << " " << !in_range << " " <<!los<<std::endl;
	fail = collide || !in_range || too_up || !los;

	sf::Color color;
	if(fail)
		color = sf::Color(0xFF, 0x80, 0x80, 0x70);
	else
		color = sf::Color(0xFF, 0xFF, 0xFF, 0x50);
	ghost_cc.draw_color(target, color);
	ghost_cc.draw_border(target, color);
}

void game::draw_ghost_trap(sf::RenderTarget & target)
{
	trap *ghost_trap = NULL;
	if(cur_spell.get_spell_type() == spell::griffin_paralyse_trap)
		ghost_trap = new trap(trap::paralyse, &anim_loader);
	else
		ghost_trap = new trap(trap::explosive, &anim_loader);

	sf::Vector2f pos = draw_area->GetMousePos().sfVector();
	pos = pos - sf::Vector2f(SIDEBAR_W, 0);
	bool in_range = true;
	if(dist(g_world.get_selected()->get_center(), pos) > cur_spell.get_range(g_world.get_selected())) {
		in_range = false;
	}

	bool too_up = pos.y < cur_spell.get_y_limit();

	pos = pos - sf::Vector2f(ghost_trap->get_w() / 2, ghost_trap->get_h() / 2);
	ghost_trap->set_pos(pos);

	object *caster = g_world.get_selected();

	bool collide = g_world.collision(*ghost_trap);

	bool los = g_world.get_map().los(caster->get_center(), ghost_trap->get_center());

	bool fail;
//	std::cout << collide << " " << !in_range << " " <<!los<<std::endl;
	fail = collide || !in_range || too_up || !los;

	sf::Color color;
	if(fail)
		color = sf::Color(0xFF, 0x80, 0x80, 0x70);
	else
		color = sf::Color(0xFF, 0xFF, 0xFF, 0x50);
	ghost_trap->draw_color(target, color);
	ghost_trap->draw_border(target, color);
	delete ghost_trap;
}

void game::ButtonClick(Gui::ID srcid)
{
	if(srcid == quit_button->GetID()) {
		quit_pressed = true;
	} else if(srcid == continue_button->GetID()) {
		menu->Hide();
		window->Lock(Gui::ID_NONE);
		paused = false;
		game_clock.unpause();
		sound_player.unpause();
	} else if(srcid == menu_button->GetID()) {
		menu->Show();
		window->Lock(menu->GetID());
		paused = true;
		game_clock.pause();
		sound_player.pause();
		g_world.pause();
	} else if(srcid == popup_win->GetID()) {
		if(g_world.is_win()) {
			if(popup_win->get_type() == popup::twobuttons &&
					popup_win->ret_val() == 1) {
				config &conf = config::get();

				window->DeleteID(popup_win->GetID());
				popup_win = new popup(Gui::Rect(0, 0, 300, 220), popup::text_box);
				std::vector<std::string> str;
				str.push_back("Enter nick");
				popup_win->set_text(str);
				popup_win->set_field_text(conf.get_string("nick"));
				window->AddCenter(popup_win);
				popup_win->Focus();
				window->Lock(popup_win->GetID());
			} else {
				if(popup_win->ret_val() == 0) {
					config &conf = config::get();
					conf.set_string("nick", popup_win->get_text());
					upload_record();
				}
				quit_pressed = true;
			}
		} else if(g_world.is_lose()) {
			quit_pressed = true;
		}
	} else {
		if(!replay_mode && selected_box->get_spell(srcid, cur_spell))
		{
			if(cur_spell.need_target()) {
				cur_mode = cast_spell_mode;
			} else {
				action a;
				a.t = action::cast_spell;
				a.id = cur_spell.get_id();
				rec.append_action(a, frame_count);

				g_world.cast(g_world.get_selected(), cur_spell, sf::Vector2f(0,0));
				cur_mode = select_mode;
			}
		} else {
//			std::cout << "something funny was pressed" << std::endl;
		}
	}
}

void game::run_replay()
{
	record::frame & fr = rec.get_next_frame(frame_count);
	for(record::action_it it = fr.actions.begin(); it != fr.actions.end(); it++) {
		action &ac = *it;

		sf::Vector2f mouse(ac.x, ac.y);

		switch(ac.t) {
			case action::none:
				break;
			case action::cast_spell:
				{
					std::vector<spell> &spells = g_world.get_selected()->get_spells();
					if(spells.size() <= ac.id) {
						std::cerr << "Invalid spell id: " << ac.id << std::endl;
						break;
					}


					cur_spell = g_world.get_selected()->get_spells()[ac.id];
					g_world.cast(g_world.get_selected(), cur_spell, mouse);
				}
				break;
			case action::select:
				g_world.select(mouse);
				break;
			case action::move_player:
				g_world.move_player(mouse);
				break;
			case action::cc_tree_selection:
				g_world.cc_tree_selection(mouse);
				break;
			case action::select_id:
				g_world.select(ac.id);
				break;
			default:
				break;
		}
	}
}

bool game::update()
{
	if(paused)
		return false;

	exp_bar->SetPercentage(g_world.get_player().get_level_progress());
	update_level_labels();
	selected_box->update(g_world);

	int loops = 0;
	while(game_time.get_time() + DT < game_clock.get_time()) {
		if(replay_mode) {
			run_replay();
		}

		p_engine.update();
		g_world.update(game_time);

		if(g_world.selected_changed())
			cur_mode = select_mode;

		game_time.step();
		frame_count++;
		loops++;
		if(loops == 200)
			break;
	}

	return true;
}

void game::update_level_labels()
{
	std::stringstream level_text(std::stringstream::out);
	std::stringstream exp_text(std::stringstream::out);

	level_text << "Level: " << g_world.get_player().get_level();
	exp_text << g_world.get_player().get_exp() << "/";
	if(!g_world.get_player().is_max_level())
		exp_text << g_world.get_player().get_req_exp();
	else
		exp_text << "inf";

	level_label->SetLabel(level_text.str());
	exp_label->SetLabel(exp_text.str());
}

void game::MouseClick(Gui::ID srcid, int x, int y, Gui::MouseButton m)
{
	sf::Vector2f mouse(x,y);
	action a;
	a.x = mouse.x;
	a.y = mouse.y;

	if(!replay_mode && srcid == draw_area->GetID()) {
		if(m == Gui::BUTTON1) {
			if(cur_mode == cast_spell_mode) {
				a.t = action::cast_spell;
				a.id = cur_spell.get_id();
				rec.append_action(a, frame_count);

				g_world.cast(g_world.get_selected(), cur_spell, mouse);
				cur_mode = select_mode;
			} else if (cur_mode == select_mode) {
				a.t = action::select;
				rec.append_action(a, frame_count);

				g_world.select(mouse);
			}
		} else if(m == Gui::BUTTON2) {
			selectable *selected = g_world.get_selected();
			creature *creatr = dynamic_cast<creature*>(selected);
			command_center *cc = dynamic_cast<command_center*>(selected);

			if(creatr) {
				a.t = action::move_player;
				rec.append_action(a, frame_count);

				g_world.move_player(mouse);

			} else if(cc) {
				a.t = action::cc_tree_selection;
				rec.append_action(a, frame_count);

				g_world.cc_tree_selection(mouse);
			}
		}
	}
}

void game::KeyDownAction(sf::Event::KeyEvent k)
{
	using namespace sf::Key;
	if(k.Control) {
		for(int i = 0; i < 9; i++) {
			if(Num1 + i == k.Code) {
				num_key_selections[i] = g_world.get_selected();
			}
		}
	} else {
		switch(k.Code) {
			case sf::Key::Escape:
			{
				cancel_casting();
			}
			break;
			default:
			{
				for(int i = 0; i < 9; i++) {
					if(Num1 + i == k.Code) {
						g_world.select(num_key_selections[i]);
						action a;
						a.t = action::select_id;
						a.id = g_world.get_selected()->get_id();
						rec.append_action(a, frame_count);
					}
				}

				if(!replay_mode && spell_hotkey.pressed(k.Code, g_world, cur_spell))
				{
					if(cur_spell.need_target()) {
						cur_mode = cast_spell_mode;
					} else {
						action a;
						a.t = action::cast_spell;
						a.id = cur_spell.get_id();
						rec.append_action(a, frame_count);

						g_world.cast(g_world.get_selected(), cur_spell, sf::Vector2f(0,0));
						cur_mode = select_mode;
					}
				}
			}
			break;
		}
	}
}

void game::cancel_casting()
{
	cur_mode = select_mode;
	cur_spell = spell();
}


// play some beautiful sounds or something
void game::hit(object *src, object *target, attack &atk)
{
	ant *a = dynamic_cast<ant*>(src);
	mushroom *m = dynamic_cast<mushroom*>(src);
	if(a) {
		sound_player.play_sound("ant_attack", false, 0.0f);
	} else if(m) {
		sound_player.play_sound("mushroom_attack", false, atk.get_delay() - 0.2f);
	}
}

void game::grow(mushroom *, mushroom::mushroom_type)
{
}

void game::near_grow(mushroom *, mushroom::mushroom_type)
{
	sound_player.play_sound("mushroom_grow");
}

void game::die(object *obj)
{
}

void game::spawn(object *obj)
{
}

void game::ding(object *obj)
{
	particle_emitter *e = new spell_emitter(obj->get_center());
	e->set_maximum_age(0.5f);
	particle_group *g = new particle_group(200, e, particle_group::point);

	p_engine.add(g);
	sound_player.play_sound("ding");
}

void game::wind_blow()
{
	sound_player.play_sound("wind");
}

void game::wind_stop()
{
	sound_player.stop_sound("wind");
}

void game::win()
{
	if(paused)
		return;
	if(!replay_mode) {
		window->DeleteID(popup_win->GetID());
		popup_win = new popup(Gui::Rect(0, 0, 300, 220), popup::twobuttons);
		window->AddCenter(popup_win);
	}

	std::vector<std::string> text;
	text.push_back("Victory!");
	std::stringstream time;
	double secs = game_time.get_time();
	double mins = (secs - fmod(secs, 60.0))/60;
	secs -= 60 * mins;
	time << int(mins) << ":";
	time.fill('0');
	time.width(5);
	time.precision(2);
	time.setf(std::ios::fixed, std::ios::floatfield);
	time << secs;
	text.push_back("Time: " + time.str());
	popup_win->set_text(text);
	popup_win->Show();

	window->Lock(popup_win->GetID());
	paused = true;

	if(!replay_mode)
		rec.write(rep_file());
	music_player.set_playlist("victory");
}

void game::lose()
{
	std::vector<std::string> text;
	text.push_back("Defeat!");
	popup_win->set_text(text);
	popup_win->Show();

	window->Lock(popup_win->GetID());
	paused = true;

	if(!replay_mode)
		rec.write(rep_file());
	music_player.set_playlist("defeat");
}


void game::trap_trigger(trap *t, ant *a)
{
	if(t->get_type() == trap::paralyse) {
		sound_player.play_sound("trap_trigger", false, t->get_delay_time() - 0.1f);
	} else {
		explosion_emitter *e = new explosion_emitter(t->get_center());
		e->set_maximum_age(0.4);
		e->delay(t->get_delay_time() - 0.3);
		particle_group *g = new particle_group(200, e, particle_group::point);
		p_engine.add(g);

		sound_player.play_sound("explosion", false, t->get_delay_time() - 0.3f);
	}
}

void game::cast_spell(object *caster, const spell &sp)
{
	if(sp.get_spell_type() == spell::unicorn_invisibility)
		sound_player.play_sound("magic_wand");
	else if(sp.get_spell_type() == spell::minotaur_attack_wall)
		sound_player.play_sound("hit_wall");
	else if(sp.get_spell_type() == spell::mushroom_attack_wall)
		sound_player.play_sound("mushroom_attack_wall");
	else if(sp.get_spell_type() == spell::minotaur_ants_bane)
		sound_player.play_sound("axe_swing");
}

std::string game::rep_file()
{
	char buf[100];
	std::string file = "replays/replay-";
	time_t t;
	time(&t);
	strftime(buf, 100, "%d-%m-%Y-%H%M%S", localtime(&t));

	return file + buf;
}

void game::upload_record()
{
	config &cfg = config::get();

	std::string r = rec.to_str();
	std::string nick = cfg.get_string("nick");

	std::string boundary = "57236a87ddcc525783bde08ac492c406";

	sf::SocketTCP sock;
	std::string host = "ds.kapsi.fi";

	std::string header;
	header += "POST /uploadreplay_sfml HTTP/1.0\r\n";
	header += "Content-Type: multipart/form-data; boundary=" + boundary + "\r\n";
	header += "Host: " + host + "\r\n";

//	sf::Http http("localhost");
//	sf::Http::Request req(sf::Http::Request::Post, "/scoreboard2");
//	req.SetField("Content-Type", "multipart/form-data, boundary=" + boundary);

	std::string body;
	encode_request_data(body, boundary, "nick", nick);
	encode_request_data(body, boundary, "replay", r, "replay");
	body += "--" + boundary + "--\r\n\r\n\r\n";

	std::stringstream len;
	len << (body.size() - 2); // two last \r\ns don't belong to body
	header += "Content-Length: " + len.str() + "\r\n\r\n";

//	req.SetBody(body);

	if(!sock.Connect(80, sf::IPAddress(host), 10.0)) {
		sock.Send(header.data(), header.size());
		sock.Send(body.data(), body.size());

		char data[10001];
		std::size_t total = 0;
		std::size_t received = 0;
		while(sock.Receive(data + total, 10000 - total, received) != sf::Socket::Disconnected)
			total += received;
		total += received;
		data[total] = 0;
		sock.Close();
	}
	//sf::Http::Response res = http.SendRequest(req, 10.0f);
//	std::cout << res.GetStatus() << std::endl;
}

void game::encode_request_data(std::string &body, const std::string &boundary,
		const std::string &name, const std::string &data,
		const std::string &filename)
{
	body += "--" + boundary + "\r\n";

	body += "Content-Disposition: form-data; name=\"" + name + "\"";
	if(filename.size()) {
		body += "; filename=\"" + filename + "\"\r\n";
		body += "Content-Type: application/octet-stream";
	}
	body += "\r\n\r\n";

	body += data;
	body += "\r\n";
}
