/***************************************************************************
*   Copyright (C) 2010 by
*    Kai Lindholm <megantti@gmail.com>
*    Santtu Keskinen <laquendi@gmail.com>
*
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include "creature.h"
#include "world.h"
#include <cstring>

const int creature::max_level = 4;
const int creature::exp_table[3][3] = {
	{100, 500, 1000},
	{100, 500, 1000},
	{100, 700, 2000}
};

const int creature::base_mushroom_attr_table[3][4][3] =
{
	//unicorn
	{
		//level 1
		{0, 25, 5},
		//level 2
		{5, 35, 15},
		//level 3
		{10, 45, 25},
		//level 4
		{15, 55, 30},
	},
	//minotaur
	{
		//level 1
		{15, 0, 10},
		//level 2
		{25, 10, 20},
		//level 3
		{35, 15, 25},
		//level 4
		{45, 25, 30},
	},
	//griffin
	{
		//level 1
		{15, 5, 15},
		//level 2
		{20, 10, 25},
		//level 3
		{25, 15, 35},
		//level 4
		{40, 30, 45}
	}
};

std::map<unsigned int, std::string> moving::anim_mode_map;

static const char *creature_anims[] = {
	"unicorn",
	"minotaur",
	"griffin"
};

const char *creature::names[] = {
	"unicorn",
	"minotaur",
	"griffin"
};

creature::creature(animation_loader *l, creature_type c_type) : 
		selectable(l?&l->get_animation(creature_anims[(int)c_type]):0), moving(12), 
		moving_to_target(false), dummy_spell(), spell_to_cast(dummy_spell),
		moved(false), t(c_type), exp(0), level(1)
{
	mushroom_attr_table = new int[4*3];
	memccpy(mushroom_attr_table, base_mushroom_attr_table[t], -1, 4*3*sizeof(int));
	set_w(45);
	set_h(45);
	set_mode(anim_mode_map[anim_mode]);
	

	switch (t) {
		case unicorn:
		{
			spells.push_back(spell(spell::unicorn_invisibility));	
			spells.push_back(spell(spell::unicorn_dimension_door));	
			spells.push_back(spell(spell::unicorn_mana_orb));	
			spells.push_back(spell(spell::unicorn_diagonal_spawn));
			spells.push_back(spell(spell::unicorn_spawn_mushroom));
			spells.push_back(spell(spell::unicorn_resurrect));
			set_speed(30);
		}
		break;
		case minotaur:
		{
			spells.push_back(spell(spell::minotaur_attack_stance));
			spells.push_back(spell(spell::minotaur_throw_mushroom));
			spells.push_back(spell(spell::minotaur_build_cc));
			spells.push_back(spell(spell::minotaur_mass_connect));
			spells.push_back(spell(spell::minotaur_attack_wall));
			spells.push_back(spell(spell::minotaur_ants_bane));
			set_speed(15);
		}
		break;
		case griffin:
		{
			spells.push_back(spell(spell::griffin_paralyse_trap));
			spells.push_back(spell(spell::griffin_kill_ant));
			spells.push_back(spell(spell::griffin_eat_mushroom));	
			spells.push_back(spell(spell::griffin_explosive_trap));
			spells.push_back(spell(spell::griffin_boost_stats));
			spells.push_back(spell(spell::griffin_immortality));
			set_speed(130);
		}
		break;
	}

	calc_spell_ids();
}

creature::~creature()
{
	delete[] mushroom_attr_table;
}

void creature::update(world &w)
{
	selectable::update(w);
	
	move(w);
}

void creature::set_move_pos(sf::Vector2f mv)
{
	set_mv_pos(mv + sf::Vector2f(-get_w()/2, -get_h()/2));
	moving_to_target = false;
}

void creature::move_towards_cast_target(spell &casted_spell, sf::Vector2f mv)
{
	spell_to_cast = casted_spell;
	moving_to_target = true;
	set_mv_pos(mv + sf::Vector2f(-get_w()/2, -get_h()/2));
	spell_target_pos = mv;
}

void creature::gain_exp(world &w, int d_exp)
{
	exp+=d_exp;
	
	w.cc_add_energy(d_exp);

	while(level>1 && exp<get_prev_level_exp())
		down_ding();
	while(level != max_level && exp>get_req_exp()) {
		ding();
		w.handler.ding(this);
	}
}

int creature::get_req_exp()
{
	if (level == max_level)
		return -1;
	return exp_table[t][level-1];

}

int creature::get_prev_level_exp()
{
	if (level == 1)
		return -1;
	return exp_table[t][level-2];
}

float creature::get_level_progress()
{
	if (level == max_level)
		return 1.0;
	return (double)(get_exp()-get_prev_level_exp())/(get_req_exp()-get_prev_level_exp());
}

void creature::ding()
{
	if(level<max_level) {
		level++;
	}
	if(level == max_level) {
		if(!spells.empty())
			spells.back().update_cooldown(spells.back().get_s_cd());
	}
}

void creature::down_ding()
{
	if(level>1) {
		level--;
	}
}

void creature::boost_stats(int boost)
{
	for(int a = 0; a < 4; a++) {
		for(int b = 0; b < 3; b++) {
			mushroom_attr_table[a*3 + b] += boost;
			if(mushroom_attr_table[a*3 + b] > 100)
				mushroom_attr_table[a*3 + b] = 100;
		}
	}
}

bool creature::collision(const object &obj) const
{
	if(t!=griffin)
	{
		ant *p = dynamic_cast<ant*>(&const_cast<object&>(obj));
		trap *r = dynamic_cast<trap*>(&const_cast<object&>(obj));
		if(!p && !r)
		{
			return object::collision(obj);
		}
	}
	return false;
}

