/***************************************************************************
*   Copyright (C) 2010 by
*    Santtu Keskinen <laquendi@gmail.com>
*   
*   This program is free software; you can redistribute it and/or modify
*   it under the terms of the GNU General Public License as published by
*   the Free Software Foundation; either version 2 of the License, or
*   (at your option) any later version.
* 
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the
*   Free Software Foundation, Inc.,
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
****************************************************************************/

#include "ant.h"
#include "world.h"
#include <cmath>
#include <sstream>
#ifndef NO_GRAPHICS
#include "gui.h"
#endif

ant::ant(animation_loader *l, int _level) : selectable(l?&l->get_animation("ant"):0), 
		moving(3), atk_skill(attack::ant_attack), paralysed(false),
		evading_obstacle(false), evades(0), level(_level), target_range(150.0f)
{
	set_w(25+level);
	set_h(25+level);
	set_mode(anim_mode_map[anim_mode]);
	set_speed(70);

	atk_skill.set_atk_boost((double)level/2);

	hp = max_hp = 70 * pow(1.2, level);
}

void ant::pause()
{
	selectable::pause();
	paralyse_timer.pause();
}

void ant::has_target(object* tmp_target, world &w)
{
	if(dist(get_center(), tmp_target->get_center()) < atk_skill.get_range()) {
		atk_skill.update(has_moved());
		attack_enemy(w);
	} else {
		if(!evading_obstacle)
			set_mv_pos(tmp_target->get_pos());
		ant_move(w);
	}
}

sf::Vector2f ant::evade_obstacle_pos(world &w, sf::Vector2f move_dir)
{
	evades++;
	sf::Vector2f normal_vector[2] = {sf::Vector2f(-move_dir.y, move_dir.x)};
	normal_vector[1] = -normal_vector[0];
	int better = 0;
	if( w.ant_dir_goodness(normal_vector[1], this) > w.ant_dir_goodness(normal_vector[0], this) )
		better = 1;
	return sf::Vector2f(get_pos().x + 50*normal_vector[better].x - move_dir.x*10, 
					get_pos().y + 50*normal_vector[better].y - move_dir.y*10 );
}

void ant::ant_move(world &w)
{
	if(!evading_obstacle) {
		std::pair<bool, sf::Vector2f> success = move(w);
		if(!success.first) {
			evading_obstacle = true;
			set_mv_pos(evade_obstacle_pos(w, success.second));
		}
	} else {
		std::pair<bool, sf::Vector2f> success = move(w);

		if(!success.first) {
			set_mv_pos(evade_obstacle_pos(w, success.second));
		}

		if(dist(get_mv_pos(), get_pos()) < speed) {
			evading_obstacle = false;
		}
	}
}

void ant::update(world &w)
{
	selectable::update(w);

	if(!paralysed) {
		set_moved(false);

		mushroom *target_mushroom = w.nearest_mushroom(get_center(), 200.0f, true, false);	
		command_center *target_cc = w.nearest_cc(get_center(), 200.0f, false);
		if(evades > 3 && (target_mushroom || target_cc)) {
			float nearest_dist = 500000;
			selectable *nearest_target = NULL;
			if(target_mushroom) {
				nearest_dist = dist(target_mushroom->get_center(), get_center());
				nearest_target = target_mushroom;
			}
			if(target_cc) {
				if(dist(target_cc->get_center(), get_center()) < nearest_dist) 
					nearest_target = target_cc;
			}
			has_target(nearest_target, w);
		} else if(preferred_target && !preferred_target->is_invisible()) {
			has_target(preferred_target, w);
		} else if(level < 5) {
			if(target_mushroom) {
				has_target(target_mushroom, w);
			}
			else if(target_cc) {
				has_target(target_cc, w);
			} else {
				if(!evading_obstacle) {
					sf::Vector2f target_location = w.get_ant_target_location();
					if(target_location.x != -1.0f && target_location.y != -1.0f)
						set_mv_pos(target_location);
					else 
						set_mv_pos(sf::Vector2f(get_pos().x - 100.0f, get_pos().y));
				}
				ant_move(w);
			}
		} else {
			if(target_cc) {
				has_target(target_cc, w);
			} else if(target_mushroom) {
				has_target(target_mushroom, w);
			} else {
				if(!evading_obstacle){
					sf::Vector2f target_location = w.get_ant_target_location();
					if(target_location.x != -1.0f && target_location.y != -1.0f)
						set_mv_pos(target_location);
					else 
						set_mv_pos(sf::Vector2f(get_pos().x - 100.0f, get_pos().y));
				}
				ant_move(w);
			}
		}
	}
		


	if(paralysed) {
		paralyse_timer.unpause();
		paralyse_timer.step();

		if(paralyse_timer.get_time() > paralyse_time) {
			paralysed = false;
			set_mode(anim_mode_map[anim_mode%8]);
		}
	}	
}

#ifndef NO_GRAPHICS
void ant::draw(sf::RenderTarget &target)
{
	object::draw(target);
	double up_bar = get_hp() / get_max_hp();

	Gui::drawFilledRectangle(target, Gui::Rect(get_x(), get_y() - 5, get_w(), 3), up_bar, sf::Color::Red);
	

	sf::Vector2f pos = get_center();
	sf::Vector2f dir = sf::Vector2f(-1, 0);
	sf::Vector2f normal = sf::Vector2f(-dir.y, dir.x);
}
#endif

void ant::attack_enemy(world &w)
{
	if(!atk_skill.can_attack())
		return;

	mushroom *target_mushroom = w.nearest_mushroom(get_center(), atk_skill.get_range(), true, false);
	command_center *target_cc = w.nearest_cc(get_center(), atk_skill.get_range(), false);	
	if(preferred_target && !preferred_target->is_invisible()) {
		if(dist(get_center(), preferred_target->get_center()) < atk_skill.get_range()) {
			attack atk = atk_skill.do_attack();
			preferred_target->damage(w, atk);
			w.handler.hit(this, preferred_target, atk);
		}
	} else if(target_mushroom) {
		attack atk = atk_skill.do_attack();
		target_mushroom->damage(w, atk);
		w.handler.hit(this, target_mushroom, atk);
	} else if (target_cc) {
		attack atk = atk_skill.do_attack();
		target_cc->damage(w, atk);
		w.handler.hit(this, target_cc, atk);
	}
}

bool ant::collision(const object &obj) const
{
	creature *p = dynamic_cast<creature*>(&const_cast<object&>(obj));
	trap *t = dynamic_cast<trap*>(&const_cast<object&>(obj));

	if(!p && !t)
	{
		return object::collision(obj);
	}
	return false;
}

void ant::paralyse(float time)
{
	paralyse_time = time;
	paralyse_timer.reset();
	paralysed = true;
	set_mode(anim_mode_map[anim_mode%8 + 16]);
}

std::string ant::level_string()
{
	std::stringstream ss(std::stringstream::out);
	ss <<"Level: "<< level <<std::endl;
	return ss.str();
}

void ant::set_target(object *t)
{
	object::set_target(t);
	evades = 0;
}

