#include <SDL.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <math.h>
#include "sdl.h"
#include "timer.h"
#include "wiimote.h"

static GLuint particle;

void init_fluid_plane()
{
  load_texture("particle.bmp", &particle);
  
  glDisable( GL_TEXTURE_2D );
  //glShadeModel( GL_SMOOTH );

  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
  glDisable(GL_DEPTH_TEST);
  

  //glEnable(GL_BLEND);
  //glBlendFunc(GL_SRC_ALPHA, GL_ONE);
  

/*
  glClearDepth(1.0f);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
  glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
*/
}

static void draw_particle(float x, float y, float z)
{
  float size = 0.3;
  glBegin(GL_TRIANGLE_STRIP);
  glTexCoord2d(1, 1);
  glVertex3f(x + size, y + size, z);
  glTexCoord2d(0, 1);
  glVertex3f(x - size, y + size, z);
  glTexCoord2d(1, 0);
  glVertex3f(x + size, y - size, z);
  glTexCoord2d(0, 0);
  glVertex3f(x - size, y - size, z);
  glEnd();
}

static float rotation = 0.0;

#define IX(i,j) ((i)+(N+2)*(j))

static void set_bnd ( int N, int b, float * x )
{
  int i;
  for ( i=1 ; i<=N ; i++ )
  {
    x[IX(0,i)]   = b==1 ? -x[IX(1,i)] : x[IX(1,i)];
    x[IX(N+1,i)] = b==1 ? -x[IX(N,i)] : x[IX(N,i)];
    x[IX(i,0)]   = b==2 ? -x[IX(i,1)] : x[IX(i,1)];
    x[IX(i,N+1)] = b==2 ? -x[IX(i,N)] : x[IX(i,N)];
  }
  x[IX(0 ,0 )] =   0.5*(x[IX(1,0 )]+x[IX(0 ,1)]);
  x[IX(0 ,N+1)] =  0.5*(x[IX(1,N+1)]+x[IX(0 ,N )]);
  x[IX(N+1,0 )] =  0.5*(x[IX(N,0 )]+x[IX(N+1,1)]);
  x[IX(N+1,N+1)] = 0.5*(x[IX(N,N+1)]+x[IX(N+1,N )]);
}

static void project ( int N, float * u, float * v, float * p, float * div )
{
    int i, j, k;
    float h;
    h = 1.0/N;
    for ( i=1 ; i<=N ; i++ ) {
        for ( j=1 ; j<=N ; j++ ) {
            div[IX(i,j)] = -0.5*h*(u[IX(i+1,j)]-u[IX(i-1,j)]+
                                   v[IX(i,j+1)]-v[IX(i,j-1)]);
            p[IX(i,j)] = 0;
        }
    }
    set_bnd ( N, 0, div ); set_bnd ( N, 0, p );
    for ( k=0 ; k<20 ; k++ ) {
        for ( i=1 ; i<=N ; i++ ) {
            for ( j=1 ; j<=N ; j++ ) {
                p[IX(i,j)] = (div[IX( i,j)]+p[IX(i-1,j)]+p[IX(i+1,j)]+
                                            p[IX(i,j-1)]+p[IX(i,j+1)])/4;
            }
        }
        set_bnd ( N, 0, p );
    }
    for ( i=1 ; i<=N ; i++ ) {
        for ( j=1 ; j<=N ; j++ ) {
            u[IX(i,j)] -= 0.5*(p[IX(i+1,j)]-p[IX(i-1,j)])/h;
            v[IX(i,j)] -= 0.5*(p[IX(i,j+1)]-p[IX(i,j-1)])/h;
        }
    }
    set_bnd ( N, 1, u ); set_bnd ( N, 2, v );
}

static void add_source(int N, float * x, float * s, float dt)
{
  int i, size=(N+2)*(N+2);
  for ( i=0 ; i<size ; i++ ) x[i] += dt*s[i];
}

static void diffuse ( int N, int b, float * x, float * x0, float diff, float dt )
{
    int i, j, k;
    float a=dt*diff*N*N;
    for ( k=0 ; k<20 ; k++ ) {
        for ( i=1 ; i<=N ; i++ ) {
            for ( j=1 ; j<=N ; j++ ) {
                x[IX(i,j)] = (x0[IX(i,j)] + a*(x[IX(i-1,j)]+x[IX(i+1,j)]+
                                               x[IX(i,j-1)]+x[IX(i,j+1)]))/(1+4*a);
            }
        }
        set_bnd ( N, b, x );
    }
}

static void advect ( int N, int b, float * d, float * d0, float * u, float * v, float dt )
{
    int i, j, i0, j0, i1, j1;
    float x, y, s0, t0, s1, t1, dt0;
    dt0 = dt*N;
    for ( i=1 ; i<=N ; i++ ) {
        for ( j=1 ; j<=N ; j++ ) {
         x = i-dt0*u[IX(i,j)]; y = j -dt0*v[IX(i,j)];
         if (x<0.5) x=0.5; if (x>N+0.5) x=N+ 0.5; i0=(int)x; i1=i0+ 1;
         if (y<0.5) y=0.5; if (y>N+0.5) y=N+ 0.5; j0=(int)y; j1=j0+1;
         s1 = x-i0; s0 = 1-s1; t1 = y-j0; t0 = 1-t1;
         d[IX(i,j)] = s0*(t0*d0[IX(i0,j0)]+t1*d0[IX(i0,j1)])+
                       s1*(t0*d0[IX(i1,j0)]+t1*d0[IX(i1,j1)]);
      }
  }
  set_bnd ( N, b, d );
}

#define SWAP(x0,x) {float *tmp=x0;x0=x;x=tmp;}

static void dens_step ( int N, float * x, float * x0, float * u, float * v, float diff,
    float dt )
{
    add_source ( N, x, x0, dt );
    SWAP ( x0, x ); diffuse ( N, 0, x, x0, diff, dt );
    SWAP ( x0, x ); advect ( N, 0, x, x0, u, v, dt );
}

static void vel_step ( int N, float * u, float * v, float * u0, float * v0,
           float visc, float dt )
{
    add_source ( N, u, u0, dt ); add_source ( N, v,v0, dt );
    SWAP ( u0, u ); diffuse ( N, 1, u, u0, visc, dt);
    SWAP ( v0, v ); diffuse ( N, 2, v, v0, visc, dt);
    project ( N, u, v, u0, v0 );
    SWAP ( u0, u ); SWAP ( v0, v );
    advect ( N, 1, u, u0, u0, v0, dt ); advect ( N,2, v, v0, u0, v0, dt );
    project ( N, u, v, u0, v0 );
}

#define GRID_N 64
#define SIZE ((GRID_N+2)*(GRID_N+2))
static float u[SIZE], v[SIZE], u_prev[SIZE], v_prev[SIZE];
static float dens[SIZE], dens_prev[SIZE];

static float ui_u = 5.0;
static float ui_v = 0.0;
static float ui_d = 5.0;

static void get_from_UI (int N, float * d, float * u, float * v )
{
  int i, j, size = (N+2)*(N+2);

  for ( i=0 ; i<size ; i++ ) {
          u[i] = v[i] = d[i] = 0.0f;
  }
  
  ui_d = fabs(wiimote1->gforce.z) * 15.0;
  i = 31;
  j = 3;
  d[IX(i, j)] = ui_d;
  u[IX(i, j)] = wiimote1->gforce.x * 10.0;
  v[IX(i, j)] = 5.0;
  
  i = 31;
  j = 60;
  d[IX(i, j)] = -ui_d / 3.0;
  u[IX(i, j)] = wiimote1->gforce.x * 10.0;
  v[IX(i, j)] = -5.0;

  i = 2;
  j = 30;
  d[IX(i, j)] = -ui_d;
  u[IX(i, j)] = 10.0;
  v[IX(i, j)] = wiimote1->gforce.y * 10.0;

  if (IS_PRESSED(wiimote1, WIIMOTE_BUTTON_B))
  {
    i = N - 1;
    for (j=N/2+1; j<=N/2+6; j++)
    {
      d[IX(i, j)] = -3.0;
      u[IX(i, j)] = -10.0;
      v[IX(i, j)] = -5.0;
    }
  }
  
  if (IS_PRESSED(wiimote1, WIIMOTE_BUTTON_A))
  {
    i = N - 1;
    for (j=N/2-6; j<=N/2-1; j++)
    {
      d[IX(i, j)] = 5.0;
      u[IX(i, j)] = -10.0;
      v[IX(i, j)] = 5.0;
    }
  }
  /*
  i = 1;
  j = 1;
  d[IX(i, j)] = -0.1;
  u[IX(i, j)] = 1.0;
  v[IX(i, j)] = 1.0;
  */
}

static void update_dens(float dt)
{
  float visc = 0.0;
  float diff = 0.0;
  int N = GRID_N;
  get_from_UI ( N, dens_prev, u_prev, v_prev );
  vel_step ( N, u, v, u_prev, v_prev, visc, dt );
  dens_step ( N, dens, dens_prev, u, v, diff, dt );
}

static int tick_counter = 0;

static void draw_dens(int N, float *dens)
{
  float x = 0, y = 0, z = 30-50;
  float r = 1.0f, g = 1.0f, b = 1.0f, a = 1.0f;
  float d;
  int i, j;
  float size = 0.3;
  /*
  if (wiimote_ticks_z)
    tick_counter = 100;
  
  if (tick_counter > 0)
    r = 0.0;
  
  tick_counter -= frame_ticks;
  */
  
  glLoadIdentity();
  //glTranslatef(-5, 0, 0);
  for (j=1; j<=N-1; j++)
  {
    for (i=1; i<=N-1; i++)
    {
      x = (i - N/2) * size;
      y = (j - N/2) * size;
      
      glBegin(GL_TRIANGLE_STRIP);
      
      d = dens[IX(i,j)];
      if (d > 1.0) d = 1.0;
      if (d < 0.0) d = 0.0;
      glColor4f(d, d, d, 1.0);
      glVertex3f(x, y, z);
      
      d = dens[IX(i,j+1)];
      if (d > 1.0) d = 1.0;
      if (d < 0.0) d = 0.0;
      glColor4f(d, d, d, 1.0);
      glVertex3f(x, y + size, z);
      
      d = dens[IX(i+1,j)];
      if (d > 1.0) d = 1.0;
      if (d < 0.0) d = 0.0;
      glColor4f(d, d, d, 1.0);
      glVertex3f(x + size, y, z);
      
      d = dens[IX(i+1,j+1)];
      if (d > 1.0) d = 1.0;
      if (d < 0.0) d = 0.0;
      glColor4f(d, d, d, 1.0);
      glVertex3f(x + size, y + size, z);
      
      glEnd();
    }
  }
}


void draw_fluid_plane()
{
  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  
  update_dens(frame_ticks / 100.0);
  draw_dens(GRID_N, dens);
  
  if (1)
  {
    glLoadIdentity();
    glColor4f(1.0, 1.0, 1.0, 1.0);
    draw_particle(-8, wiimote_x, -20);
    draw_particle(-10, wiimote_z, -20);
  }
  
  //printf("%f %f\n", wiimote_z, wiimote_amp_z);
  /*
  glLoadIdentity();
  glTranslatef(0, 0, -10);
  
  glBegin(GL_TRIANGLE_STRIP);
  glColor4f(1.0, 0.0, 0.0, 1.0);
  glVertex3f(1.0, 0.0, 0.0);
  
  glColor4f(0.0, 0.0, 0.0, 1.0);
  glVertex3f(0.0, 0.0, 0.0);
  
  glColor4f(1.0, 1.0, 1.0, 1.0);
  glVertex3f(1.0, 1.0, 0.0);
  
  glColor4f(0.0, 1.0, 0.0, 1.0);
  glVertex3f(0.0, 1.0, 0.0);
  glEnd();
  */
}
