#include <SDL.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <math.h>
#include "sdl.h"
#include "timer.h"
#include "wiimote.h"

static GLuint particle;

#define GRID_N 64
#define SIZE ((GRID_N+2)*(GRID_N+2))
#define COLORS_N 100
#define PALETTE_N 2
#define MAX_DENSITY 1
#define SWAP(x0,x) {float *tmp=x0;x0=x;x=tmp;}
#define IX(i,j) ((i)+(N+2)*(j))

static float rotation = 0.0;

static float u[SIZE], v[SIZE], u_prev[SIZE], v_prev[SIZE];
static float dens[SIZE], dens_prev[SIZE];

struct color {
  float r;
  float g;
  float b;
};

static struct color *colors;
static struct color *pals[PALETTE_N];

static int current_pal = 0;

static float ui_u = 5.0;
static float ui_v = 0.0;
static float ui_d = 5.0;


static float fluid_x = GRID_N / 2;
static float fluid_y = GRID_N / 2;

void gradient(struct color *colors, int pos1, int pos2, float r1, float g1, float b1, float r2, float g2, float b2) {
  int i=0;
  float step = pos2 - pos1;
  float r_step = (r2 - r1) / step;
  float g_step = (g2 - g1) / step;
  float b_step = (b2 - b1) / step;
  
  //printf("%f %f %f", r_step, g_step ,b_step);
  
  for (i = pos1; i < pos2; i++)
  {  
    colors[i].r = r1 + (i - pos1) * r_step;
    colors[i].g = g1 + (i - pos1) * g_step;
    colors[i].b = b1 + (i - pos1) * b_step;
  }
}

void init_fluid()
{
  load_texture("particle.bmp", &particle);
 
  glEnable( GL_TEXTURE_2D );
  glShadeModel( GL_SMOOTH );

  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
  glDisable(GL_DEPTH_TEST);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE);
  
  int i;
  struct color *pal;
  for (i = 0; i < PALETTE_N; i++){
    pal = (struct color *)malloc(COLORS_N * sizeof(struct color));
    switch(i) {
      case 0:
        gradient(pal,0,20,0,0,0,.5,0,0);
        gradient(pal,20,30,.5,0,0,1,0,0);
        gradient(pal,30,90,1,0,0,1,1,0);
        gradient(pal,90,100,1,1,0,1,1,0.7);
      break;
      case 1:
        gradient(pal,0,20,0,0,0,0,0.5,0);
        gradient(pal,20,30,0,.5,0,0,1,0);
        gradient(pal,30,90,0,1,0,0,1,1);
        gradient(pal,90,100,0,1,1,0.7,1,1);
      break;
      case 2:
        gradient(pal,0,20,0,0,0,1,0,0);
        gradient(pal,20,40,1,0,0,1,1,0);
        gradient(pal,40,60,1,1,0,0,1,1);
        gradient(pal,60,80,0,1,1,0,1,1);
        gradient(pal,60,80,0,1,1,0,1,1);
        gradient(pal,80,100,0,1,1,1,0,1);
        break;
      case 3:
        gradient(pal,0,20,0,0,0,.5,.5,.5);
        gradient(pal,20,80,.5,.5,.5,.8,.8,.8);
        gradient(pal,80,100,.8,.8,.8,1,1,1);
        break;
    }
    pals[i] = pal;
  }
  
  colors = pals[0];
/*
  glClearDepth(1.0f);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
  glHint(GL_POINT_SMOOTH_HINT, GL_NICEST);
*/
}

static void draw_particle(float x, float y, float z)
{
  float size = 0.3;
  glBindTexture( GL_TEXTURE_2D, particle );
  glBegin(GL_TRIANGLE_STRIP);
  glTexCoord2d(1, 1);
  glVertex3f(x + size, y + size, z);
  glTexCoord2d(0, 1);
  glVertex3f(x - size, y + size, z);
  glTexCoord2d(1, 0);
  glVertex3f(x + size, y - size, z);
  glTexCoord2d(0, 0);
  glVertex3f(x - size, y - size, z);
  glEnd();
}

static void set_bnd ( int N, int b, float * x )
{
  int i;
  for ( i=1 ; i<=N ; i++ )
  {
    x[IX(0,i)]   = b==1 ? -x[IX(1,i)] : x[IX(1,i)];
    x[IX(N+1,i)] = b==1 ? -x[IX(N,i)] : x[IX(N,i)];
    x[IX(i,0)]   = b==2 ? -x[IX(i,1)] : x[IX(i,1)];
    x[IX(i,N+1)] = x[IX(i,N)];
  }
  x[IX(0 ,0 )] =   0.5*(x[IX(1,0 )]+x[IX(0 ,1)]);
  x[IX(0 ,N+1)] =  0.5*(x[IX(1,N+1)]+x[IX(0 ,N )]);
  x[IX(N+1,0 )] =  0.5*(x[IX(N,0 )]+x[IX(N+1,1)]);
  x[IX(N+1,N+1)] = 0.5*(x[IX(N,N+1)]+x[IX(N+1,N )]);
}

static void project ( int N, float * u, float * v, float * p, float * div )
{
    int i, j, k;
    float h;
    h = 1.0/N;
    for ( i=1 ; i<=N ; i++ ) {
        for ( j=1 ; j<=N ; j++ ) {
            div[IX(i,j)] = -0.5*h*(u[IX(i+1,j)]-u[IX(i-1,j)]+
                                   v[IX(i,j+1)]-v[IX(i,j-1)]);
            p[IX(i,j)] = 0;
        }
    }
    set_bnd ( N, 0, div ); set_bnd ( N, 0, p );
    for ( k=0 ; k<20 ; k++ ) {
        for ( i=1 ; i<=N ; i++ ) {
            for ( j=1 ; j<=N ; j++ ) {
                p[IX(i,j)] = (div[IX( i,j)]+p[IX(i-1,j)]+p[IX(i+1,j)]+
                                            p[IX(i,j-1)]+p[IX(i,j+1)])/4;
            }
        }
        set_bnd ( N, 0, p );
    }
    for ( i=1 ; i<=N ; i++ ) {
        for ( j=1 ; j<=N ; j++ ) {
            u[IX(i,j)] -= 0.5*(p[IX(i+1,j)]-p[IX(i-1,j)])/h;
            v[IX(i,j)] -= 0.5*(p[IX(i,j+1)]-p[IX(i,j-1)])/h;
        }
    }
    set_bnd ( N, 1, u ); set_bnd ( N, 2, v );
}

static void add_source(int N, float * x, float * s, float dt)
{
  int i, size=(N+2)*(N+2);
  for ( i=0 ; i<size ; i++ ) x[i] += dt*s[i];
}

static void diffuse ( int N, int b, float * x, float * x0, float diff, float dt )
{
    int i, j, k;
    float a=dt*diff*N*N;
    for ( k=0 ; k<20 ; k++ ) {
        for ( i=1 ; i<=N ; i++ ) {
            for ( j=1 ; j<=N ; j++ ) {
                x[IX(i,j)] = (x0[IX(i,j)] + a*(x[IX(i-1,j)]+x[IX(i+1,j)]+
                                               x[IX(i,j-1)]+x[IX(i,j+1)]))/(1+4*a);
            }
        }
        set_bnd ( N, b, x );
    }
}

static void advect ( int N, int b, float * d, float * d0, float * u, float * v, float dt )
{
    int i, j, i0, j0, i1, j1;
    float x, y, s0, t0, s1, t1, dt0;
    dt0 = dt*N;
    for ( i=1 ; i<=N ; i++ ) {
        for ( j=1 ; j<=N ; j++ ) {
         x = i-dt0*u[IX(i,j)]; y = j -dt0*v[IX(i,j)];
         if (x<0.5) x=0.5; if (x>N+0.5) x=N+ 0.5; i0=(int)x; i1=i0+ 1;
         if (y<0.5) y=0.5; if (y>N+0.5) y=N+ 0.5; j0=(int)y; j1=j0+1;
         s1 = x-i0; s0 = 1-s1; t1 = y-j0; t0 = 1-t1;
         d[IX(i,j)] = s0*(t0*d0[IX(i0,j0)]+t1*d0[IX(i0,j1)])+
                       s1*(t0*d0[IX(i1,j0)]+t1*d0[IX(i1,j1)]);
      }
  }
  set_bnd ( N, b, d );
}

static void dens_step ( int N, float * x, float * x0, float * u, float * v, float diff,
    float dt )
{
    add_source ( N, x, x0, dt );
    SWAP ( x0, x ); diffuse ( N, 0, x, x0, diff, dt );
    SWAP ( x0, x ); advect ( N, 0, x, x0, u, v, dt );
}

static void vel_step ( int N, float * u, float * v, float * u0, float * v0,
           float visc, float dt )
{
    add_source ( N, u, u0, dt ); add_source ( N, v,v0, dt );
    SWAP ( u0, u ); diffuse ( N, 1, u, u0, visc, dt);
    SWAP ( v0, v ); diffuse ( N, 2, v, v0, visc, dt);
    project ( N, u, v, u0, v0 );
    SWAP ( u0, u ); SWAP ( v0, v );
    advect ( N, 1, u, u0, u0, v0, dt ); advect ( N,2, v, v0, u0, v0, dt );
    project ( N, u, v, u0, v0 );
}

static int changed_mark = 0;

void fluid_mark(int mark)
{
  changed_mark = 1;
  if (mark % 4 == 0)
    switch_colors();
}

static void get_from_UI (int N, float * d, float * u, float * v )
{
  int i, j, size = (N+2)*(N+2);
  float z,sum, sum_nchuk;
  float x_accel, y_accel;
  
  for ( i=0 ; i<size ; i++ ) {
          u[i] = v[i] = d[i] = 0.00f;
  }
  
  sum = wiimote1->gforce.x * wiimote1->gforce.x + wiimote1->gforce.y * wiimote1->gforce.y + wiimote1->gforce.z * wiimote1->gforce.z;
  //sum_nchuk = wiimote1->exp.nunchuk.gforce.x * wiimote1->exp.nunchuk.gforce.x + wiimote1->exp.nunchuk.gforce.y * wiimote1->exp.nunchuk.gforce.y + wiimote1->exp.nunchuk.gforce.z * wiimote1->exp.nunchuk.gforce.z;
  
  struct gforce_t source;
  float density;
  
  if (isfinite(wiimote1->exp.nunchuk.js.ang))
  {
    x_accel = wiimote1->exp.nunchuk.js.mag * sin(wiimote1->exp.nunchuk.js.ang / 180.0 * M_PI);
    y_accel = wiimote1->exp.nunchuk.js.mag * cos(wiimote1->exp.nunchuk.js.ang / 180.0 * M_PI);
    //printf("mag: %f ang: %f\n", wiimote1->exp.nunchuk.js.mag, wiimote1->exp.nunchuk.js.ang);
    //printf("cos(ang): %f\n", cos(wiimote1->exp.nunchuk.js.ang / 180.0 * M_PI));
    //printf("x_accel: %f y_accel: %f\n", x_accel, y_accel);
    
    fluid_x = GRID_N /  2 + x_accel * (GRID_N - 10) / 2;
    fluid_y = GRID_N / 2 + y_accel * (GRID_N - 10) / 2;
    
    if (fluid_x > GRID_N)
      fluid_x = GRID_N;
    if (fluid_x < 0)
      fluid_x = 0;
    if (fluid_y > GRID_N)
      fluid_y = GRID_N;
    if (fluid_y < 0)
      fluid_y = 0;
    
    //printf("%d %d\n", (int)round(fluid_x), (int)round(fluid_y));
  }
  
  //fluid_x = GRID_N / 2;
  //fluid_y = GRID_N / 2;
  
  if ((sum < 0.8 || sum > 1.2))
  {
    source = wiimote1->gforce;
    density = sum;

    d[IX((int)round(fluid_x), (int)round(fluid_y))] = density * 25;
    u[IX((int)round(fluid_x), (int)round(fluid_y))] = wiimote1->gforce.x * 10; 
    v[IX((int)round(fluid_x), (int)round(fluid_y))] = (wiimote1->gforce.z - 1) * 10;
  }
  
  if (changed_mark)
  {
    d[IX((int)round(fluid_x), (int)round(fluid_y))] += 1000;
    u[IX((int)round(fluid_x), (int)round(fluid_y))] += (rand() / (RAND_MAX + 1.0)) * 20.0 - 10.0;
    v[IX((int)round(fluid_x), (int)round(fluid_y))] += (rand() / (RAND_MAX + 1.0)) * 20.0 - 10.0;
    
    changed_mark = 0;
  }
  
 //printf("\nz: %f pitch: %f roll: %f density: %f sum: %f", wiimote1->gforce.z, wiimote1->orient.pitch, wiimote1->orient.roll, ui_d, sum);
  
}

void switch_colors()
{
    current_pal += 1;
    colors = pals[current_pal % PALETTE_N];
}

void update_fluid(){
  if (IS_JUST_PRESSED(wiimote1, WIIMOTE_BUTTON_B) ||
      IS_JUST_PRESSED(wiimote1, NUNCHUK_BUTTON_C))
  {
    switch_colors();
    //printf("%d: just pressed!!\n", frame_ticks );
  }
  if (IS_JUST_PRESSED(wiimote1, WIIMOTE_BUTTON_A) ||
      IS_JUST_PRESSED(wiimote1, NUNCHUK_BUTTON_Z))
  {
    current_pal -= 1;
    if (current_pal < 0)
      current_pal = PALETTE_N - 1;
    colors = pals[current_pal % PALETTE_N];
    //printf("%d: just pressed!!\n", frame_ticks );
  }
}

static void update_dens(float dt)
{
  float visc = 0.0;
  float diff = 0.0;
  int N = GRID_N, i;
  get_from_UI( N, dens_prev, u_prev, v_prev );
  vel_step ( N, u, v, u_prev, v_prev, visc, dt );
  dens_step( N, dens, dens_prev, u, v, diff, dt );
  
  for (i=0 ; i<(N+2)*(N+2) ; i++ ) {
    if (dens[i] > 1 )
      dens[i] -=  frame_ticks * 0.001f;
    else
      dens[i] -=  frame_ticks * 0.00025f;
    if (dens[i] < 0) dens[i] = 0;
  }
}

static int tick_counter = 0;

static void draw_dens(int N, float *dens)
{
  float x = 0, y = 0, z = 30-50;
  float r = 1.0f, g = 1.0f, b = 1.0f, a;
  int i, j, pos;
  
  glLoadIdentity();
  float min_d = 100, max_d = 0;
  float min_pos = 101, max_pos = 0;
  
  for (i=1; i<=N; i++)
    for (j=1; j<=N; j++)
    {
      x = (((float)i / GRID_N) * 2.0 - 1.0) * 12;
      y = (((float)j / GRID_N) * 2.0 - 1.0) * 8;
      
      pos = (int)round(dens[IX(i,j)] * COLORS_N);
      if (pos >= COLORS_N) pos = COLORS_N - 1;
      
      if (dens[IX(i,j)] > max_d) 
      {
        max_d = dens[IX(i,j)];
        max_pos = pos;
      }
      if (dens[IX(i,j)] < min_d) 
      {
        min_d = dens[IX(i,j)];
        min_pos = pos;
      }
      
      glColor4f(colors[pos].r, colors[pos].g, colors[pos].b, 1);
      draw_particle(x, y, z);
    }
    
    //printf ("density : min : %g pos : %g\n", min_d, min_pos);
    //printf ("density : max : %g pos : %g\n", max_d, max_pos);
}


void draw_palette()
{
  int i;
  for (i=0;i<COLORS_N;i++)
  {
    glLoadIdentity();
    glColor4f(colors[i].r, colors[i].g, colors[i].b, 1.0);
    draw_particle(-10, (i - COLORS_N / 2.0) / 5.0 , -40);
  }
}


void draw_fluid()
{
  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  
  update_dens(frame_ticks / 100.0);
  draw_dens(GRID_N, dens);
  
  if (0)
  {
    glLoadIdentity();
    glColor4f(1.0, 1.0, 1.0, 1.0);
    draw_particle(-8, wiimote_x, -20);
    draw_particle(-10, wiimote1->gforce.z - 1, -20);

  }
  
  //draw_palette();
}

