// -------------------------------------------------------------
// Phong Lighting
// 
// Copyright (c) 2003 Wolfgang F. Engel (wolf@direct3d.net)
// All rights reserved.
// -------------------------------------------------------------

// -------------------------------------------------------------
// variables that are provided by the application
// -------------------------------------------------------------
float4x4 matWorldViewProj;	
float4x4 matWorld;	
float4 LightPos;
float4 LightPos2;
float4 LightPos3;
float4 LightPos4;

float genSkinPos;

float LightRange = 0.0f;  
float LightRange2 = 0.0f;  
float LightRange3 = 0.0f;  
float LightRange4 = 0.0f;  

float4 vecEye;

texture ColorMap;
sampler ColorMapSampler = sampler_state
{
   Texture = <ColorMap>;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;   
   AddressU  = Clamp;
   AddressV  = Clamp;
};

texture BumpMap;
sampler BumpMapSampler = sampler_state
{
   Texture = <BumpMap>;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;   
   AddressU  = Clamp;
   AddressV  = Clamp;
};

// -------------------------------------------------------------
// Output channels
// -------------------------------------------------------------
struct VS_OUTPUT
{
    float4 Pos  : POSITION;
    float2 Tex : TEXCOORD0;
    float4 Light : TEXCOORD1;
    float3 View : TEXCOORD2;
    float4 Light2 : TEXCOORD4;
    float4 Light3 : TEXCOORD3;
    float4 Light4 : TEXCOORD5;     
};

// -------------------------------------------------------------
// vertex shader function (input channels)
// -------------------------------------------------------------
VS_OUTPUT VS(float4 Pos : POSITION, float2 Tex : TEXCOORD, float3 Normal : NORMAL, float3 Tangent : TANGENT  )
{
    VS_OUTPUT Out = (VS_OUTPUT)0;      
    Out.Pos = mul(Pos, matWorldViewProj);	// transform Position
    
    // compute the 3x3 tranform matrix 
    // to transform from world space to tangent space
    float3x3 worldToTangentSpace;
    worldToTangentSpace[0] = mul(Tangent, matWorld);
    worldToTangentSpace[1] = mul(cross(Tangent, Normal), matWorld);
    worldToTangentSpace[2] = mul(Normal, matWorld);
        
    Out.Tex = Tex.xy;

    float3 PosWorld = normalize(mul(Pos, matWorld));

      

    float3 Light = LightPos - PosWorld;
    Out.Light.xyz = normalize(mul(worldToTangentSpace, Light));							// L
    Out.Light.w = saturate(1 - dot(Light * LightRange, Light * LightRange));			// 1 - Attenuation    

    float3 Light2 = LightPos2 - PosWorld; 
    Out.Light2.xyz = normalize(mul(worldToTangentSpace, Light2));						// L2
    Out.Light2.w = saturate(1 - dot(Light2 * LightRange2, Light2 * LightRange2));			// 1 - Attenuation 

    float3 Light3 = LightPos3 - PosWorld; 
    Out.Light3.xyz = normalize(mul(worldToTangentSpace, Light3));						// L3
    Out.Light3.w = saturate(1 - dot(Light3 * LightRange3, Light3 * LightRange3));			// 1 - Attenuation  

    float3 Light4 = LightPos4 - PosWorld; 
    Out.Light4.xyz = normalize(mul(worldToTangentSpace, Light4));						// L4
    Out.Light4.w = saturate(1 - dot(Light4 * LightRange4, Light4 * LightRange4));			// 1 - Attenuation  

    float3 Viewer = vecEye - PosWorld;						
    Out.View = normalize(mul(worldToTangentSpace, Viewer));								// V
    
   return Out;
}

// -------------------------------------------------------------
// Pixel Shader (input channels):output channel
// -------------------------------------------------------------
float4 PS(  float2 Tex: TEXCOORD0, 
			float4 Light : TEXCOORD1, 
			float3 View : TEXCOORD2, 
			float4 Light2 : TEXCOORD4, 
			float4 Light3 : TEXCOORD3, 
			float4 Light4 : TEXCOORD5) : COLOR
{
    // fetch color map
    float4 Color = tex2D(ColorMapSampler, Tex);	
    
    // fetch bump map					
    float3 Normal = (2 * tex2D(BumpMapSampler, Tex)) - 1.0;				
	
    float3 LightDir = normalize(Light.xyz);
    float3 LightDir2 = normalize(Light2.xyz);
    float3 LightDir3 = normalize(Light3.xyz);
    float3 LightDir4 = normalize(Light4.xyz);
    
    float3 ViewDir = normalize(View);

    float Diff = saturate(dot(Normal,  LightDir));					
    float Diff2 = saturate(dot(Normal, LightDir2));					  
    float Diff3 = saturate(dot(Normal, LightDir3));					 
    float Diff4 = saturate(dot(Normal, LightDir4));					  
    
	float3 Reflect = normalize(2.0f * Normal * Diff - LightDir);		
	float Spec = pow(saturate(dot(Reflect, ViewDir)), 15);

    float3 Reflect2 = normalize(2.0f * Normal * Diff2 - LightDir2);		
    float Spec2 = pow(saturate(dot(Reflect2, ViewDir)), 15);  

    float3 Reflect3 = normalize(2.0f * Normal * Diff3 - LightDir3);		
    float Spec3 = pow(saturate(dot(Reflect3, ViewDir)), 15);  

    float3 Reflect4 = normalize(2.0f * Normal * Diff4 - LightDir4);		
    float Spec4 = pow(saturate(dot(Reflect4, ViewDir)), 15);
    
    

	float4 ret = Color * 0.2 + (((Color * Diff + Spec) * (Light.w)) + 
							((Color * Diff2 + Spec2) * (Light2.w)) + 
							((Color * Diff3 + Spec3) * (Light3.w)) + 
							((Color * Diff4 + Spec4) * (Light4.w))); 
							
	ret.a = (Tex.y * genSkinPos)+genSkinPos;
										
	return ret;
}

// -------------------------------------------------------------
// 
// -------------------------------------------------------------
technique TShader
{
    pass HLSL
    {
        Sampler[0] = (ColorMapSampler);		
        Sampler[1] = (BumpMapSampler);		
    
        // compile shaders
        VertexShader = compile vs_1_1 VS();
        PixelShader  = compile ps_2_0 PS();
    }
}
