#include "consts.fxh"
#include "lambertblinn.fxh"

textureCUBE sceneCubeTex <
	string TextureType = "CUBE";
	string UIName = "Scene Cube Map";
	bool VIExport = true;
>;

float2 swapNormal = float2(1, 1);

float IOR = 1.5;
float time = 0;


float bumpIntensity <
    string UIName = "Bump Intensity";
    float UIMin = 0;
    float UIMax = 1;
    float UIStep = 0.01;
    bool KGEexport = true;
> = 1;

samplerCUBE sceneCubeSamp = sampler_state {
	texture = <sceneCubeTex>;
	magFilter = linear;
	minFilter = linear;
	mipFilter = linear;
};



int texcoord0 : Texcoord <
    int Texcoord = 0;
    int MapChannel = 1;
    string UIName = "dont use";
    string UIWidget = "none";
>;



struct CWVertexInput {
	float3 oPosition: POSITION;
	float3 oNormal: NORMAL;
	float3 oTangent: TANGENT;
	float3 oBinormal: BINORMAL;
	float2 texCoord: TEXCOORD;
};


struct CWPixelInput {
	float4 pPosition: POSITION;
	float2 texCoord: TEXCOORD0;
	float3 wPosition: TEXCOORD1;
	float3 wTangentBasis1: TEXCOORD2;
	float3 wTangentBasis2: TEXCOORD3;
	float3 wTangentBasis3: TEXCOORD4;
};


struct CWPixelOutput {
	float4 color: COLOR;
};


CWPixelInput CubicWaterVS(CWVertexInput input) {
	// transform position
	float4 oPosition = float4(input.oPosition, 1);
	float4 wPosition = mul(oPosition, matObjectToWorld);
	float4 vPosition = mul(wPosition, matWorldToView);
	float4 pPosition = mul(vPosition, matViewToProj);
	
	float3 wTangent = -mul(input.oBinormal, (float4x3)matObjectToWorld);
	float3 wBinormal = mul(input.oTangent, (float4x3)matObjectToWorld);
	float3 wNormal = mul(input.oNormal, (float4x3)matObjectToWorld);
	
	// create output
	CWPixelInput output = (PixelInput)0;
	output.texCoord = input.texCoord;
	output.pPosition = pPosition;
	output.wPosition = wPosition;
	output.wTangentBasis1 = wTangent;
	output.wTangentBasis2 = wBinormal;
	output.wTangentBasis3 = wNormal;
	return output;
}



float Fresnel(float3 normal, float3 viewDir, float ior){
   float cosTheta = dot(normal, viewDir);
   float fresnel0 = pow(ior-1, 2) / pow(ior+1, 2);
   float fresnel = fresnel0 + pow(1-cosTheta, 5) * (1-fresnel0);
   return fresnel;
}


CWPixelOutput CubicWaterPS(CWPixelInput input) {
	CWPixelOutput output;

	float3 wNormal = normalize(input.wTangentBasis3);
	float3 wCameraDir = normalize(cameraPosition - input.wPosition);
	float3 tCameraDir;
	tCameraDir.x = dot(wCameraDir, input.wTangentBasis1);
	tCameraDir.y = dot(wCameraDir, input.wTangentBasis2);
	tCameraDir.z = dot(wCameraDir, input.wTangentBasis3);
	tCameraDir = normalize(tCameraDir);
	
	float3 tNormal = normalize(tex2D(normalSamp, input.texCoord)*2-1);
	tNormal.xy *= swapNormal;
	tNormal = lerp(float3(0, 0, 1), tNormal, bumpIntensity);
	wNormal += tNormal.x * input.wTangentBasis1;
	wNormal += tNormal.y * input.wTangentBasis2;
	float2 t;
	t.x = cos(time)*wNormal.x - sin(time)*wNormal.y;
	t.y = sin(time)*wNormal.x + cos(time)*wNormal.y;
	wNormal = normalize(float3(t, wNormal.z));
	
	float3 wReflectDir = reflect(wCameraDir, wNormal);
	float3 wRefractDir = refract(wCameraDir, wNormal, 1 / IOR);
	float4 sceneCubeReflectTx = texCUBE(sceneCubeSamp, wReflectDir);
	float fresnel = 1 - Fresnel(tNormal, tCameraDir, IOR);
	
	float3 result = 0;
	result += sceneCubeReflectTx;
	
	output.color = float4(result, fresnel);
	return output;
}


CWPixelOutput CubicWaterVolumePS(CWPixelInput input, uniform bool useLightCube) {
	CWPixelOutput output;
	output.color.rgb = ComputeVolumeLight(depthShadowSamp, lightCubeSamp, input.wPosition, useLightCube) * lightColor * 0.8;
	output.color.a = 1;
	return output;
}


technique defaultSh {
	pass ambient {
		vertexShader = compile vs_3_0 CubicWaterVS();
		pixelShader = compile ps_3_0 CubicWaterPS();

/*		alphaBlendEnable = true;
		srcBlend = SRCALPHA;
		destBlend = INVSRCALPHA;*/
	}
	
	pass shadowDepth {
		vertexShader = compile vs_3_0 ShadowDepthVS();
		pixelShader = compile ps_3_0 ShadowDepthPS();
	}
	
	
	
	pass diffuseVolume {
		vertexShader = compile vs_3_0 CubicWaterVS();
		pixelShader = compile ps_3_0 CubicWaterVolumePS(false);
	}
	
	pass diffuseLightCube {
		vertexShader = compile vs_3_0 CubicWaterVS();
		pixelShader = compile ps_3_0 CubicWaterVolumePS(true);
	}
}
