<?php

require_once('DB.php');
require_once('ProtocolViolationException.php');

/**
 * Manages a game session.
 *
 * The game proceeds in lockstep simulation in which clients may continue
 * to simulate past any point in time only if permitted by the server.
 * The server only extends this time limit when all clients have
 * received all commands scheduled for before that time.
 */
final class SessionManager {

    /**
     * How many ping entries are stored per client for averaging.
     */
    const PING_AVG_WINDOW = 20;

    /**
     * @var DB
     */
    private $_db;

    /**
     * @var int
     */
    private $_endOfTimeIncrement;

    /**
     * (cache for database query)
     * @var boolean
     */
    private $_isPaused;

    /**
     * Constructor.
     *
     * @param DB $db The database connection.
     * @param int $sessionId The ID of the session (set of clients) to handle.
     */
    public function __construct(DB $db) {
        $this->_db = $db;
        $this->_isPaused = null;
    }

    /**
     * Creates a new empty joinable session.
     *
     * @param string $name The session name.
     * @param int $masterClientId The ID of the master client.
     * @param int $maxClients The maximum number of connections.
     * @param string $metadata Session metadata.
     * @return array (session id, master key)
     */
    public function createSession($name, $masterClientId, $maxClients, $metadata) {
        $masterKey = md5(uniqid(rand(), true));
        assert('strlen($masterKey) == 32');

        $q = 'INSERT INTO session ' .
             '(id, name, joinable, max_clients, master_client, master_key, metadata) ' .
             'VALUES (nextval(pg_get_serial_sequence(\'session\', \'id\')), ?, TRUE, ?, ?, ?, ?)';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($name, $maxClients, $masterClientId, $masterKey, $metadata));

        // PDO doesn't support INSERT ... RETURNING, so we have to get the new ID using currval()
        $q = 'SELECT currval(pg_get_serial_sequence(\'session\', \'id\'))';
        $stmt = $this->_db->prepare($q);
        $stmt->execute();
        $sessionId = $stmt->fetchColumn();
        assert('ctype_digit($sessionId)');
        $sessionId = intval($sessionId);

        return array($sessionId, $masterKey);
    }

    /**
     * Creates a new client.
     * @param string $name The name of the client.
     * @return int The client ID.
     */
    public function createClient($name)
    {
        $q = 'INSERT INTO client (id, name) ' .
             'VALUES (nextval(pg_get_serial_sequence(\'client\', \'id\')), ?)';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($name));

        $q = 'SELECT currval(pg_get_serial_sequence(\'client\', \'id\'))';
        $stmt = $this->_db->prepare($q);
        $stmt->execute();
        $clientId = $stmt->fetchColumn();
        assert('ctype_digit(strval($clientId))');
        return intval($clientId);
    }

    /**
     * Returns the data of the client, if it exists.
     *
     * @return array|null
     */
    private function _getClient($clientId)
    {
        $q = 'SELECT * FROM client WHERE id = ?';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($clientId));
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * Returns whether a client exists in the database.
     *
     * @param int $clientId
     * @return boolean Whether the client exists.
     */
    public function clientExists($clientId)
    {
        return $this->_getClient($clientId) ? true : false;
    }

    /**
     * Returns the ID of the session the client is in (if any).
     *
     * @param int $clientId
     * @return int|null The session ID.
     */
    public function getClientSession($clientId)
    {
        $client = $this->_getClient($clientId);
        if ($client)
            return $client['session'];
        else
            return null;
    }

    /**
     * Checks whether the client is a master client.
     *
     * @param int $clientId
     * @param string $masterKey
     * @return boolean Whether the client is a master client.
     */
    public function isMasterClient($clientId, $masterKey)
    {
        $q = 'SELECT 1 FROM session WHERE master_client = ? AND master_key = ?';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($clientId, $masterKey));
        return $stmt->fetch(PDO::FETCH_ASSOC) ? true : false;
    }

    /**
     * Returns a list of joinable sessions.
     *
     * @return array mapping session ID to session name.
     */
    public function getJoinableSessions() {
        $stmt = $this->_db->prepare('SELECT id, name FROM session WHERE joinable = TRUE');
        $stmt->execute();
        $ret = array();
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $ret[$row['id']] = $row['name'];
        }
        return $ret;
    }

    /**
     * Joins a client to a session, if possible.
     *
     * The players player number if selected automatically.
     * 
     * @param int $sessionId The ID of the session to join.
     * @param int The ID of the joining client.
     * @return int The assigned player number.
     * @throws Exception If the session cannot be joined for any reason.
     */
    public function joinSession($sessionId, $clientId) {
        $this->_db->beginTransaction();

        $sessProps = $this->getSessionProperties($sessionId);
        if (!$sessProps['joinable'])
            throw new ProtocolViolationException('This session cannot be joined any more.');

        $allPlayerNums = range(1, $sessProps['maxClients']);
        $usedPlayerNums = array();
        foreach ($sessProps['clients'] as $clientProps) {
            $usedPlayerNums[] = intval($clientProps['playerNum']);
        }
        $availPlayerNums = array_diff($allPlayerNums, $usedPlayerNums);
        sort($availPlayerNums);

        if (empty($availPlayerNums))
            throw new Exception('No more players may join.');

        $playerNum = array_shift($availPlayerNums);

        /*
         * Depending on the kind of transaction PDO gives us, this may or
         * may not work reliably. Can we be sure that two processes can't
         * arrive at this point simultaneously, having the same idea of
         * available player numbers?
         */

        $q = 'UPDATE client ' .
             'SET session = ?, ' .
             '    player_num = ? ' .
             'WHERE id = ?';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($sessionId, $playerNum, $clientId));

        if ($stmt->rowCount() !== 1)
            throw new Exception("Can't join that session.");

        $this->_db->commit();
        return $playerNum;
    }

    /**
     * Drops lagging clients from a session and deletes the entire session
     * if all clients have left and the session is over 10 seconds old.
     *
     * @param int $sessionId The session to clean.
     * @param int $dropTimeLimit The number of seconds of lagging to allow.
     */
    public function dropLaggers($dropTimeLimit = 30) {
        $clientDropTimeLimit = intval($dropTimeLimit) . ' seconds';
        $sessionDropTimeLimit = '10 seconds';

        $this->_db->exec("DELETE FROM client WHERE last_communication_time + '$clientDropTimeLimit' < CURRENT_TIMESTAMP");

        $this->_db->exec('DELETE FROM session AS s ' .
                         'WHERE (SELECT COUNT(*) FROM client AS c WHERE c.session = s.id) = 0 AND ' .
                         "      create_time_abs + '$sessionDropTimeLimit' < CURRENT_TIMESTAMP");
    }

    /**
     * Returns the properties of the session.
     *
     * Returns an array with the following fields:
     * - name: the session name.
     * - joinable: whether the session is joinable.
     * - maxClients: the maximum number of clients.
     * - metadata: the session metadata.
     * - masterClient: the master client ID.
     * - clients: an array of client records indexed by client ID,
     *            which are arrays with the fields
     *            'id', 'name', 'playerNum' and 'timeSinceLastUpdate' (ms).
     *
     * @param int $sessionId The session ID.
     * @return array (see above)
     * @throws Exception
     */
    public function getSessionProperties($sessionId) {
        $q = 'SELECT name, joinable, max_clients, master_client, metadata FROM session WHERE id = ?';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($sessionId));
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row)
            throw new Exception('The session doesn\'t exist.');

        $props['name'] = $row['name'];
        $props['joinable'] = $row['joinable'] ? true : false;
        $props['maxClients'] = $row['max_clients'];
        $props['metadata'] = $row['metadata'];
        $props['masterClient'] = $row['master_client'];
        $props['clients'] = array();

        $q = 'SELECT id, '.
             '       name, '.
             '       player_num, ' .
             '       TO_CHAR(CURRENT_TIMESTAMP - last_communication_time, \'MI SS MS\') AS tsl ' .
             'FROM client WHERE session = ? ' .
             'ORDER BY player_num, id';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($sessionId));
        while ($client = $stmt->fetch(PDO::FETCH_ASSOC)) {
            list($min, $sec, $ms) = explode(' ', $client['tsl']);
            $client['timeSinceLastUpdate'] = $min * 60*1000 + $sec * 1000 + $ms;
            unset($client['tsl']);
            $client['playerNum'] = $client['player_num'];
            unset($client['player_num']);
            $props['clients'][$client['id']] = $client;
        }

        return $props;
    }

    /**
     * Inserts a new chat message.
     *
     * @param int $sessionId The session ID.
     * @param int $sessionId The client ID.
     * @param string $sessionId The message.
     */
    public function addChatMessage($sessionId, $clientId, $msg)
    {
        $q = 'INSERT INTO chat_message (session, sender, msg) ' .
             'VALUES (?, ?, ?)';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($sessionId, $clientId, $msg));
    }

    /**
     * Returns chat messages as arrays with the following keys:
     * - id: The ID of the message.
     * - sender: The client ID of the sender.
     * - msg: The message.
     *
     * The messages are returned in ascending ID order.
     *
     * @param int $sessionId The session ID.
     * @param int $minId The minimum chat message ID to return.
     * @return array (see above)
     */
    public function getChatMessages($sessionId, $minId)
    {
        $q = 'SELECT id, sender, msg ' .
             'FROM chat_message ' .
             'WHERE session = ? AND ' .
             '      id >= ? ' .
             'ORDER BY id ASC';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($sessionId, $minId));
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Returns whether a game has started.
     *
     * A game is started iff it is not joinable.
     *
     * @param int $sessionId The session ID.
     * @return boolean Whether the game has started.
     * @throws ProtocolViolationException If the session doesn't exist.
     */
    public function hasGameStarted($sessionId)
    {
        $props = $this->getSessionProperties($sessionId);
        if (!$props)
            throw new ProtocolViolationException("Session doesn't exist");
        return $props['joinable'] ? false : true;
    }

    /**
     * Makes a session non-joinable and unpaused.
     *
     * @throws ProtocolViolationException If the session doesn't exist.
     */
    public function startGame($sessionId)
    {
        $q = 'UPDATE session SET joinable = FALSE, is_paused = FALSE WHERE id = ?';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($sessionId));
        if ($stmt->rowCount() != 1)
            throw new ProtocolViolationException("Session doesn't exist");
    }


    /**
     * Runs the query to update the end of time for the session.
     *
     * @param int $sessionId
     */
    private function _updateEndOfTime($sessionId) {

        /*
         * The new end of time will be the time offset of the first command
         * not received by all clients. It also has an upper limit set in
         * session.endOfTimeLimit.
         */

        //TODO: retry this transaction if there is indeed a serialization conflict
        $this->_db->exec('SET TRANSACTION ISOLATION LEVEL SERIALIZABLE');

        // The first subquery in LEAST may be NULL, which casues the second to be used.
        // This is what we want.
        $q = 'UPDATE session ' .
             'SET end_of_time = (' .
             '    SELECT LEAST(' .
             '            (SELECT time_offset ' .
             '             FROM command ' .
             '             WHERE session = ? AND ' .
             '                   serial = ' .
             '                       (SELECT MIN(last_command_synced) ' .
             '                        FROM client ' .
             '                        WHERE session = ?) + 1 ' .
             '            ), ' .
             '            (SELECT end_of_time_limit AS time_offset FROM session WHERE id = ?) ' .
             '        ) ' .
             ')';
        $this->_db->prepare($q)->execute(array($sessionId, $sessionId, $sessionId));
        
        $this->_db->exec('SET TRANSACTION ISOLATION LEVEL READ COMMITTED');
    }

    /**
     * Updates the last communication time of a client.
     *
     * This must be done periodically to avoid the client being dropped.
     *
     * @param int $clientId The client ID.
     */
    function updateLastCommunicationTime($clientId) {
        $clientId = intval($clientId);
        $this->_db->exec("UPDATE client SET last_communication_time = CURRENT_TIMESTAMP WHERE id = $clientId");
    }

    /**
     * Marks the last message a client has received.
     *
     * This also updates the last communication time of the client
     * (no need to call updateLastCommunicationTime() separately).
     *
     * @param int $sessionId The session ID.
     * @param int $clientId The client ID.
     * @param int $latestCommandId The ID of the command the client has received.
     *                           All previous command IDs will be considered received as well.
     */
    public function updateLastCommandSynced($sessionId, $clientId, $latestCommandId) {
        if ($latestCommandId === 0 || $latestCommandId === '0')
            $latestCommandId = null; // To satisfy FKs

        $q = 'UPDATE client ' .
             'SET last_command_synced = ?, ' .
             '    last_communication_time = CURRENT_TIMESTAMP ' .
             'WHERE id = ? AND ' .
             '      (last_command_synced IS NULL OR last_command_synced <= ?)';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($latestCommandId, $clientId, $latestCommandId));

        $this->_updateEndOfTime($sessionId);
    }

    /**
     * Sets a new upper limit to the end of known time value.
     * The upper limit must not be lower than any previously set limit.
     * If it is, this method does nothing.
     *
     * @param int $sessionId The session ID.
     * @param int $newLimit The new limit.
     */
    public function setEndOfTimeLimit($sessionId, $newLimit) {
        $q = 'UPDATE session SET end_of_time_limit = ? WHERE id = ? AND end_of_time_limit < ?';
        $this->_db->prepare($q)->execute(array($newLimit, $sessionId, $newLimit));

        $this->_updateEndOfTime($sessionId);
    }

    /**
     * Enqueues an command for the next timestep.
     *
     * @param int $sessionId The session ID.
     * @param int $clientId The ID of the client that sent the command.
     * @param string $commandMessage The message of the command.
     * @return int The ID of the new command.
     * @throws ProtocolViolationException If the command message is too long or the current session is invalid.
     */
    public function enqueueCommand($sessionId, $clientId, $commandMessage) {
        $q = 'INSERT INTO command (session, serial, time_offset, message) ' .
             'SELECT session.id, ' .
             '       COALESCE( (SELECT MAX(serial) + 1 FROM command WHERE session = ?), 1), ' .
             '       end_of_time, ' .
             '       ? ' .
             'FROM session WHERE session.id = ?';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($sessionId, $commandMessage, $sessionId));
        if ($stmt->rowCount() != 1)
            throw new ProtocolViolationException("Failed to store command: $commandMessage in session $sessionId", $clientId);

        // PDO doesn't support RETURNING, so we have to use currval to get the ID of the new command
        $q = 'SELECT currval(pg_get_serial_sequence(\'command\', \'id\'))';
        $stmt = $this->_db->prepare($q);
        $stmt->execute();
        $commandId = $stmt->fetchColumn();
        assert('ctype_digit(strval($commandId))');
        return $commandId;
    }

    /**
     * Returns the last command that was sent to the client.
     *
     * @param int $clientId
     * @return int The ID of the last command sent to the client.
     */
    public function getLastSyncedCommand($clientId) {
        $q = 'SELECT last_command_synced FROM client WHERE id = ?';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($clientId));
        return $stmt->fetchColumn();
    }

    /**
     * Returns all commands for the next timestep that a client has not yet received.
     *
     * @param int $sessionId The session ID.
     * @param int $minId The smallest command ID to send.
     * @return array A mapping of command serial => array(time offset, message)
     *               [ordered by command serial asc].
     */
    public function getNewCommands($sessionId, $minId) {
        $q = 'SELECT c.id AS id, c.serial AS serial, c.time_offset AS time_offset, c.message AS message ' .
             'FROM command AS c ' .
             'WHERE c.session = ? AND ' .
             '      c.serial >= ? ' .
             'ORDER BY c.serial ASC';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($sessionId, $minId));

        $commands = array();
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $commands[$row['serial']] = array($row['time_offset'], $row['message']);
        }
        return $commands;
    }

    /**
     * Returns the maximum time offset to which the clients may simulate.
     * This is the time up to which all clients have received all commands.
     *
     * @param int $sessionId The session ID.
     * @return int The time offset in milliseconds.
     */
    public function getEndOfTime($sessionId) {
        $q = 'SELECT end_of_time FROM session WHERE id = ?';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($sessionId));
        return $stmt->fetchColumn();
    }

    /**
     * Stores a ping value reported by a client.
     *
     * @param int $clientId The client ID.
     * @param int $ping The ping, in ms, reported by the client.
     */
    public function storePing($clientId, $ping) {
        $q = 'INSERT INTO client_ping (client, ping) VALUES (?, ?)';
        $this->_db->prepare($q)->execute(array($clientId, $ping));

        $q = 'DELETE FROM client_ping WHERE client = ? AND id <= ' .
             '  (SELECT id FROM client_ping WHERE client = ? ORDER BY id DESC LIMIT 1 OFFSET ?)';
        $this->_db->prepare($q)->execute(array($clientId, $clientId, self::PING_AVG_WINDOW));
    }

    /**
     * Stores a game state checksum in the checksum log.
     *
     * @param int $sessionId The session ID.
     * @param int $clientId The client ID.
     * @param int $gameTime The time when the checksum was taken.
     * @param string $checksum The checksum string.
     * @throws Exception If something goes wrong (such as a duplicate row).
     */
    public function logChecksum($sessionId, $clientId, $gameTime, $checksum)
    {
        $q = 'INSERT INTO checksum_log (session, sender, game_time, checksum) ' .
             'VALUES (?, ?, ?, ?)';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($sessionId, $clientId, $gameTime, $checksum));
        if ($stmt->rowCount() !== 1)
            throw new ProtocolViolationException('Failed to log game state checksum');
    }

    /**
     * Verifies that the checksum log matches that of the master client's log.
     *
     * @param int $sessionId The session ID.
     * @param int $clientId The current client ID.
     * @return int|null Returns the game time of the earliest mismatch,
                        or null if all is good.
     * @throws Exception If something goes wrong (such as a duplicate row).
     */
    public function verifyChecksums($sessionId, $clientId)
    {
        $sessProps = $this->getSessionProperties($sessionId);
        if (!$sessProps)
            throw new Exception("Session not found.");
        $masterClientId = $sessProps['masterClient'];
        if ($clientId == $masterClientId)
            return null;

        $q = 'SELECT l.game_time AS game_time ' .
             'FROM checksum_log AS l ' .
             'INNER JOIN checksum_log AS ml ' . // Join with the master client's log
             '  ON (ml.sender = ? AND ' .
             '      ml.game_time = l.game_time AND ' .
             '      ml.checksum != l.checksum) ' .
             'WHERE l.sender = ? ' .
             'ORDER BY game_time ASC ' .
             'LIMIT 1';
        $stmt = $this->_db->prepare($q);
        $stmt->execute(array($masterClientId, $clientId));
        $gameTime = $stmt->fetchColumn();
        if ($gameTime === false)
            return null;
        else
            return $gameTime;
    }

    /**
     * Writes an entry to the debug log.
     *
     * @param int $clientId
     * @param int $gameTime
     * @param string $msg
     * @return int
     */
    public function debugLog($clientId, $gameTime, $msg)
    {
        $q = 'INSERT INTO debug_log (sender, game_time, msg) VALUES (?, ?, ?)';
        $this->_db->prepare($q)->execute(array($clientId, $gameTime, $msg));
    }
}

// other. get time from epox in millseconds
function utime()
{
    $utime = preg_match("/^(.*?) (.*?)$/", microtime(), $match);
    $utime = $match[2] + $match[1];
    $utime *= 1000;
    return round($utime);
}
