
/*
 * GameObj:
 * - id: a unique identification number (> 0).
 * - descriptor: a reference to a UnitDescriptor or similar object, as specified by the type field.
 * - cellX, cellY: coordinates in map cells (top right corner if large).
 * - isUnit: if the object is some sort of a moveable unit.
 * - Owner: Player's id, 0 if neutral.
 * 
 * If the game object is a moveable unit:
 * - destXY: map cell to which the object is commanded to move
 * - getNextCell(): returns the next cell on the way towards destX, destY as an [x, y] pair.
 * - direction: 0 for north, 1 for north-east, ..., 7 for north-west
 
 */
function GameObj(id, makerArray) {
    
    this.id = id;
    this.desc = makerArray[0];
    this.descriptor = new ObjectDescriptor(makerArray[0]);
    this.cellXY = [makerArray[1],makerArray[2]];
    this.owner = makerArray[3]; // Player id
    if (makerArray[4] < 0)
        this.hp = this.descriptor.maxhp;
    else
        this.hp = makerArray[4];
    this.selectedBuild = 0;
    this.buildPercent = 0;
    this.constructing = 0;
    this.isUnderConstruction = false;
    this.direction = makerArray[5];
    this.cellXYAdd = [0,0];
    this.attackTarget = false;
    this.destXY = this.cellXY;
    this.selectedBuild = 0;
}

/* GameObjects is a container for all game objects.
 * _obj contains all objects
 * getObjectById takes id and returns the gameobject of taht id.
 * getObjectByPos takes position as array (pos[0] = x pos[1] = y) and returns gameobject (slow)
 * addObject add's an object to the container with arguments name, posx, posy and owner (player id)
 * spawn - testing
 */

function GameObjects(load)
{
    // use next availible game object id.
    this.getFirstAvailibleId = function() {
        var availibleId = 1;
        while (this.getObjectById(availibleId))
            availibleId ++;
        
        return availibleId;
    }
    
    var self = this;
    this._obj = Array();
    
    this.getObjectById = function(id)
    {
        for (i in this._obj)
            if (this._obj[i].id == id)
                return this._obj[i];
        return false;
    }
    
    
    this.getObjectsSortedByDrawingPriority = function()
    {
        var sorted = this._obj.sort(function (a, b) {
            var diff;
            // All flat objects go before non-flat objects
            var aHeight = a.descriptor.isFlat ? 0 : 1;
            var bHeight = b.descriptor.isFlat ? 0 : 1;
            diff = aHeight - bHeight;
            if (diff != 0)
                return diff;
            
            // All structures go before non-structure objects
            var aStructure = a.descriptor.isUnit ? 1 : 0;
            var bStructure = b.descriptor.isUnit ? 1 : 0;
            diff = aStructure - bStructure;
            if (diff != 0)
                return diff;
            
            // Rows are drawn from top to bottom i.e. sort by Y
            var dy = (a.cellXY[1] + a.cellXYAdd[1]) -
                     (b.cellXY[1] + b.cellXYAdd[1]);
            diff = dy;
            if (diff != 0)
                return diff;
            
            // Columns are drawn from right to left
            var dx = (a.cellXY[0] + a.cellXYAdd[0]) -
                     (b.cellXY[0] + b.cellXYAdd[0]);
            diff = -dx;
            
            return diff;
        });

        return sorted;
    }
    
    this.getObjectsSortedById = function()
    {
        var sorted = this._obj.sort(function (a, b) {
            return a.id-b.id;
        });

        return sorted;
    }
    
    this.getFirstObjectByOwner = function(owner)
    {
        var obj = this.getObjectsSortedById();
        
        for (o in obj)
        {
            if (obj[o].owner == owner)
                return obj[o];
        }
    }
    
    
    // WARNING! slow function
    this.getNearestFreeRefinery = function(pos)
    {
        var near = 1000;
        var nearest = false;
        var obj = this.getObjectsSortedById();
        for (o in obj)
        {
            if (obj[o].desc == "refinery")
            {
                var xy = obj[o].cellXY;
                if (xy[0] > 0 && xy[1] > 0 && 
                    map.occupence.getOccupied([(xy[0]+2),xy[1]]) == obj[o].id)
                {
                    var comp = (Math.abs(parseInt(pos[0])-parseInt(xy[0]))
                               +Math.abs(parseInt(pos[1])-parseInt(xy[1])));
                    if (comp < near)
                    {
                        nearest = obj[o];
                        near = comp;
                    }
                }
            }
        }
        return nearest;
    }
    
    // pos = where i am
    // id = object's id
    // range = how close to it should we be
    this.isNextTo = function(pos,id,range)
    {
        var obj = this.getObjectById(id);
        if (!obj)
            return false;
        if (range == undefined)
            range = 1;
        
        return (within(pos,[(obj.cellXY[0]-range),(obj.cellXY[1]-range)],
            [(obj.descriptor.size[0]+range),(obj.descriptor.size[1]+range)]));
    }
    
    this.getObjectByPos = function(crd)
    {
        // there must be a better way than looping through every object
        // i'll fix it up if this becomes more used.
        var obj = this.getObjectsSortedById();
        for (o in obj)
        {
            if (within(crd,obj[o].cellXY,obj[o].descriptor.size))
                return obj[o];
        }
        return false;
    }
    this.addObject = function(array, id)
    {
        deb("addiing: "+id);
        if (id == undefined)
            id = this.getFirstAvailibleId();
        this._obj = this._obj.sort();
        var slot = this._obj.length;
        this._obj[slot] = new GameObj(id, array);
        return this._obj[slot];
    }
    
    // morph into some other object
    this.morph = function(obj, desc)
    {
        var ret = this.addObject([desc, 
            obj.cellXY[0],
            obj.cellXY[1],
            obj.owner,
            -1,
            obj.direction]);
        this.del(obj);
        return ret;
    }
    
    this.isNextToMyConstruction = function(crd, size)
    {
        for (var x = crd[0]; x <= crd[0]+size[0]; x ++)
            for (var y = crd[1]; y <= crd[1]+size[1]; y ++)
                if (x == crd[0] || x == crd[0]+size[0] ||
                    y == crd[1] || y == crd[1]+size[1])
                {
                    var obj = this.getObjectByPos([x,y]);
                    if (obj.descriptor != undefined)
                        if (!obj.descriptor.isUnit && obj.owner == settings.playerId)
                            return true;
                }
                
        return false;
    }
    
    this.getNumberOfObjectsOwned = function(desc, player)
    {
        var cnt = 0;
        for (i in this._obj)
            if (this._obj[i].desc == desc && this._obj[i].owner == player)
                cnt += 1;
        return cnt;
    }
    
    this.del = function(obj)
    {
        for (i in this._obj)
            if (this._obj[i].id == obj.id)
                delete(this._obj[i]);
        this._obj = this._obj.filter( function(element, index, array) {
            return (element != undefined);
        });
    }
    
    // unserialize game object, coming fro a save file.
    this.unserialize = function(serial)
    {
        if (serial == undefined)
        {
            serial = "1,mcv,5,5,1,-1,0|2,mcv,"+(map.mapSize[0]-5)+","+
            (map.mapSize[1]-5)+",2,-1,0|3,mcv,"+(map.mapSize[0]-5)+
            ",5,3,-1,0|4,mcv,5,"+(map.mapSize[1]-5)+",4,-1,0";
        }
        var strings = serial.split("|");
        for (var s in strings)
        {
            var obj = strings[s].split(",");
            
            this.addObject(
            [obj[1], 
            parseInt(obj[2]), 
            parseInt(obj[3]),
            parseInt(obj[4]),
            parseInt(obj[5]),
            parseInt(obj[6])], 
            
            parseInt(obj[0]));
        }
    }
    this.unserialize(load);
}
