
function ActionScheduler()
{
    this._queue = []; // Sorted by start time, then by id
    this._nextId = 1;
    this._time = 0;
    this._unprocessedColor = false;
    
    // actions use this time.
    this.getTime = function()
    {
        return this._time;
    };
    
    /*
     * Executes all uncompleted actions in the queue that have
     * started before time.getTime().
     */
    this.update = function()
    {
        /*
         * New actions added during the loop will be marked unprocessed.
         * We loop until all actions are marked processed.
         * 
         * This loop is O(n^2) but that shouldn't be too significant.
         * If it is, we could optimize the inner loop away by
         * detecting whether any actions were added.
         */
        var unprocessed = this._unprocessedColor;
        var processed = !unprocessed;
        var i = 0;
        
        while (true) {
            for (i = 0; i < this._queue.length; ++i) {
                if (this._queue[i].color == unprocessed)
                    break;
            }
            if (i == this._queue.length)
                break; // All actions have been marked as processed.
            
            var action = this._queue[i];
            
            action.color = processed;
            
            if (action.actionTime + action.length <= time.getTime()) {
                // Action has completed
                this._time = action.actionTime + action.length;
                debugLog("ACTION[" + action.id + "], " + action.actionTime + "(+" + action.length + "), p=1: " + action.func);
                action.func(1);
                this._queue.splice(i, 1);
                
            } else if (action.length) {
                // Action in progress.
                
                // Actions that use "progress" must not affect the essential synchronized game state
                // They are _not_ executed an identical number of times on each client.
                // Therefore:
                //   massive hax: restore the old time once this action is done
                var canonicalTime = this._time;
                this._time = time.getTime();
                
                var progress = (this._time - action.actionTime) / action.length;
                //debugLog("ACTION, p=" + progress + ": " + action.func);
                if (progress == 1) {
                    debugLog("Warning: got progress = 1 in partial action!", true);
                    progress = 0.99;
                }
                action.func(progress);
                
                this._time = canonicalTime;
            }
        }
        
        // All actions are marked with the 'processed' color. Flip it to mean 'unprocessed'.
        this._unprocessedColor = !this._unprocessedColor;
    }
    
    this.addAction = function(startTime, func, length)
    {
        assert(length == undefined || isNumeric(length));
        
        var action = {
            id: this._nextId++,
            actionTime: startTime,
            length: (length == undefined) ? 0 : length,
            func: func,
            color: this._unprocessedColor
        }

        debugLog("Scheduling action " + action.id + " for " + startTime + " at " + actions.getTime() + "/" + time.getTime() + "|" + time.getTimeLimit());
        this._queue.push(action); 
        
        // It's often not optimal to sort here
        this._queue.sort(function(a, b) {
            var timeDiff = a.actionTime - b.actionTime;
            if (timeDiff != 0)
                return timeDiff;
            return a.id - b.id;
        });
    }
}
