
DROP TABLE IF EXISTS debug_log;
DROP TABLE IF EXISTS checksum_log;
DROP TABLE IF EXISTS chat_message;
DROP TABLE IF EXISTS client_ping;
DROP TABLE IF EXISTS client;
DROP TABLE IF EXISTS command;
DROP TABLE IF EXISTS session;

CREATE TABLE session (
    id SERIAL NOT NULL,
    name VARCHAR(128) NOT NULL,
    joinable BOOLEAN NOT NULL DEFAULT TRUE,
    max_clients INT NOT NULL,
    is_paused BOOLEAN NOT NULL DEFAULT TRUE,
    create_time_abs TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    metadata TEXT NULL,
    -- TODO: can this be simplified even more?
    end_of_time INT NOT NULL DEFAULT 0,
    end_of_time_limit INT NOT NULL DEFAULT 0,
    master_client INT NULL, -- not foreign keyed, so be careful!
    master_key CHAR(32) NOT NULL,
    PRIMARY KEY (id),
    CHECK (LENGTH(TRIM(name)) > 0),
    CHECK (end_of_time >= 0),
    CHECK (end_of_time_limit >= 0),
    CHECK (end_of_time <= end_of_time_limit)
);
COMMENT ON COLUMN session.max_clients IS
'The maximum number of clients. Checked while joining even is joinable is true.';
COMMENT ON COLUMN session.metadata IS
'Any data. Settable by the master client and readable by all.';
COMMENT ON COLUMN session.end_of_time IS
'The game time (ms) at which new commands are scheduled. '
'Clients may never simulate beyond this point in time.';
COMMENT ON COLUMN session.end_of_time_limit IS
'The maximum value for endOfTime.';
COMMENT ON COLUMN session.master_client IS
'The client that may initiate end of time increments and other special commands.';
COMMENT ON COLUMN session.master_key IS
'A secret key sent to the master client, used to access master client operations.';

CREATE TABLE command (
    id SERIAL NOT NULL,
    session INT NOT NULL,
    serial INT NOT NULL,
    time_offset INT NOT NULL,
    message VARCHAR(4096) NOT NULL,
    PRIMARY KEY (id),
    UNIQUE (session, serial),
    FOREIGN KEY (session) REFERENCES session (id) ON DELETE CASCADE,
    CHECK (serial > 0)
);
COMMENT ON COLUMN command.serial IS
'The serial number of the command relative to this session. Starts from 1.';
COMMENT ON COLUMN command.time_offset IS
'Game time in milliseconds.';

CREATE TABLE client (
    id SERIAL NOT NULL,
    -- TODO: add a random secret key to prevent some trivial hacks
    session INT NULL,
    player_num INT NULL,
    name VARCHAR(128) NOT NULL,
    last_command_synced INT NULL DEFAULT NULL,
    last_communication_time TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (id),
    UNIQUE (session, player_num),
    FOREIGN KEY (session) REFERENCES session (id) ON DELETE SET NULL,
    FOREIGN KEY (session, last_command_synced) REFERENCES command (session, serial) ON DELETE SET NULL,
    CHECK (LENGTH(TRIM(name)) > 0)
    -- CHECK ((session IS NULL) = (player_num IS NULL)) -- Problem: FOREIGN KEY to session has SET NULL
);
COMMENT ON COLUMN client.player_num IS
'The player sequence number in the session. Starts counting from 1.';

CREATE TABLE client_ping (
    id SERIAL NOT NULL,
    client INT NOT NULL,
    ping INT NOT NULL,
    PRIMARY KEY (id),
    FOREIGN KEY (client) REFERENCES client (id) ON DELETE CASCADE
);
CREATE INDEX client_ping_client_id ON client_ping (client, id);

CREATE TABLE chat_message (
    id SERIAL NOT NULL,
    session INT NOT NULL,
    sender INT NOT NULL,
    msg TEXT NOT NULL,
    PRIMARY KEY (id),
    FOREIGN KEY (session) REFERENCES session (id) ON DELETE CASCADE,
    FOREIGN KEY (sender) REFERENCES client (id) ON DELETE CASCADE
);

CREATE TABLE checksum_log (
    id SERIAL NOT NULL,
    session INT NOT NULL,
    sender INT NOT NULL,
    game_time INT NOT NULL,
    checksum VARCHAR(256) NOT NULL,
    PRIMARY KEY (id),
    UNIQUE (session, sender, game_time),
    FOREIGN KEY (session) REFERENCES session (id) ON DELETE CASCADE,
    FOREIGN KEY (sender) REFERENCES client (id) ON DELETE CASCADE
);
COMMENT ON TABLE checksum_log IS
'Checksums of the game state are stored here and compared to detect out of sync conditions.';

CREATE TABLE debug_log (
    id SERIAL NOT NULL,
    game_time INT NOT NULL,
    log_time TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    sender INT NOT NULL,
    msg TEXT NOT NULL,
    PRIMARY KEY (id)
);
COMMENT ON TABLE debug_log IS
'A log of debug messages. '
'Note that this is not foreign-keyed to anything and must be cleared manually if used.';

CREATE INDEX debug_log_game_time ON debug_log (game_time);
