// ===============================================================================================
//  NormalMap_Fresnel.shader
//
//  Normal-mapping shader algorithm with fresnel lighting.
//
//  Created on:      14-May-2006
//  Original author: Sami Karjalainen
//
//  Copyright (c) 2006 Sami Karjalainen.
// ===============================================================================================

description
{
    Normal mapping shader with fresnel lighting.
}

shared
{
    struct Vertex2Fragment
    {
        float4 projectedPt : POSITION; // Clip-space position of the vertex.
        float4 uv : TEXCOORD0;         // Texture coordinates for the vertex.
        float4 eyeVec : TEXCOORD1;     // Eye vector in TANGENT-SPACE. w component = projected Z
        float4 light0 : TEXCOORD2;     // Light-0 definition in TANGENT-SPACE.
        float4 light1 : TEXCOORD3;     // Light-1 definition in TANGENT-SPACE.
        float4 light2 : TEXCOORD4;     // Light-2 definition in TANGENT-SPACE.
    };
}

vertex
{
    uniform float4x4 m2pMatrix; /*{ bind: ModelViewProjection }*/
    uniform float3 camera;      /*{ bind: Camera.Relative     }*/
    uniform float4 light0;      /*{ bind: Light0.Relative     }*/
    uniform float4 light1;      /*{ bind: Light1.Relative     }*/
    uniform float4 light2;      /*{ bind: Light2.Relative     }*/

    struct App2Vertex
    {
        float4 pt : POSITION;       // Vertex's "position" in model-space.
        float3 normal : NORMAL;     // Vertex's normal vector in model-space.
        float4 uv : TEXCOORD0;      // Texture coordinates for the vertex.
        float4 tangent : TEXCOORD1; // Vertex's tangent vector in model-space.
    };

    Vertex2Fragment main( App2Vertex input )
    {
        // ### Create object-space to tangent-space transform.

        float3 bitangent = cross(input.normal, input.tangent.xyz) * input.tangent.w;    


        // ### Compute the EYE VECTOR in TANGENT SPACE.

        float3 objEyeVec = camera - input.pt.xyz;
        float3 tanEyeVec;
        tanEyeVec.x = dot(input.tangent.xyz, objEyeVec);
        tanEyeVec.y = dot(bitangent, objEyeVec);
        tanEyeVec.z = dot(input.normal, objEyeVec);


        // ### Compute LIGHT-0 VECTOR (& distance if point light) in TANGENT SPACE.

        float light0wCeil = ceil(light0.w);
        float3 objLight0 = light0.xyz - (input.pt.xyz * light0wCeil);
        float4 tanLight0;
        tanLight0.x = dot(input.tangent.xyz, objLight0);
        tanLight0.y = dot(bitangent, objLight0);
        tanLight0.z = dot(input.normal, objLight0);
        tanLight0.w = sqrt(dot(objLight0, objLight0)) * light0wCeil; // Light distance


        // ### Compute LIGHT-1 VECTOR (& distance if point light) in TANGENT SPACE.

        float light1wCeil = ceil(light1.w);
        float3 objLight1 = light1.xyz - (input.pt.xyz * light1wCeil);
        float4 tanLight1;
        tanLight1.x = dot(input.tangent.xyz, objLight1);
        tanLight1.y = dot(bitangent, objLight1);
        tanLight1.z = dot(input.normal, objLight1);
        tanLight1.w = sqrt(dot(objLight1, objLight1)) * light1wCeil; // Light distance

        // ### Compute LIGHT-1 VECTOR (& distance if point light) in TANGENT SPACE.

        float light2wCeil = ceil(light2.w);
        float3 objLight2 = light2.xyz - (input.pt.xyz * light2wCeil);
        float4 tanLight2;
        tanLight2.x = dot(input.tangent.xyz, objLight2);
        tanLight2.y = dot(bitangent, objLight1);
        tanLight2.z = dot(input.normal, objLight2);
        tanLight2.w = sqrt(dot(objLight2, objLight2)) * light2wCeil; // Light distance


        // ### Finalise the ouput data, including calculating the projected position of the vertex

        Vertex2Fragment output;

		output.projectedPt = mul(m2pMatrix, input.pt);
        output.eyeVec.xyz = tanEyeVec;
        output.eyeVec.w = output.projectedPt.z;
        output.light0 = tanLight0;
        output.light1 = tanLight1;
        output.light2 = tanLight2;
        output.uv = input.uv;

        return output;
    }
}

pixel
{
    uniform sampler2D diffuseMap  : TEXUNIT0; /*{ bind: DiffuseMap  }*/
    uniform sampler2D normalMap   : TEXUNIT1; /*{ bind: NormalMap   }*/

    uniform float4 lightingCoeff; /*{ bind: Ambient, Diffuse, Specular, GlossPower }*/
    uniform float4 l0_AttCoeff;   /*{ bind: Light0.Attenuation, null, null, null   }*/
    uniform float4 l1_AttCoeff;   /*{ bind: Light1.Attenuation, null, null, null   }*/
    uniform float4 l2_AttCoeff;   /*{ bind: Light2.Attenuation, null, null, null   }*/
    uniform float4 miscData;      /*{ bind: Opacity, BumpRatio, null, null         }*/
	uniform float4 fogParams;     /*{ bind: Opacity, BumpRatio, null, null         }*/
	
    float4 main( Vertex2Fragment input ) : COLOR    
    {
        float const ambientFactor = lightingCoeff.x;
        float const diffuseFactor = lightingCoeff.y;
        float const specularFactor = lightingCoeff.z;
        float const glossiness = lightingCoeff.w;
        float const opaqueness = miscData.x;
        float const bumpRatio = miscData.y;
		float3 const fogColour = float3( 0.07, 0.07, 0.08 );
	
        float3 const E = normalize(input.eyeVec);
        float const kR = (glossiness + 2.0) * 0.1591549;

        float3 const L0 = normalize(input.light0.xyz);
        float3 const L1 = normalize(input.light1.xyz);
        float3 const L2 = normalize(input.light2.xyz);
        float3 const H0 = normalize(L0 + E);
        float3 const H1 = normalize(L1 + E);
        float3 const H2 = normalize(L2 + E);
        float const EL0 = dot(L0, E);
        float const EL1 = dot(L1, E);
        float const EL2 = dot(L2, E);

        float termC0 = l0_AttCoeff.x;
        float termC1 = l0_AttCoeff.y * input.light0.w;
        float termC2 = l0_AttCoeff.z * input.light0.w * input.light0.w;
        float L0att = saturate(1.0 / (termC0 + termC1 + termC2));

        termC0 = l1_AttCoeff.x;
        termC1 = l1_AttCoeff.y * input.light1.w;
        termC2 = l1_AttCoeff.z * input.light1.w * input.light1.w;
        float L1att = saturate(1.0 / (termC0 + termC1 + termC2));

        termC0 = l2_AttCoeff.x;
        termC1 = l2_AttCoeff.y * input.light2.w;
        termC2 = l2_AttCoeff.z * input.light2.w * input.light2.w;
        float L2att = saturate(1.0 / (termC0 + termC1 + termC2));

        float3 tc = input.uv.xyz;
        float4 colour = tex2D(diffuseMap, tc.xy);
        float3 N = normalize(tex2D(normalMap, tc.xy).xyz * 2.0 - 1.0);


        // ### Calculate DIFFUSE & SPECULAR REFLECTION co-efficients.

        float NE = dot(N, E);

        float kF = pow(1.0 - NE, 5);
        float F = (kF * (glossiness - 1.0)) + 1.0;

        float NH0 = dot(N, H0);
        float NL0 = dot(N, L0);  
        float minCosPheta0 = max(NE, NL0);
        float R0 = kR * pow(saturate(2.0 * NE * NL0 - EL0), glossiness) / minCosPheta0;
        float I0 = L0att * ((diffuseFactor * NL0) + (specularFactor * R0 * F)) * ceil(NL0);

        float NH1 = dot(N, H1);
        float NL1 = dot(N, L1);  
        float minCosPheta1 = max(NE, NL1);
        float R1 = kR * pow(saturate(2.0 * NE * NL1 - EL1), glossiness) / minCosPheta1;
        float I1 = L1att * ((diffuseFactor * NL1) + (specularFactor * R1 * F)) * ceil(NL1);

        float NH2 = dot(N, H2);
        float NL2 = dot(N, L2);  
        float minCosPheta2 = max(NE, NL2);
        float R2 = kR * pow(saturate(2.0 * NE * NL2 - EL2), glossiness) / minCosPheta2;
        float I2 = L2att * ((diffuseFactor * NL2) + (specularFactor * R2 * F)) * ceil(NL2);

        float4 finalColour;
        finalColour.rgb = lerp(colour.rgb * saturate(I0 + I1 + I2), colour.rgb, ambientFactor);
        finalColour.a = colour.a * opaqueness;
	
		float depthFactor = clamp( input.eyeVec.w / 10000.0, 0, 1 );
		depthFactor = depthFactor * depthFactor * depthFactor;
		
		finalColour.rgb = lerp(finalColour.rgb, fogColour.rgb, depthFactor );
	
        return finalColour;
    }
}