// ===============================================================================================
//  DistanceMap_BRDF_8.shader
//
//  Distance mapping shader with bi-directional reflection distribution function lighting.
//
//  Created on:      24-May-2006
//  Original author: Jacob Stanley & Sami Karljalainen
//
//  Copyright (c) 2006 Jacob Stanley & Sami Karljalainen.
// ===============================================================================================

description
{
    Distance mapping shader with bi-directional reflection distribution function lighting.
}

shared
{
    struct Vertex2Fragment
    {
        float4 projectedPt : POSITION;     // Clip-space position of the vertex.
        float4 uv : TEXCOORD0;             // Texture coordinates for the vertex.
        float3 eyeVec : TEXCOORD1;         // Eye vector in TANGENT-SPACE.
        float4 light0 : TEXCOORD2;         // Light-0 definition in TANGENT-SPACE.
        float4 light1 : TEXCOORD3;         // Light-1 definition in TANGENT-SPACE.
    };
}

vertex
{
    uniform float4x4 m2pMatrix; /*{ bind: ModelViewProjection }*/
    uniform float3 camera;      /*{ bind: Camera.Relative     }*/
    uniform float4 light0;      /*{ bind: Light0.Relative     }*/
    uniform float4 light1;      /*{ bind: Light1.Relative     }*/

    struct App2Vertex
    {
        float4 pt : POSITION;       // Vertex's "position" in model-space.
        float3 normal : NORMAL;     // Vertex's normal vector in model-space.
        float4 uv : TEXCOORD0;      // Texture coordinates for the vertex.
        float4 tangent : TEXCOORD1; // Vertex's tangent vector in model-space.
    };

    Vertex2Fragment main( App2Vertex input )
    {
        // ### Create object-space to tangent-space transform.

        float3 bitangent = cross(input.normal, input.tangent.xyz) * input.tangent.w;    


        // ### Compute the EYE VECTOR in TANGENT SPACE.

        float3 objEyeVec = camera - input.pt.xyz;
        float3 tanEyeVec;
        tanEyeVec.x = dot(input.tangent.xyz, objEyeVec);
        tanEyeVec.y = dot(bitangent, objEyeVec);
        tanEyeVec.z = dot(input.normal, objEyeVec);

        // ### Compute LIGHT-0 VECTOR (& distance if point light) in TANGENT SPACE.

        float light0wCeil = ceil(light0.w);
        float3 objLight0 = light0.xyz - (input.pt.xyz * light0wCeil);
        float4 tanLight0;
        tanLight0.x = dot(input.tangent.xyz, objLight0);
        tanLight0.y = dot(bitangent, objLight0);
        tanLight0.z = dot(input.normal, objLight0);
        tanLight0.w = sqrt(dot(objLight0, objLight0)) * light0wCeil; // Light distance


        // ### Compute LIGHT-1 VECTOR (& distance if point light) in TANGENT SPACE.

        float light1wCeil = ceil(light1.w);
        float3 objLight1 = light1.xyz - (input.pt.xyz * light1wCeil);
        float4 tanLight1;
        tanLight1.x = dot(input.tangent.xyz, objLight1);
        tanLight1.y = dot(bitangent, objLight1);
        tanLight1.z = dot(input.normal, objLight1);
        tanLight1.w = sqrt(dot(objLight1, objLight1)) * light1wCeil; // Light distance

        
        // ### Finalise the ouput data, including calculating the projected position of the vertex

        Vertex2Fragment output;

        output.projectedPt = mul(m2pMatrix, input.pt);
        output.eyeVec = tanEyeVec;
        output.light0 = tanLight0;
        output.light1 = tanLight1;
        output.uv = input.uv;
        output.uv.z = 1.0;   // Required by the DistanceMap Fragment Shaders.
        output.uv.w = output.projectedPt.z / 100.0;

        return output;
    }
}

pixel
{
    uniform sampler2D diffuseMap  : TEXUNIT0; /*{ bind: DiffuseMap  }*/
    uniform sampler2D normalMap   : TEXUNIT1; /*{ bind: NormalMap   }*/
    uniform sampler3D distanceMap : TEXUNIT2; /*{ bind: DistanceMap }*/

    uniform float4 lightingCoeff; /*{ bind: Ambient, Diffuse, Specular, GlossPower }*/
    uniform float4 l0_AttCoeff;   /*{ bind: Light0.Attenuation, null, null, null   }*/
    uniform float4 l1_AttCoeff;   /*{ bind: Light1.Attenuation, null, null, null   }*/
    uniform float4 brdf;          /*{ bind: nu, nv, K0, null                       }*/
    uniform float4 miscData;      /*{ bind: Opacity, BumpRatio, null, null         }*/

    // Forward Declarations
    float spectral(float dp, float C);
    float A(float dp, float p);
    float G(float dp, float r);
    float Z(float dp, float r);
    float directional(float t, float v, float vdash, float w, float r, float p);

    float4 main( Vertex2Fragment input ) : COLOR    
    {
        float const ambientFactor = lightingCoeff.x;
        float const diffuseFactor = lightingCoeff.y;
        float const specularFactor = lightingCoeff.z;
        float const glossiness = lightingCoeff.w;
        float const opaqueness = miscData.x;
        float const bumpRatio = miscData.y;

        float3 const E = normalize(input.eyeVec);
        float const kR = (glossiness + 2.0) * 0.1591549;
        float3 const L0 = normalize(input.light0.xyz);
        float3 const L1 = normalize(input.light1.xyz);
        float3 const H0 = normalize(L0 + E);
        float3 const H1 = normalize(L1 + E);
        float3 const SurfN = float3( 0.0, 1.0, 0.0 );

        float L0att = 0.0;
        float L1att = 0.0;
        
        if ( dot( SurfN, L0 ) >= 0.0 )
        {
            float termC0 = l0_AttCoeff.x;
            float termC1 = l0_AttCoeff.y * input.light0.w;
            float termC2 = l0_AttCoeff.z * input.light0.w * input.light0.w;
            L0att = saturate(1.0 / (termC0 + termC1 + termC2));
        }

        if ( dot( SurfN, L1 ) >= 0.0 )
        {
            float termC0 = l1_AttCoeff.x;
            float termC1 = l1_AttCoeff.y * input.light1.w;
            float termC2 = l1_AttCoeff.z * input.light1.w * input.light1.w;
            L1att = saturate(1.0 / (termC0 + termC1 + termC2));
        }

        float3 tc = input.uv.xyz;
        float2 const dsdx = ddx(tc.xy);
        float2 const dsdy = ddy(tc.xy);


        // ### Compute BIASED TEXTURE COORDINATES using the DISTANCE MAPPING technique

//      float3 dmRay = -E;
//      dmRay.xy *= bumpRatio;

//      for (int i = 0; i < 16; i++)
//      {
//          float dist = tex3D(distanceMap, tc).x;
//          tc += dist * dmRay;
//      }   

        float4 colour = tex2D(diffuseMap, tc.xy, dsdx, dsdy);
        float3 N = normalize(tex2D(normalMap, tc.xy, dsdx, dsdy).xyz * 2.0 - 1.0);

        float NH0 = saturate( dot(N, H0) );
        float EH0 = clamp( dot(E, H0), 0.05, 1.0 );
        float NL0 = saturate( dot(N, L0) );

        float NH1 = saturate( dot(N, H1) );
        float EH1 = clamp( dot(E, H1), 0.05, 1.0 );
        float NL1 = saturate( dot(N, L1) );

        float EN = clamp( dot(E, N), 0.05, 1.0 );

        float3 tangRotAxis = normalize(float3(N.y, -N.x, 0.0));
        float tangCosA = N.z;

        float oneMinusTangCosA = 1.0 - tangCosA;

        float3 rotB;
        rotB.x = tangRotAxis.x * tangRotAxis.y * oneMinusTangCosA;
        rotB.y = (tangRotAxis.y * tangRotAxis.y * oneMinusTangCosA) + tangCosA;
        rotB.z = tangRotAxis.x * sqrt(1.0 - tangCosA * tangCosA);
        rotB = normalize(rotB);


        float THproj0 = saturate( dot(rotB, normalize(cross(H0, N))) );
        float THproj1 = saturate( dot(rotB, normalize(cross(H1, N))) );

        float i0 = (spectral(EH0, 0.8) * directional(NH0, EN, NL0, THproj0, 0.05, 2.0)) * NL0;
        float i1 = (spectral(EH1, 0.8) * directional(NH1, EN, NL1, THproj1, 0.05, 2.0)) * NL1;

        float i = clamp( i0 + i1, 0.0, 2.0 );

        float4 finalColour;
        finalColour.rgb = lerp( i * colour.rgb, colour.rgb, ambientFactor );
        finalColour.a = colour.a * opaqueness;

        return finalColour;
    }

    float spectral(float dp, float C)
    {
        return C + ((1.0 - C) * pow(1.0 - dp, 5));
    }


    float A(float dp, float p)
    {
        float dp2 = dp * dp;
        float p2 = p * p;
        float divisor = p2 + dp2 - ( p2 * dp2 );
        return sqrt( p / divisor );
    }


    float G(float dp, float r)
    {
        float divisor = r + ( dp * ( 1 - r ) );
        return dp / divisor;
    }


    float Z(float dp, float r)
    {
        float dp2 = dp * dp;

        return r * pow(1 - dp2 * (1 - r), -2);
    }


    float directional(float t, float v, float vdash, float w, float r, float p)
    {
        float G0 = G(v, r);
        float G1 = G(vdash, r);
        float Aw = A(w, p);
        float Zt = Z(t, r);

        float divisor = max( v * vdash, 0.00001 );


        float k0 = 0.079577471 / divisor;
        float G0G1 = G0 * G1;

        return ( k0 * (G0G1 * (Aw * Zt - 1.0) + 1.0) );
    }
}