#define NUM_TAPS 12						// number of taps the shader will use

uniform vec2 pixelSizeScene;			// pixel size of full resolution image
uniform vec2 pixelSizeBlur;				// pixel size of downsampled and blurred image

uniform sampler2D Sharp;				// full resolution image
uniform sampler2D Depth;				// full resolution image with depth values
uniform sampler2D Blurred;					// downsampled and blurred image

const vec2 poisson[NUM_TAPS] = {		// containts poisson-distributed positions on the
	vec2( 0.00,  0.00),					// unit circle
	vec2( 0.07, -0.45),
	vec2(-0.15, -0.33),
	vec2( 0.35, -0.32),
	vec2(-0.39, -0.26),
	vec2( 0.10, -0.23),
	vec2( 0.36, -0.12),
	vec2(-0.31, -0.01),
	vec2(-0.38,  0.22),
	vec2( 0.36,  0.23),
	vec2(-0.13,  0.29),
	vec2( 0.14,  0.41)};

const vec2 maxCoC = vec2(5.0, 10.0);	// maximum circle of confusion (CoC) radius
										// and diameter in pixels

const float radiusScale = 0.4;			// scale factor for minimum CoC size on low res. image

void main()
{
	// Get depth of center tap and convert it into blur radius in pixels
	float centerDepth = texture2D(Depth, gl_TexCoord[0].st).r;
	float discRadiusScene = abs(centerDepth * maxCoC.y - maxCoC.x);
	float discRadiusBlur = discRadiusScene * radiusScale; // radius on low res. image

	vec4 sum = vec4(0.0);

	float s = 0.0;
	for (int i = 0; i < NUM_TAPS; ++i)
	{
		// compute texture coordinates
		vec2 coordScene = gl_TexCoord[0].st + (pixelSizeScene * poisson[i] * discRadiusScene);
		vec2 coordBlur = gl_TexCoord[0].st + (pixelSizeBlur * poisson[i] * discRadiusBlur);
	
		// fetch taps and depth
		vec4 tapScene = texture2D(Sharp, coordScene);
		float tapDepth = texture2D(Depth, coordScene).r;
		vec4 tapBlur = texture2D(Blurred, coordBlur);
		
		// mix low and high res. taps based on tap blurriness
		float blurAmount = abs(tapDepth * 2.0 - 1.0); // put blurriness into [0, 1]
		vec4 tap = mix(tapScene, tapBlur, blurAmount);
	
		// "smart" blur ignores taps that are closer than the center tap and in focus
		float factor = (tapDepth >= centerDepth) ? 1.0 : abs(tapDepth * 2.0 - 1.0);
	
		// accumulate
		sum.rgb += tap.rgb * factor;
		sum.a += factor;
		s += sum.r + sum.g + sum.b;
	}

	gl_FragColor = vec4(sum.rgb / sum.a, s);
}