
#include <wendy/Wendy.h>

using namespace moira;
using namespace wendy;

#include "Water.h"

void Water::prepare(render::Queue& queue) const
{
  canvas->begin();
  canvas->clearColorBuffer();
  canvas->clearDepthBuffer();

  GL::Renderer* renderer = GL::Renderer::get();

  queue.getCamera().begin();

  Matrix4 scaling;
  scaling.setScaling(Vector3(1.f, -1.f, 1.f));

  renderer->pushTransform(scaling);

  GL::Pass::setCullingInversion(true);

  const render::OperationList& operations = queue.getOperations();

  for (unsigned int i = 0;  i < operations.size();  i++)
  {
    const render::Operation& operation = *operations[i];

    renderer->pushTransform(operation.transform);

    for (unsigned int j = 0;  j < operation.technique->getPassCount();  j++)
    {
      const GL::Pass& pass = operation.technique->getPass(j);
      if (!pass.getName().empty())
	continue;

      pass.apply();

      if (operation.indexBuffer)
        operation.indexBuffer->render(*(operation.vertexBuffer),
	                              operation.renderMode,
	                              operation.start,
				      operation.count);
      else
        operation.vertexBuffer->render(operation.renderMode,
	                               operation.start,
				       operation.count);
    }

    renderer->popTransform();
  }

  GL::Pass::setCullingInversion(false);

  queue.getCamera().end();

  canvas->end();
}

void Water::enqueue(render::Queue& queue, const Transform3& transform) const
{
  GL::Renderer* renderer = GL::Renderer::get();

  render::Technique* technique = style->getActiveTechnique();
  if (!technique)
    return;

  const unsigned int segments = 20;

  GL::VertexRange range;

  if (!renderer->allocateVertices(range, segments + 1, GL::Vertex3fv::format))
    return;

  GL::Vertex3fv* vertices = (GL::Vertex3fv*) range.lock();
  if (!vertices)
    return;

  vertices[0].position = Vector3::ZERO;

  for (unsigned int i = 0;  i < segments;  i++)
  {
    const float angle = M_PI * 2.f * (float) i / (float) (segments - 1);

    vertices[i + 1].position.set(bounds.radius * cosf(angle),
                                 0.f,
				 bounds.radius * sinf(angle));
  }

  range.unlock();

  render::Operation& operation = queue.createOperation();
  operation.renderMode = GL_TRIANGLE_FAN;
  operation.start = range.getStart();
  operation.count = range.getCount();
  operation.technique = technique;
  operation.vertexBuffer = range.getVertexBuffer();

  const render::Camera& camera = queue.getCamera();

  Transform3 reverse;
  reverse.position = camera.getTransform().position;
  reverse.position.y = transform.position.y - reverse.position.y;

  operation.transform = reverse;
}

const Sphere& Water::getBounds(void) const
{
  return bounds;
}

Water* Water::createInstance(float radius)
{
  Ptr<Water> water = new Water();
  if (!water->init(radius))
    return NULL;

  return water.detachObject();
}

Water::Water(void)
{
}

bool Water::init(float radius)
{
  GL::Context* context = GL::Context::get();

  canvas = GL::TextureCanvas::createInstance(context->getWidth(),
                                             context->getHeight(),
					     "water");
  if (!canvas)
    return false;

  style = render::Style::readInstance("water");
  if (!style)
    return false;

  bounds.set(Vector3::ZERO, radius);
  return true;
}

void WaterNode::setWater(Water* newWater)
{
  water = newWater;
  
  if (water)
    setLocalBounds(water->getBounds());
  else
    setLocalBounds(Sphere(Vector3::ZERO, 0.f));
}

void WaterNode::enqueue(render::Queue& queue, render::QueuePhase phase) const
{
  if (phase == render::COLLECT_GEOMETRY)
    water->enqueue(queue, getWorldTransform());

  render::SceneNode::enqueue(queue, phase);
}

