
#include <wendy/Wendy.h>

using namespace moira;
using namespace wendy;

#include "Glow.h"
#include "Laser.h"
#include "Hyper.h"

void HyperCube::enqueue(render::Queue& queue, const Transform3& transform) const
{
  Segment3 segment;
  
  for (unsigned int i = 0;  i < 4;  i++)
  {
    segment.start = vertices[i].projected;
    segment.end = vertices[(i + 1) % 4].projected;
    enqueueLaserBeam(queue, segment, transform);

    segment.start = vertices[i + 4].projected;
    segment.end = vertices[(i + 1) % 4 + 4].projected;
    enqueueLaserBeam(queue, segment, transform);

    segment.start = vertices[i + 4].projected;
    segment.end = vertices[i].projected;
    enqueueLaserBeam(queue, segment, transform);

    segment.start = vertices[i + 8].projected;
    segment.end = vertices[(i + 1) % 4 + 8].projected;
    enqueueLaserBeam(queue, segment, transform);

    segment.start = vertices[i + 12].projected;
    segment.end = vertices[(i + 1) % 4 + 12].projected;
    enqueueLaserBeam(queue, segment, transform);

    segment.start = vertices[i + 12].projected;
    segment.end = vertices[i + 8].projected;
    enqueueLaserBeam(queue, segment, transform);

    segment.start = vertices[i].projected;
    segment.end = vertices[i + 8].projected;
    enqueueLaserBeam(queue, segment, transform);

    segment.start = vertices[i + 4].projected;
    segment.end = vertices[i + 12].projected;
    enqueueLaserBeam(queue, segment, transform);
  }

  for (unsigned int i = 0;  i < 16;  i++)
  {
    Transform3 inner;
    inner.position = vertices[i].projected;

    Transform3 final = inner;
    final.concatenate(transform);

    ball->enqueue(queue, final);
  }
}

const Sphere& HyperCube::getBounds(void) const
{
  return bounds;
}

void HyperCube::setTimeElapsed(Time newTime)
{
  const float angle = newTime;

  Matrix4 transform;
  transform.setIdentity();

  // produce 4D rotation transform
  {
    Matrix4 temp;

    temp.setIdentity();
    temp.y.y = cosf(angle);
    temp.w.w = cosf(angle);
    temp.y.w = -sinf(angle);
    temp.w.y = sinf(angle);

    transform.concatenate(temp);

    temp.setIdentity();
    temp.z.z = cosf(angle);
    temp.w.w = cosf(angle);
    temp.z.w = -sinf(angle);
    temp.w.z = sinf(angle);

    transform.concatenate(temp);

    temp.setIdentity();
    temp.x.x = cosf(angle);
    temp.w.w = cosf(angle);
    temp.x.w = -sinf(angle);
    temp.w.x = sinf(angle);

    transform.concatenate(temp);
  }

  float maxRadius = 0.f;

  for (unsigned int i = 0;  i < 16;  i++)
  {
    Vector4 result = vertices[i].position;
    transform.transformVector(result);

    if (result.w)
    {
      vertices[i].projected.x = 5.f * result.x / (result.w + 3.f);
      vertices[i].projected.y = 5.f * result.y / (result.w + 3.f);
      vertices[i].projected.z = 5.f * result.z / (result.w + 3.f);

      maxRadius = std::max(maxRadius, vertices[i].projected.length());
    }
    else
      vertices[i].projected = Vector3::ZERO;
  }

  bounds.radius = maxRadius;
}

HyperCube* HyperCube::createInstance(void)
{
  Ptr<HyperCube> cube = new HyperCube();
  if (!cube->init())
    return NULL;

  return cube.detachObject();
}

HyperCube::HyperCube(void)
{
}

bool HyperCube::init(void)
{
  style = render::Style::readInstance("purple");
  if (!style)
    return false;

  ball = render::Mesh::readInstance("ball");
  if (!ball)
    return false;

  vertices[0x0].position.set(-1.f,  1.f,  1.f,  1.f);
  vertices[0x1].position.set(-1.f, -1.f,  1.f,  1.f);
  vertices[0x2].position.set( 1.f, -1.f,  1.f,  1.f);
  vertices[0x3].position.set( 1.f,  1.f,  1.f,  1.f);
  vertices[0x4].position.set(-1.f,  1.f, -1.f,  1.f);
  vertices[0x5].position.set(-1.f, -1.f, -1.f,  1.f);
  vertices[0x6].position.set( 1.f, -1.f, -1.f,  1.f);
  vertices[0x7].position.set( 1.f,  1.f, -1.f,  1.f);
  vertices[0x8].position.set(-1.f,  1.f,  1.f, -1.f);
  vertices[0x9].position.set(-1.f, -1.f,  1.f, -1.f);
  vertices[0xA].position.set( 1.f, -1.f,  1.f, -1.f);
  vertices[0xB].position.set( 1.f,  1.f,  1.f, -1.f);
  vertices[0xC].position.set(-1.f,  1.f, -1.f, -1.f);
  vertices[0xD].position.set(-1.f, -1.f, -1.f, -1.f);
  vertices[0xE].position.set( 1.f, -1.f, -1.f, -1.f);
  vertices[0xF].position.set( 1.f,  1.f, -1.f, -1.f);

  return true;
}

HyperNode::HyperNode(void):
  elapsed(0.0)
{
}

void HyperNode::setCube(HyperCube* newCube)
{
  cube = newCube;
}

void HyperNode::update(Time deltaTime)
{
  SceneNode::update(deltaTime);

  elapsed += deltaTime;
  cube->setTimeElapsed(elapsed);

  setLocalBounds(cube->getBounds());
}

void HyperNode::restart(void)
{
  SceneNode::restart();

  elapsed = 0.0;
}

void HyperNode::enqueue(render::Queue& queue, render::QueuePhase phase) const
{
  SceneNode::enqueue(queue, phase);

  if (phase != render::COLLECT_GEOMETRY)
    return;

  cube->enqueue(queue, getWorldTransform());
}

HyperEffect::HyperEffect(demo::EffectType& type, const String& name):
  demo::Effect(type, name),
  sequence(*this, "Sequence")
{
  sequence.addSymbol("Initial", INITIAL);
  sequence.addSymbol("Presents", PRESENTS);
  sequence.addSymbol("Orbit", ORIBIT);
}

bool HyperEffect::init(void)
{
  render::Mesh* dome = render::Mesh::readInstance("dome");
  if (!dome)
    return false;

  domeNode = new render::MeshNode();
  domeNode->setMesh(dome);
  scene.addNode(*domeNode);

  presentsTexture = GL::Texture::readInstance("presents");
  if (!presentsTexture)
    return false;

  presentsPass.setDefaultColor(ColorRGBA::BLACK);
  presentsPass.setBlendFactors(GL_ONE, GL_ONE);
  presentsPass.setDepthTesting(false);
  presentsPass.setDepthWriting(false);

  GL::TextureLayer& presentsLayer = presentsPass.createTextureLayer();
  presentsLayer.setTexture(presentsTexture);
  presentsLayer.setCombineMode(GL_MODULATE);
  presentsLayer.setAddressMode(GL_CLAMP_TO_EDGE);

  glow = Glow::createInstance(256);
  if (!glow)
    return false;

  Ref<HyperCube> cube = HyperCube::createInstance();
  if (!cube)
    return false;

  RandomVolume volume(Vector3(-10.f, -10.f, -10.f),
                      Vector3(10.f, 10.f, 10.f));

  {
    const Vector3 start(0.f, 0.f, 10.f);
    const Vector3 end(15.f, 0.f, 30.f);

    slideTrack.points.push_back(start);
    slideTrack.points.push_back(end);
    slideTrack.points.push_back(end);
    slideTrack.points.push_back(start);

    BezierPoint3 point;
    point.position = start;
    point.direction = start - end;

    curveTrack.points.push_back(point);
  }

  for (unsigned int i = 0;  i < 4;  i++)
  {
    BezierPoint3 point;

    Vector3 position;

    do
    {
      position = volume.generate();
    }
    while (position.length() < 5.f);

    point.position = position;
    position.normalize();

    Vector3 direction;

    do
    {
      direction = (volume.generate() - volume.generate()).normalize();
    }
    while (fabsf(direction.normalize().dotProduct(position.normalize())) > 0.4f);

    point.direction = direction;

    curveTrack.points.push_back(point);
  }

  hyperNode = new HyperNode();
  hyperNode->setCube(cube);
  scene.addNode(*hyperNode);

  camera.setFOV(60.f);

  cameraNode = new render::CameraNode();
  cameraNode->setCameraName(camera.getName());
  scene.addNode(*cameraNode);

  return true;
}

void HyperEffect::update(Time deltaTime)
{
  presentsPass.setDefaultColor(ColorRGBA::BLACK);

  const Time elapsed = getTimeElapsed();

  const Time start = sequence.getSequenceStart(elapsed);
  const Time duration = sequence.getSequenceDuration(elapsed);

  float progress = (float) (elapsed - start) / (float) duration;

  switch (sequence.getValue(elapsed))
  {
    case INITIAL:
    {
      cameraNode->getLocalTransform().position = slideTrack(0.f);
      cameraNode->getLocalTransform().rotation.setDefaults();
      break;
    }

    case PRESENTS:
    {
      if (progress > 0.25f && progress < 0.75f)
      {
	const float value = sinf(M_PI * (progress - 0.25f) * 2.f);
	presentsPass.setDefaultColor(ColorRGBA(value, value, value, 1.f));
      }

      cameraNode->getLocalTransform().position = slideTrack(progress);
      cameraNode->getLocalTransform().rotation.setDefaults();
      break;
    }

    case ORIBIT:
    {
      Vector3 position = curveTrack(progress);

      cameraNode->getLocalTransform().position = position;
      cameraNode->getLocalTransform().rotation.setVectorRotation(position.normalize());
      break;
    }
  }

  domeNode->getLocalTransform().position = cameraNode->getLocalTransform().position;

  scene.setTimeElapsed(elapsed);
}

void HyperEffect::prepare(void) const
{
  render::Queue queue(camera);
  scene.enqueue(queue);
  glow->prepare(queue);
}

void HyperEffect::render(void) const
{
  render::Queue queue(camera);
  scene.enqueue(queue);
  queue.render();

  GL::Renderer* renderer = GL::Renderer::get();

  renderer->begin2D();

  presentsPass.apply();

  const float size = 0.55f;

  render::Sprite2 sprite;
  sprite.position.set(0.6f, 0.5f);
  sprite.size.set(size, size * presentsTexture->getHeight() /
                        (float) presentsTexture->getWidth());
  sprite.render();

  renderer->end();

  glow->render();
}

