///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////

#include <moira/Config.h>
#include <moira/Portability.h>
#include <moira/Core.h>
#include <moira/Vector.h>
#include <moira/Color.h>
#include <moira/Stream.h>
#include <moira/Resource.h>
#include <moira/Image.h>
#include <moira/ImageGenerators.h>

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

BlobImageGenerator::BlobImageGenerator(void):
  umbraRadius(0.25f),
  penumbraRadius(0.5f),
  umbraColor(ColorRGBA::WHITE),
  exponent(1.f)
{
}

float BlobImageGenerator::getPenumbraRadius(void) const
{
  return penumbraRadius;
}

void BlobImageGenerator::setPenumbraRadius(float newRadius)
{
  penumbraRadius = newRadius;
}

float BlobImageGenerator::getUmbraRadius(void) const
{
  return umbraRadius;
}

void BlobImageGenerator::setUmbraRadius(float newRadius)
{
  umbraRadius = newRadius;
}

const ColorRGBA& BlobImageGenerator::getUmbraColor(void) const
{
  return umbraColor;
}

void BlobImageGenerator::setUmbraColor(const ColorRGBA& newColor)
{
  umbraColor = newColor;
}

float BlobImageGenerator::getExponent(void) const
{
  return exponent;
}

void BlobImageGenerator::setExponent(float newExponent)
{
  exponent = newExponent;
}

Image* BlobImageGenerator::generate(const ImageFormat& format,
                                    unsigned int width,
				    unsigned int height)
{
  Ptr<Image> result = new Image(format, width, height);

  Byte* pixels = (Byte*) result->getPixels();

  const float factor = 1.f / (penumbraRadius - umbraRadius); 

  for (unsigned int y = 0;  y < height;  y++)
  {
    for (unsigned int x = 0;  x < width;  x++)
    {
      const Vector2 position(x / (float) (width - 1) - 0.5f,
	                     y / (float) (height - 1) - 0.5f);
      const float distance = position.length();

      if (distance > penumbraRadius)
	format.encode(pixels, getDefaultColor());
      else if (distance > umbraRadius)
      {
	const float t = powf((distance - umbraRadius) * factor, exponent);
	format.encode(pixels, getDefaultColor() * t + umbraColor * (1.f - t));
      }
      else
	format.encode(pixels, umbraColor);

      pixels += format.getSize();
    }
  }

  return result.detachObject();
}

///////////////////////////////////////////////////////////////////////

CheckerImageGenerator::CheckerImageGenerator(void):
  checkerSize(1),
  checkerColor(ColorRGBA::WHITE)
{
}

unsigned int CheckerImageGenerator::getCheckerSize(void) const
{
  return checkerSize;
}

void CheckerImageGenerator::setCheckerSize(unsigned int newSize)
{
  checkerSize = newSize;
}

const ColorRGBA& CheckerImageGenerator::getCheckerColor(void) const
{
  return checkerColor;
}

void CheckerImageGenerator::setCheckerColor(const ColorRGBA& newColor)
{
  checkerColor = newColor;
}

Image* CheckerImageGenerator::generate(const ImageFormat& format,
                                       unsigned int width,
				       unsigned int height)
{
  Ptr<Image> result = new Image(format, width, height);

  Byte* pixels = (Byte*) result->getPixels();

  for (unsigned int y = 0;  y < height;  y++)
  {
    for (unsigned int x = 0;  x < width;  x++)
    {
      if ((x / checkerSize) & 1)
      {
        if ((y / checkerSize) & 1)
	  format.encode(pixels, getDefaultColor());
	else
	  format.encode(pixels, checkerColor);
      }
      else
        if ((y / checkerSize) & 1)
	  format.encode(pixels, checkerColor);
	else
	  format.encode(pixels, getDefaultColor());

      pixels += format.getSize();
    }
  }

  return result.detachObject();
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
