///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented; you
//    must not claim that you wrote the original software. If you use
//    this software in a product, an acknowledgment in the product
//    documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such, and
//    must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source
//    distribution.
//
///////////////////////////////////////////////////////////////////////

#include <moira/Config.h>
#include <moira/Portability.h>
#include <moira/Core.h>
#include <moira/Color.h>
#include <moira/Stream.h>
#include <moira/Resource.h>
#include <moira/Image.h>

///////////////////////////////////////////////////////////////////////

namespace moira
{

///////////////////////////////////////////////////////////////////////

namespace
{

bool convertPixelsGeneric(Block& data,
			  const ImageFormat& targetFormat,
			  const ImageFormat& sourceFormat,
			  unsigned int width,
		          unsigned int height)
{
  Block scratch(width * height * targetFormat.getSize());

  ColorRGBA pixel;

  Byte* target = scratch;
  size_t targetSize = targetFormat.getSize();

  const Byte* source = data;
  size_t sourceSize = sourceFormat.getSize();

  for (size_t count = width * height;  count--; )
  {
    sourceFormat.decode(pixel, source);
    targetFormat.encode(target, pixel);

    source += sourceSize;
    target += targetSize;
  }
  
  data.attach(scratch.detach(), width * height * targetSize);
  return true;
}

bool convertPixels(Block& data,
		   const ImageFormat& targetFormat,
                   const ImageFormat& sourceFormat,
		   unsigned int width,
		   unsigned int height)
{
  switch (sourceFormat)
  {
    case ImageFormat::ALPHA8:
    {
      switch (targetFormat)
      {
	case ImageFormat::GREY8:
	  return true;
      }

      break;
    }

    case ImageFormat::GREY8:
    {
      switch (targetFormat)
      {
	case ImageFormat::ALPHA8:
	  return true;
      }

      break;
    }

    case ImageFormat::ALPHA32F:
    {
      switch (targetFormat)
      {
	case ImageFormat::GREY32F:
	  return true;
      }

      break;
    }

    case ImageFormat::GREY32F:
    {
      switch (targetFormat)
      {
	case ImageFormat::ALPHA32F:
	  return true;
      }

      break;
    }
  }

  return convertPixelsGeneric(data, targetFormat, sourceFormat, width, height);
}

void samplePixelsNearest1D(Byte* target,
			   unsigned int targetWidth,
			   const Byte* source,
			   unsigned int sourceWidth,
			   const ImageFormat& format)
{
  const size_t pixelSize = format.getSize();

  const float sx = (sourceWidth - 1) / (float) (targetWidth - 1);

  Byte* targetPixel = target;

  for (unsigned int x = 0;  x < targetWidth;  x++)
  {
    const Byte* sourcePixel = source + (size_t) (x * sx) * pixelSize;
    for (unsigned int i = 0;  i < pixelSize;  i++)
      targetPixel[i] = sourcePixel[i];

    targetPixel += pixelSize;
  }
}

void samplePixelsLinear1D(Byte* target,
			  unsigned int targetWidth,
			  const Byte* source,
			  unsigned int sourceWidth,
			  const ImageFormat& format)
{
  const size_t pixelSize = format.getSize();

  const float sx = (sourceWidth - 1) / (float) (targetWidth - 1);

  Byte* targetPixel = target;

  ColorRGBA targetValue;
  ColorRGBA corners[2];

  for (unsigned int x = 0;  x < targetWidth;  x++)
  {
    // TODO: Optimize for speed.

    const float u = x * sx;

    const size_t minU = (size_t) floorf(u);
    const size_t maxU = (size_t) ceilf(u);

    const float fracU = u - (float) minU;

    format.decode(corners[0], source + minU * pixelSize);
    format.decode(corners[1], source + maxU * pixelSize);

    targetValue = corners[0] * (1.f - fracU) + corners[1] * fracU;

    format.encode(targetPixel, targetValue);

    targetPixel += pixelSize;
  }
}

void samplePixelsNearest2D(Byte* target,
			   unsigned int targetWidth,
			   unsigned int targetHeight,
			   const Byte* source,
			   unsigned int sourceWidth,
			   unsigned int sourceHeight,
			   const ImageFormat& format)
{
  const size_t pixelSize = format.getSize();

  const float sx = (sourceWidth - 1) / (float) (targetWidth - 1);
  const float sy = (sourceHeight - 1) / (float) (targetHeight - 1);

  Byte* targetPixel = target;

  for (unsigned int y = 0;  y < targetHeight;  y++)
  {
    for (unsigned int x = 0;  x < targetWidth;  x++)
    {
      const Byte* sourcePixel = source + ((size_t) (x * sx) + 
				          (size_t) (y * sy) * sourceWidth) * pixelSize;
      for (unsigned int i = 0;  i < pixelSize;  i++)
	targetPixel[i] = sourcePixel[i];

      targetPixel += pixelSize;
    }
  }
}

void samplePixelsLinear2D(Byte* target,
			  unsigned int targetWidth,
			  unsigned int targetHeight,
			  const Byte* source,
			  unsigned int sourceWidth,
			  unsigned int sourceHeight,
			  const ImageFormat& format)
{
  const size_t pixelSize = format.getSize();

  const float sx = (sourceWidth - 1) / (float) (targetWidth - 1);
  const float sy = (sourceHeight - 1) / (float) (targetHeight - 1);

  Byte* targetPixel = target;

  ColorRGBA targetValue;
  ColorRGBA corners[4];

  for (unsigned int y = 0;  y < targetHeight;  y++)
  {
    for (unsigned int x = 0;  x < targetWidth;  x++)
    {
      // TODO: Optimize for speed.

      const float u = x * sx;
      const float v = y * sy;

      const size_t minU = (size_t) floorf(u);
      const size_t minV = (size_t) floorf(v);
      const size_t maxU = (size_t) ceilf(u);
      const size_t maxV = (size_t) ceilf(v);

      const float fracU = u - (float) minU;
      const float fracV = v - (float) minV;

      format.decode(corners[0], source + (minU + minV * sourceWidth) * pixelSize);
      format.decode(corners[1], source + (maxU + minV * sourceWidth) * pixelSize);
      format.decode(corners[2], source + (minU + maxV * sourceWidth) * pixelSize);
      format.decode(corners[3], source + (maxU + maxV * sourceWidth) * pixelSize);

      targetValue = corners[0] * (1.f - fracU) * (1.f - fracV) +
	            corners[1] * (fracU) * (1.f - fracV) +
	            corners[2] * (1.f - fracU) * (fracV) +
	            corners[3] * (fracU) * (fracV);

      format.encode(targetPixel, targetValue);

      targetPixel += pixelSize;
    }
  }
}

}

///////////////////////////////////////////////////////////////////////

ImageArea::ImageArea(void)
{
}

ImageArea::ImageArea(unsigned int initX,
                     unsigned int initY,
		     unsigned int initWidth,
		     unsigned int initHeight):
  x(initX),
  y(initY),
  width(initWidth),
  height(initHeight)
{
}

void ImageArea::set(unsigned int newX,
                    unsigned int newY,
		    unsigned int newWidth,
		    unsigned int newHeight)
{
  x = newX;
  y = newY;
  width = newWidth;
  height = newHeight;
}

///////////////////////////////////////////////////////////////////////

ImageFormat::ImageFormat(Type initType):
  type(initType)
{
}

void ImageFormat::decode(ColorRGBA& result, const void* source) const
{
  switch (type)
  {
    case ALPHA8:
    {
      Byte value = *((const Byte*) source);
      result.r = 0.f;
      result.g = 0.f;
      result.b = 0.f;
      result.a = value / 255.f;
      break;
    }
    
    case GREY8:
    {
      Byte value = *((const Byte*) source);
      result.r = value / 255.f;
      result.g = value / 255.f;
      result.b = value / 255.f;
      result.a = 1.f;
      break;
    }
    
    case GREYALPHA88:
    {
      const Byte* values = (const Byte*) source;
      result.r = values[0] / 255.f;
      result.g = values[0] / 255.f;
      result.b = values[0] / 255.f;
      result.a = values[1] / 255.f;
      break;
    }

    case RGB888:
    {
      const Byte* values = (const Byte*) source;
      result.r = values[0] / 255.f;
      result.g = values[1] / 255.f;
      result.b = values[2] / 255.f;
      result.a = 1.f;
      break;
    }

    case RGBX8888:
    {
      const Byte* values = (const Byte*) source;
      result.r = values[0] / 255.f;
      result.g = values[1] / 255.f;
      result.b = values[2] / 255.f;
      result.a = 1.f;
      break;
    }

    case RGBA8888:
    {
      const Byte* values = (const Byte*) source;
      result.r = values[0] / 255.f;
      result.g = values[1] / 255.f;
      result.b = values[2] / 255.f;
      result.a = values[3] / 255.f;
      break;
    }

    case ALPHA32F:
    {
      const float value = *((const float*) source);
      result.r = 0.f;
      result.g = 0.f;
      result.b = 0.f;
      result.a = value;
    }

    case GREY32F:
    {
      const float value = *((const float*) source);
      result.r = value;
      result.g = value;
      result.b = value;
      result.a = 1.f;
      break;
    }
    
    case GREYALPHA32F:
    {
      const float* values = (const float*) source;
      result.r = values[0];
      result.g = values[0];
      result.b = values[0];
      result.a = values[1];
      break;
    }

    case RGB32F:
    {
      const float* values = (const float*) source;
      result.r = values[0];
      result.g = values[1];
      result.b = values[2];
      result.a = 1.f;
      break;
    }

    case RGBX32F:
    {
      const float* values = (const float*) source;
      result.r = values[0];
      result.g = values[1];
      result.b = values[2];
      result.a = 1.f;
      break;
    }

    case RGBA32F:
    {
      const float* values = (const float*) source;
      result.r = values[0];
      result.g = values[1];
      result.b = values[2];
      result.a = values[3];
      break;
    }

    default:
      result.set(0.f, 0.f, 0.f, 1.f);
      break;
  }
}

void ImageFormat::encode(void* result, const ColorRGBA& source) const
{
  switch (type)
  {
    case ALPHA8:
    {
      Byte* values = (Byte*) result;
      *values = (Byte) (source.a * 255.f);
      break;
    }
    
    case GREY8:
    {
      Byte* values = (Byte*) result;
      *values = (Byte) ((source.r * 0.30f +
                         source.g * 0.59f +
			 source.b * 0.11f) * 255.f);
      break;
    }
    
    case GREYALPHA88:
    {
      Byte* values = (Byte*) result;
      values[0] = (Byte) (source.a * 255.f);
      values[1] = (Byte) ((source.r * 0.30f +
                           source.g * 0.59f +
			   source.b * 0.11f) * 255.f);
      break;
    }

    case RGB888:
    {
      Byte* values = (Byte*) result;
      values[0] = (Byte) (source.r * 255.f);
      values[1] = (Byte) (source.g * 255.f);
      values[2] = (Byte) (source.b * 255.f);
      break;
    }

    case RGBX8888:
    {
      Byte* values = (Byte*) result;
      values[0] = (Byte) (source.r * 255.f);
      values[1] = (Byte) (source.g * 255.f);
      values[2] = (Byte) (source.b * 255.f);
      break;
    }

    case RGBA8888:
    {
      Byte* values = (Byte*) result;
      values[0] = (Byte) (source.r * 255.f);
      values[1] = (Byte) (source.g * 255.f);
      values[2] = (Byte) (source.b * 255.f);
      values[3] = (Byte) (source.a * 255.f);
      break;
    }

    case ALPHA32F:
    {
      float* values = (float*) result;
      *values = source.a;
      break;
    }
    
    case GREY32F:
    {
      float* values = (float*) result;
      *values = source.r * 0.30f + source.g * 0.59f + source.b * 0.11f;
      break;
    }
    
    case GREYALPHA32F:
    {
      float* values = (float*) result;
      values[0] = source.a;
      values[1] = source.r * 0.30f + source.g * 0.59f + source.b * 0.11f;
      break;
    }

    case RGB32F:
    {
      float* values = (float*) result;
      values[0] = source.r;
      values[1] = source.g;
      values[2] = source.b;
      break;
    }

    case RGBX32F:
    {
      float* values = (float*) result;
      values[0] = source.r;
      values[1] = source.g;
      values[2] = source.b;
      break;
    }

    case RGBA32F:
    {
      float* values = (float*) result;
      values[0] = source.r;
      values[1] = source.g;
      values[2] = source.b;
      values[3] = source.a;
      break;
    }
  }
}

bool ImageFormat::hasAlpha(void) const
{
  switch (type)
  {
    case ALPHA8:
    case RGBX8888:
      return true;
  }

  return false;
}

size_t ImageFormat::getSize(void) const
{
  switch (type)
  {
    case ALPHA8:
    case GREY8:
      return 1;
    
    case GREYALPHA88:
      return 2;

    case RGB888:
      return 3;

    case RGBX8888:
    case RGBA8888:
    case ALPHA32F:
    case GREY32F:
      return 4;

    case GREYALPHA32F:
      return 8;

    case RGB32F:
      return 12;

    case RGBX32F:
    case RGBA32F:
      return 16;

    default:
      return 0;
  }
}

unsigned int ImageFormat::getChannelCount(void) const
{
  switch (type)
  {
    case ALPHA8:
    case GREY8:
    case ALPHA32F:
    case GREY32F:
      return 1;

    case GREYALPHA88:
    case GREYALPHA32F:
      return 2;

    case RGB888:
    case RGBX8888:
    case RGB32F:
    case RGBX32F:
      return 3;

    case RGBA8888:
    case RGBA32F:
      return 4;

    default:
      return 0;
  }
}

unsigned int ImageFormat::getChannelBitCount(Channel channel) const
{
  switch (channel)
  {
    case GREY:
    {
      switch (type)
      {
        case GREY8:
        case GREYALPHA88:
          return 8;
	case GREY32F:
	case GREYALPHA32F:
	  return 32;
      }
      break;
    }
    
    case RED:
    {
      switch (type)
      {
        case RGB888:
        case RGBX8888:
        case RGBA8888:
          return 8;
	case RGB32F:
	case RGBX32F:
	case RGBA32F:
	  return 32;
      }
      break;
    }

    case GREEN:
    {
      switch (type)
      {
        case RGB888:
        case RGBX8888:
        case RGBA8888:
          return 8;
	case RGB32F:
	case RGBX32F:
	case RGBA32F:
	  return 32;
      }
      break;
    }

    case BLUE:
    {
      switch (type)
      {
        case RGB888:
        case RGBX8888:
        case RGBA8888:
	  return 8;
	case RGB32F:
	case RGBX32F:
	case RGBA32F:
	  return 32;
      }
      break;
    }

    case ALPHA:
    {
      switch (type)
      {
        case ALPHA8:
        case GREYALPHA88:
        case RGBA8888:
          return 8;
	case ALPHA32F:
	case GREYALPHA32F:
	case RGBA32F:
	  return 32;
      }
      break;
    }
  }

  return 0;
}

ImageFormat::operator Type (void) const
{
  return type;
}

///////////////////////////////////////////////////////////////////////

Image::Image(const ImageFormat& initFormat,
             unsigned int initWidth,
	     unsigned int initHeight,
	     const void* initData,
	     size_t pitch,
	     const String& name):
  Resource<Image>(name),
  format(initFormat),
  width(initWidth),
  height(initHeight)
{
  if (format == ImageFormat::INVALID)
    throw Exception("Invalid image format");

  if (width == 0 || height == 0)
    throw Exception("Invalid image size");

  if ((height > 1) && (width == 1))
  {
    width = height;
    height = 1;
  }

  if (initData)
  {
    if (pitch)
    {
      size_t size = format.getSize();
      data.resize(width * height * size);

      Byte* target = data;
      const Byte* source = (const Byte*) initData;

      for (unsigned int y = 0;  y < height;  y++)
      {
	std::memcpy(target, source, width * size);
	source += pitch;
	target += width * size;
      }
    }
    else
      data.copyFrom((const Byte*) initData, width * height * format.getSize());
  }
  else
  {
    const size_t size = width * height * format.getSize();
    data.reserve(size);
    std::memset(data, 0, size);
  }
}

Image::Image(const Image& source):
  Resource<Image>(source),
  locks(0)
{
  operator = (source);
}

void Image::fill(const ColorRGBA& color)
{
  Byte* pixels = data;
  size_t pixelSize = format.getSize();

  for (size_t count = width * height;  count--; )
  {
    format.encode(pixels, color);
    pixels += pixelSize;
  }
}

bool Image::resize(unsigned int targetWidth,
                   unsigned int targetHeight,
		   Method method)
{
  if (targetWidth == 0 || targetHeight == 0)
    throw Exception("Invalid image target size");

  if (targetWidth == width && targetHeight == height)
    return true;

  const size_t pixelSize = format.getSize();

  Block scratch(targetWidth * targetHeight * pixelSize);

  switch (method)
  {
    case SAMPLE_NEAREST:
    {
      if (getDimensionCount() == 1)
	samplePixelsNearest1D(scratch, targetWidth, data, width, format);
      else
	samplePixelsNearest2D(scratch, targetWidth, targetHeight, data, width, height, format);

      break;
    }

    case SAMPLE_LINEAR:
    {
      if (getDimensionCount() == 1)
	samplePixelsLinear1D(scratch, targetWidth, data, width, format);
      else
	samplePixelsLinear2D(scratch, targetWidth, targetHeight, data, width, height, format);

      break;
    }

    default:
      Log::writeError("Invalid image resampling filter");
      return false;
  }

  width = targetWidth;
  height = targetHeight;

  data.attach(scratch.detach(), width * height * pixelSize);
  return true;
}

bool Image::convert(const ImageFormat& targetFormat)
{
  if (targetFormat == ImageFormat::INVALID)
  {
    Log::writeError("Invalid image target format");
    return false;
  }

  if (targetFormat == format)
    return true;

  if (!convertPixels(data, targetFormat, format, width, height))
    return false;
  
  format = targetFormat;
  return true;
}

bool Image::crop(const ImageArea& area)
{
  if ((area.x >= width) || (area.y >= height))
  {
    Log::writeError("Invalid image area dimensions");
    return false;
  }

  ImageArea targetArea = area;

  if (area.x + area.width > width)
    targetArea.width = width - area.x;
  if (area.y + area.height > height)
    targetArea.height = height - area.y;
  
  const size_t pixelSize = format.getSize();

  Block scratch(targetArea.width * targetArea.height * pixelSize);

  for (unsigned int y = 0;  y < targetArea.height;  y++)
  {
    scratch.copyFrom(data + ((y + targetArea.y) * width + targetArea.x) * pixelSize,
                     targetArea.width * pixelSize,
		     y * targetArea.width * pixelSize);
  }

  width = targetArea.width;
  height = targetArea.height;

  data.attach(scratch.detach(), width * height * pixelSize);
  return true;
}

void Image::flipHorizontal(void)
{
  size_t pixelSize = format.getSize();

  Block scratch(width * height * pixelSize);

  for (unsigned int y = 0;  y < height;  y++)
  {
    scratch.copyFrom(data + y * width * pixelSize,
                     width * pixelSize,
		     (height - y - 1) * width * pixelSize);
  }

  data.attach(scratch.detach(), width * height * pixelSize);
}

void Image::flipVertical(void)
{
  size_t pixelSize = format.getSize();

  Block scratch(width * height * pixelSize);

  for (unsigned int y = 0;  y < height;  y++)
  {
    const Byte* source = data + y * width * pixelSize;
    Byte* target = scratch + ((y + 1) * width - 1) * pixelSize;

    for (unsigned int x = 0;  x < width;  x++)
    {
      for (unsigned int i = 0;  i < pixelSize;  i++)
	target[i] = source[i];

      source += pixelSize;
      target -= pixelSize;
    }
  }

  data.attach(scratch.detach(), width * height * pixelSize);
}

Image& Image::operator = (const Image& source)
{
  width = source.width;
  height = source.height;
  format = source.format;

  locks = 0;

  data = source.data;
  return *this;
}

unsigned int Image::getWidth(void) const
{
  return width;
}

unsigned int Image::getHeight(void) const
{
  return height;
}

void* Image::getPixels(void)
{
  return data;
}

const void* Image::getPixels(void) const
{
  return data;
}

void* Image::getPixel(unsigned int x, unsigned int y)
{
  if (x >= width || y >= height)
    return NULL;

  return data + (y * width + x) * format.getSize();
}

const void* Image::getPixel(unsigned int x, unsigned int y) const
{
  if (x >= width || y >= height)
    return NULL;

  return data + (y * width + x) * format.getSize();
}

const ImageFormat& Image::getFormat(void) const
{
  return format;
}

unsigned int Image::getDimensionCount(void) const
{
  if (height > 1)
    return 2;
  else
    return 1;
}

Image* Image::getArea(const ImageArea& area)
{
  if (area.x >= width || area.y >= height)
    return NULL;

  ImageArea targetArea = area;

  if (area.x + area.width > width)
    targetArea.width = width - area.x;
  if (area.y + area.height > height)
    targetArea.height = height - area.y;
  
  const size_t pixelSize = format.getSize();

  Image* result = new Image(format, targetArea.width, targetArea.height);

  for (unsigned int y = 0;  y < targetArea.height;  y++)
  {
    const Byte* source = data + ((y + targetArea.y) * width + targetArea.x) * pixelSize;
    Byte* target = result->data + y * result->width * pixelSize;
    memcpy(target, source, result->width * pixelSize);
  }

  return result;
}

bool Image::copyPixels(void* target,
                       size_t targetPitch,
                       const ImageFormat& targetFormat,
	               const ImageArea& targetArea,
                       const void* source,
                       size_t sourcePitch,
                       const ImageFormat& sourceFormat,
                       const ImageArea& sourceArea)
{
  if (sourceArea.width == targetArea.width && sourceArea.height == targetArea.height)
  {
    // TODO: The code.
  }
  else
  {
    // TODO: The code.
  }

  return false;
}

///////////////////////////////////////////////////////////////////////

ImageGenerator::ImageGenerator(void):
  defaultColor(ColorRGBA::BLACK)
{
}

ImageGenerator::~ImageGenerator(void)
{
}

const ColorRGBA& ImageGenerator::getDefaultColor(void) const
{
  return defaultColor;
}

void ImageGenerator::setDefaultColor(const ColorRGBA& newColor)
{
  defaultColor = newColor;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
