///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////

#include <moira/Config.h>
#include <moira/Portability.h>
#include <moira/Core.h>

#if MOIRA_HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if MOIRA_HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if MOIRA_HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if MOIRA_HAVE_DIRECT_H
#include <direct.h>
#endif

#if MOIRA_HAVE_UNISTD_H
#include <unistd.h>
#endif

#if MOIRA_HAVE_IO_H
#include <io.h>
#endif

#if MOIRA_HAVE_STDARG_H
#include <stdarg.h>
#endif

#if MOIRA_HAVE_DIRENT_H
  #include <dirent.h>
  #define NAMLEN(dirent) strlen((dirent)->d_name)
#else
  #define dirent direct
  #define NAMLEN(dirent) (dirent)->d_namlen
#if MOIRA_HAVE_SYS_NDIR_H
  #include <sys/ndir.h>
  #endif
    #if MOIRA_HAVE_SYS_DIR_H
      #include <sys/dir.h>
    #endif
  #if MOIRA_HAVE_NDIR_H
    #include <ndir.h>
  #endif
#endif

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

Exception::Exception(const char* initMessage):
  message(initMessage)
{
}

const char* Exception::what(void) const throw()
{
  return message;
}

///////////////////////////////////////////////////////////////////////

Block::Block(size_t initSize):
  size(0),
  grain(0),
  data(NULL)
{
  resize(initSize);
}

Block::Block(const Byte* source, size_t sourceSize):
  size(0),
  grain(0),
  data(NULL)
{
  reserve(sourceSize);
  std::memcpy(data, source, sourceSize);
}

Block::Block(const Block& source):
  size(0),
  grain(0),
  data(NULL)
{
  operator = (source);
}

Block::~Block(void)
{
  destroy();
}

void Block::copyTo(Byte* target, size_t targetSize, size_t offset) const
{
  std::memcpy(target, data + offset, targetSize);
}

void Block::copyFrom(const Byte* source, size_t sourceSize, size_t offset)
{
  reserve(sourceSize + offset);
  std::memcpy(data + offset, source, sourceSize);
}

void Block::resize(size_t newSize)
{
  if (newSize == 0)
    destroy();
  else
  {
    if (grain != 0)
      newSize = grain * ((newSize + grain - 1) / grain);
    
    if (size == 0)
      data = (Byte*) malloc(newSize);
    else
    {
      Byte* newData = (Byte*) realloc(data, newSize);
      if (!newData)
        throw Exception("Failed to reallocate memory block");
      
      data = newData;
    }
    
    size = newSize;
  }
}

void Block::reserve(size_t minSize)
{
  if (minSize > size)
    resize(minSize);
}

void Block::attach(Byte* newData, size_t newSize)
{
  destroy();

  data = newData;
  size = newSize;
}

Byte* Block::detach(void)
{
  Byte* detached = data;

  data = NULL;
  size = 0;

  return detached;
}

void Block::destroy(void)
{
  if (size > 0)
  {
    free(data);
    data = NULL;
    size = 0;
  }
}

Block::operator Byte* (void)
{
  return data;
}

Block::operator const Byte* (void) const
{
  return data;
}

Block& Block::operator = (const Block& source)
{
  if (source.size == 0)
    destroy();
  else
  {
    resize(source.size);
    copyFrom(source.data, source.size);
  }
  
  return *this;
}

size_t Block::getSize(void) const
{
  return size;
}

size_t Block::getGrain(void) const
{
  return grain;
}

void Block::setGrain(size_t newGrain)
{
  grain = newGrain;
}

Byte* Block::getData(void)
{
  return data;
}

const Byte* Block::getData(void) const
{
  return data;
}

///////////////////////////////////////////////////////////////////////

Variant::Variant(void)
{
}

Variant::Variant(const String& initValue):
  value(initValue)
{
}

float Variant::asFloat(void) const
{
  return convertToFloat(value);
}

void Variant::setFloatValue(float newValue)
{
  convertToString(value, newValue);
}

int Variant::asInteger(void) const
{
  return convertToInteger(value);
}

void Variant::setIntegerValue(int newValue)
{
  convertToString(value, newValue);
}

bool Variant::asBoolean(void) const
{
  return convertToBoolean(value);
}

void Variant::setBooleanValue(bool newValue)
{
  convertToString(value, newValue);
}

const String& Variant::asString(void) const
{
  return value;
}

void Variant::setStringValue(const String& newValue)
{
  value = newValue;
}

float Variant::convertToFloat(const String& value)
{
  return strtof(value.c_str(), NULL);
}

int Variant::convertToInteger(const String& value)
{
  return strtol(value.c_str(), NULL, 0);
}

bool Variant::convertToBoolean(const String& value)
{
  if (value == "true")
    return true;

  return convertToInteger(value) ? true : false;
}

void Variant::convertToString(String& result, float value)
{
  char buffer[32];
  snprintf(buffer, sizeof(buffer), "%f", value);
  buffer[sizeof(buffer) - 1] = '\0';
  result = buffer;
}

void Variant::convertToString(String& result, int value)
{
  char buffer[32];
  snprintf(buffer, sizeof(buffer), "%i", value);
  buffer[sizeof(buffer) - 1] = '\0';
  result = buffer;
}

void Variant::convertToString(String& result, bool value)
{
  if (value)
    result = "true";
  else
    result = "false";
}

///////////////////////////////////////////////////////////////////////

Path::Path(const String& initName)
{
  operator = (initName);
}

Path::Path(const char* format, ...)
{
  va_list vl;
  char* message;

  va_start(vl, format);
  vasprintf(&message, format, vl);
  va_end(vl);

  operator = (message);

  free(message);
}

bool Path::createDirectory(void) const
{
#if _WIN32
  return _mkdir(name.c_str()) == 0;
#else
  return mkdir(name.c_str(), 0777) == 0;
#endif
}

bool Path::destroyDirectory(void) const
{
#if _WIN32
  return _rmdir(name.c_str()) == 0;
#else
  return rmdir(name.c_str()) == 0;
#endif
}

bool Path::exists(void) const
{
  return access(name.c_str(), F_OK) == 0;
}

const String& Path::asString(void) const
{
  return name;
}

Path Path::operator + (const String& child) const
{
  return Path(name + '/' + child);
}

Path& Path::operator += (const String& child)
{
  String newName = name;
  newName.append(1, '/');
  newName.append(child);

  return operator = (newName);
}

bool Path::operator == (const Path& other) const
{
  // TODO: Implement sane code.

  return name == other.name;
}

bool Path::operator != (const Path& other) const
{
  // TODO: Implement sane code.

  return name != other.name;
}

Path& Path::operator = (const String& newName)
{
  name = newName;

  if (!name.empty())
  {
    // Remove extraneous trailing slashes

    while (String::size_type end = name.length() - 1)
    {
      if (name[end] != '/')
	break;

      name.erase(end);
    }

    // TODO: Compact repeated slashes.
  }

  return *this;
}

bool Path::isReadable(void) const
{
  return access(name.c_str(), R_OK) == 0;
}

bool Path::isWritable(void) const
{
  return access(name.c_str(), W_OK) == 0;
}

bool Path::isFile(void) const
{
  struct stat sb;

  if (stat(name.c_str(), &sb) != 0)
    return false;

  return S_ISREG(sb.st_mode) ? true : false;
}

bool Path::isDirectory(void) const
{
  struct stat sb;

  if (stat(name.c_str(), &sb) != 0)
    return false;

  return S_ISDIR(sb.st_mode) ? true : false;
}

Path Path::getParent(void) const
{
  // TODO: Fix this.

  String::size_type offset = name.find_last_of('/');
  if (offset == String::npos)
    return Path(".");

  return Path(name.substr(0, offset + 1));
}

bool Path::getChildren(List& children) const
{
#if _WIN32
  // TODO: Implement.
  return true;
#else
  DIR* stream;
  dirent* entry;

  stream = opendir(name.c_str());
  if (!stream)
    return false;

  while (entry = readdir(stream))
    children.push_back(operator + (entry->d_name));

  closedir(stream);
  return true;
#endif
}

String Path::getSuffix(void) const
{
  String last;

  String::size_type start = name.rfind('/');
  if (start == String::npos)
    last = name;
  else
    last = name.substr(start, String::npos);

  String::size_type offset = last.find_last_of('.');
  if (offset == String::npos)
    return "";

  return last.substr(offset + 1, String::npos);
}

///////////////////////////////////////////////////////////////////////
  
Log::~Log(void)
{
}

void Log::writeError(const char* format, ...)
{
  va_list vl;
  char* message;

  va_start(vl, format);
  vasprintf(&message, format, vl);
  va_end(vl);
  
  if (Log* log = Log::get())
    log->write(ERROR, "%s", message);
  else
    fprintf(stderr, "%s\n", message);

  free(message);
}

void Log::writeWarning(const char* format, ...)
{
  va_list vl;
  char* message;

  va_start(vl, format);
  vasprintf(&message, format, vl);
  va_end(vl);
  
  if (Log* log = Log::get())
    log->write(WARNING, "%s", message);
  else
    fprintf(stderr, "%s\n", message);

  free(message);
}

void Log::writeInformation(const char* format, ...)
{
  va_list vl;
  char* message;

  va_start(vl, format);
  vasprintf(&message, format, vl);
  va_end(vl);
  
  if (Log* log = Log::get())
    log->write(INFORMATION, "%s", message);
  else
    fprintf(stderr, "%s\n", message);

  free(message);
}

///////////////////////////////////////////////////////////////////////

FileLog::~FileLog(void)
{
  if (fd != -1)
    close(fd);
}

void FileLog::write(EntryType type, const char* format, ...)
{
  va_list vl;
  char* message;

  va_start(vl, format);
  vasprintf(&message, format, vl);
  va_end(vl);
  
  // TODO: Print log message type.
  ::write(fd, message, (unsigned int) strlen(message));

  free(message);
}

bool FileLog::create(const Path& path)
{
  Ptr<FileLog> log = new FileLog();
  if (!log->init(path))
    return false;

  set(log.detachObject());
  return true;
}

FileLog::FileLog(void):
  fd(-1)
{
}

bool FileLog::init(const Path& path)
{
  fd = open(path.asString().c_str(), O_APPEND | O_CREAT | O_WRONLY);
  if (fd == -1)
    return false;

  return true;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
