///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2006 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////

#include <moira/Config.h>
#include <moira/Portability.h>
#include <moira/Core.h>
#include <moira/Signal.h>
#include <moira/Attribute.h>

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

Attribute::Attribute(Decorated& initDecorated, const String& initName):
  decorated(initDecorated),
  name(initName)
{
  decorated.createAttributeSignal.emit(*this);
}

Attribute::~Attribute(void)
{
  decorated.destroyAttributeSignal.emit(*this);
}

Attribute& Attribute::createMember(const String& name)
{
  return operator [] (name);
}

void Attribute::destroyMember(const String& name)
{
  for (List::iterator i = members.begin();  i != members.end();  i++)
  {
    if ((*i).name == name)
    {
      members.erase(i);
      break;
    }
  }
}

void Attribute::destroyMembers(void)
{
  members.clear();
}

Attribute& Attribute::operator [] (const String& name)
{
  if (Attribute* member = getMember(name))
    return *member;

  members.push_front(Attribute(decorated, name));
  return members.front();
}

const Attribute& Attribute::operator [] (const String& name) const
{
  if (const Attribute* member = getMember(name))
    return *member;

  throw Exception("Requested attribute does not exist");
}

float Attribute::asFloat(void) const
{
  return strtof(value.c_str(), NULL);
}

void Attribute::setFloatValue(float newValue)
{
  char buffer[32];
  snprintf(buffer, sizeof(buffer), "%f", newValue);
  buffer[sizeof(buffer) - 1] = '\0';
  value = buffer;

  decorated.changeValueSignal.emit(*this);
}

int Attribute::asInteger(void) const
{
  return strtol(value.c_str(), NULL, 0);
}

void Attribute::setIntegerValue(int newValue)
{
  char buffer[32];
  snprintf(buffer, sizeof(buffer), "%i", newValue);
  buffer[sizeof(buffer) - 1] = '\0';
  value = buffer;

  decorated.changeValueSignal.emit(*this);
}

bool Attribute::asBoolean(void) const
{
  if (value == "true")
    return true;

  return asInteger() ? true : false;
}

void Attribute::setBooleanValue(bool newValue)
{
  if (newValue)
    value = "true";
  else
    value = "false";

  decorated.changeValueSignal.emit(*this);
}

const String& Attribute::asString(void) const
{
  return value;
}

void Attribute::setStringValue(const String& newValue)
{
  value = newValue;

  decorated.changeValueSignal.emit(*this);
}

const String& Attribute::getName(void) const
{
  return name;
}

Attribute* Attribute::getMember(const String& name)
{
  for (List::iterator i = members.begin();  i != members.end();  i++)
  {
    if ((*i).name == name)
      return &(*i);
  }

  return NULL;
}

const Attribute* Attribute::getMember(const String& name) const
{
  for (List::const_iterator i = members.begin();  i != members.end();  i++)
  {
    if ((*i).name == name)
      return &(*i);
  }

  return NULL;
}

///////////////////////////////////////////////////////////////////////

Attribute& Decorated::createAttribute(const String& name)
{
  if (Attribute* attribute = getAttribute(name))
    return *attribute;

  attributes.push_front(Attribute(*this, name));
  return attributes.front();
}

void Decorated::destroyAttribute(const String& name)
{
  for (AttributeList::iterator i = attributes.begin();  i != attributes.end();  i++)
  {
    if ((*i).getName() == name)
    {
      attributes.erase(i);
      break;
    }
  }
}

void Decorated::destroyAttributes(void)
{
  attributes.clear();
}

Attribute* Decorated::getAttribute(const String& name)
{
  for (AttributeList::iterator i = attributes.begin();  i != attributes.end();  i++)
  {
    if ((*i).getName() == name)
      return &(*i);
  }

  return NULL;
}

const Attribute* Decorated::getAttribute(const String& name) const
{
  for (AttributeList::const_iterator i = attributes.begin();  i != attributes.end();  i++)
  {
    if ((*i).getName() == name)
      return &(*i);
  }

  return NULL;
}

SignalProxy1<void, const Attribute&> Decorated::getChangeValueSignal(void)
{
  return changeValueSignal;
}

SignalProxy1<void, const Attribute&> Decorated::getCreateAttributeSignal(void)
{
  return createAttributeSignal;
}

SignalProxy1<void, const Attribute&> Decorated::getDestroyAttributeSignal(void)
{
  return destroyAttributeSignal;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
