///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2005 Camilla Berglund <elmindreda@elmindreda.org>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////
#ifndef MOIRA_VECTOR_H
#define MOIRA_VECTOR_H
///////////////////////////////////////////////////////////////////////

#if MOIRA_HAVE_MATH_H
#include <math.h>
#endif

#include <vector>

///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

class Vector2
{
public:
  inline Vector2(void);
  inline Vector2(float sx, float sy);
  inline float length(void) const;
  inline float lengthSquared(void) const;
  inline float dotProduct(const Vector2& vector) const;
  inline Vector2 interpolateTo(float t, const Vector2& other) const;
  inline Vector2 absolute(void) const;
  inline Vector2& scaleBy(float factor);
  inline Vector2& scaleTo(float length);
  inline Vector2& negate(void);
  inline Vector2& normalize(void);
  inline operator float* (void);
  inline operator const float* (void) const;
  inline Vector2 operator - (void) const;
  inline Vector2 operator + (float value) const;
  inline Vector2 operator - (float value) const;
  inline Vector2 operator * (float value) const;
  inline Vector2 operator / (float value) const;
  inline Vector2 operator += (float value);
  inline Vector2& operator -= (float value);
  inline Vector2& operator *= (float value);
  inline Vector2& operator /= (float value);
  inline Vector2 operator + (const Vector2& vector) const;
  inline Vector2 operator - (const Vector2& vector) const;
  inline Vector2 operator * (const Vector2& vector) const;
  inline Vector2 operator / (const Vector2& vector) const;
  inline Vector2 operator += (const Vector2& vector);
  inline Vector2& operator -= (const Vector2& vector);
  inline Vector2& operator *= (const Vector2& vector);
  inline Vector2& operator /= (const Vector2& vector);
  inline bool operator == (const Vector2& vector) const;
  inline bool operator != (const Vector2& vector) const;
  inline void setDefaults(void);
  inline void set(float sx, float sy);
  float x;
  float y;
  static const Vector2 ZERO;
  static const Vector2 ONE;
  static const Vector2 X;
  static const Vector2 Y;
};

///////////////////////////////////////////////////////////////////////

class Vector2i
{
public:
  inline Vector2i(void);
  inline Vector2i(int x, int y);
  inline Vector2i operator + (const Vector2i& other) const;
  inline Vector2i operator - (const Vector2i& other) const;
  inline Vector2i operator * (const Vector2i& other) const;
  inline Vector2i operator / (const Vector2i& other) const;
  inline Vector2i& operator += (const Vector2i& other);
  inline Vector2i& operator -= (const Vector2i& other);
  inline Vector2i& operator *= (const Vector2i& other);
  inline Vector2i& operator /= (const Vector2i& other);
  inline void setDefaults(void);
  inline void set(int newX, int newY);
  int x;
  int y;
};

///////////////////////////////////////////////////////////////////////

class Vector3
{
public:
  inline Vector3(void);
  inline Vector3(float sx, float sy, float sz);
  inline float length(void) const;
  inline float lengthSquared(void) const;
  inline float dotProduct(const Vector3& vector) const;
  inline Vector3 crossProduct(const Vector3& vector) const;
  inline Vector3 interpolateTo(float t, const Vector3& other) const;
  inline Vector3 absolute(void) const;
  inline Vector3& scaleBy(float factor);
  inline Vector3& scaleTo(float length);
  inline Vector3& mirrorBy(const Vector3& vector);
  inline Vector3& negate(void);
  inline Vector3& normalize(void);
  inline operator float* (void);
  inline operator const float* (void) const;
  inline Vector3 operator - (void) const;
  inline Vector3 operator + (float value) const;
  inline Vector3 operator - (float value) const;
  inline Vector3 operator * (float value) const;
  inline Vector3 operator / (float value) const;
  inline Vector3 operator += (float value);
  inline Vector3& operator -= (float value);
  inline Vector3& operator *= (float value);
  inline Vector3& operator /= (float value);
  inline Vector3 operator + (const Vector3& vector) const;
  inline Vector3 operator - (const Vector3& vector) const;
  inline Vector3 operator * (const Vector3& vector) const;
  inline Vector3 operator / (const Vector3& vector) const;
  inline Vector3 operator += (const Vector3& vector);
  inline Vector3& operator -= (const Vector3& vector);
  inline Vector3& operator *= (const Vector3& vector);
  inline Vector3& operator /= (const Vector3& vector);
  inline bool operator == (const Vector3& vector) const;
  inline bool operator != (const Vector3& vector) const;
  inline void setDefaults(void);
  inline void set(float sx, float sy, float sz);
  float x;
  float y;
  float z;
  static const Vector3 ZERO;
  static const Vector3 ONE;
  static const Vector3 X;
  static const Vector3 Y;
  static const Vector3 Z;
};

///////////////////////////////////////////////////////////////////////

class Vector4
{
public:
  inline Vector4(void);
  inline Vector4(float sx, float sy, float sz, float sw);
  inline Vector4(const Vector3& xyz, float sw);
  inline float length(void) const;
  inline float lengthSquared(void) const;
  inline float dotProduct(const Vector4& vector) const;
  inline Vector4 crossProduct(const Vector4& vector) const;
  inline Vector4 interpolateTo(float t, const Vector4& other) const;
  inline Vector4 absolute(void) const;
  inline Vector4& scaleBy(float factor);
  inline Vector4& scaleTo(float length);
  inline Vector4& normalize(void);
  inline operator float* (void);
  inline operator const float* (void) const;
  inline Vector4 operator - (void) const;
  inline Vector4 operator + (float value) const;
  inline Vector4 operator - (float value) const;
  inline Vector4 operator * (float value) const;
  inline Vector4 operator / (float value) const;
  inline Vector4 operator += (float value);
  inline Vector4& operator -= (float value);
  inline Vector4& operator *= (float value);
  inline Vector4& operator /= (float value);
  inline Vector4 operator + (const Vector4& vector) const;
  inline Vector4 operator - (const Vector4& vector) const;
  inline Vector4 operator * (const Vector4& vector) const;
  inline Vector4 operator / (const Vector4& vector) const;
  inline Vector4 operator += (const Vector4& vector);
  inline Vector4& operator -= (const Vector4& vector);
  inline Vector4& operator *= (const Vector4& vector);
  inline Vector4& operator /= (const Vector4& vector);
  inline bool operator == (const Vector4& vector) const;
  inline bool operator != (const Vector4& vector) const;
  inline void setDefaults(void);
  inline void set(float sx, float sy, float sz, float sw);
  inline void set(const Vector3& xyz, float sw);
  float x;
  float y;
  float z;
  float w;
  static const Vector4 ZERO;
  static const Vector4 ONE;
  static const Vector4 X;
  static const Vector4 Y;
  static const Vector4 Z;
  static const Vector4 W;
};

///////////////////////////////////////////////////////////////////////

template <typename T>
class VectorTrack
{
public:
  typedef std::vector<T> PointList;
  void evaluate(float t, T& result) const;
  T operator () (float t) const;
  PointList points;
};

///////////////////////////////////////////////////////////////////////

typedef VectorTrack<Vector2> VectorTrack2;
typedef VectorTrack<Vector3> VectorTrack3;
typedef VectorTrack<Vector4> VectorTrack4;

///////////////////////////////////////////////////////////////////////

inline Vector2::Vector2(void)
{
}

inline Vector2::Vector2(float sx, float sy):
  x(sx),
  y(sy)
{
}

inline float Vector2::length(void) const
{
  return (float) sqrtf(float(x * x + y * y));
}

inline float Vector2::lengthSquared(void) const
{
  return x * x + y * y;
}

inline float Vector2::dotProduct(const Vector2& vector) const
{
  return x * x + y * y;
}

inline Vector2 Vector2::interpolateTo(float t, const Vector2& other) const
{
  return Vector2(x * (1.f - t) + other.x * t,
                 y * (1.f - t) + other.y * t);
}

inline Vector2 Vector2::absolute(void) const
{
  return Vector2(fabsf(x), fabsf(y));
}

inline Vector2& Vector2::scaleTo(float len)
{
  const float scale = len / length();
  
  x *= scale;
  y *= scale;
  return *this;
}

inline Vector2& Vector2::negate(void)
{
  x = -x;
  y = -y;
  return *this;
}

inline Vector2& Vector2::normalize(void)
{
  const float scale = 1.f / length();
  
  x *= scale;
  y *= scale;
  return *this;
}

inline Vector2::operator float* (void)
{
  return &(x);
}

inline Vector2::operator const float* (void) const
{
  return &(x);
}

inline Vector2 Vector2::operator - (void) const
{
  return Vector2(-x, -y);
}

inline Vector2 Vector2::operator + (float value) const
{
  return Vector2(x + value, y + value);
}

inline Vector2 Vector2::operator - (float value) const
{
  return Vector2(x - value, y - value);
}

inline Vector2 Vector2::operator * (float value) const
{
  return Vector2(x * value, y * value);
}

inline Vector2 Vector2::operator / (float value) const
{
  return Vector2(x / value, y / value);
}

inline Vector2 Vector2::operator += (float value)
{
  x += value;
  y += value;
  return *this;
}

inline Vector2& Vector2::operator -= (float value)
{
  x -= value;
  y -= value;
  return *this;
}

inline Vector2& Vector2::operator *= (float value)
{
  x *= value;
  y *= value;
  return *this;
}

inline Vector2& Vector2::operator /= (float value)
{
  x /= value;
  y /= value;
  return *this;
}

inline Vector2 Vector2::operator + (const Vector2& vector) const
{
  return Vector2(x + vector.x, y + vector.y);
}

inline Vector2 Vector2::operator - (const Vector2& vector) const
{
  return Vector2(x - vector.x, y - vector.y);
}

inline Vector2 Vector2::operator * (const Vector2& vector) const
{
  return Vector2(x * vector.x, y * vector.y);
}

inline Vector2 Vector2::operator / (const Vector2& vector) const
{
  return Vector2(x / vector.x, y / vector.y);
}

inline Vector2 Vector2::operator += (const Vector2& vector)
{
  x += vector.x;
  y += vector.y;
  return *this;
}

inline Vector2& Vector2::operator -= (const Vector2& vector)
{
  x -= vector.x;
  y -= vector.y;
  return *this;
}

inline Vector2& Vector2::operator *= (const Vector2& vector)
{
  x *= vector.x;
  y *= vector.y;
  return *this;
}

inline Vector2& Vector2::operator /= (const Vector2& vector)
{
  x /= vector.x;
  y /= vector.y;
  return *this;
}

inline bool Vector2::operator == (const Vector2& vector) const
{
  return x == vector.x && y == vector.y;
}

inline bool Vector2::operator != (const Vector2& vector) const
{
  return x != vector.x || y != vector.y;
}

inline void Vector2::setDefaults(void)
{
  x = 0.f;
  y = 0.f;
}

inline void Vector2::set(float sx, float sy)
{
  x = sx;
  y = sy;
}

///////////////////////////////////////////////////////////////////////

inline Vector2i::Vector2i(void)
{
}

inline Vector2i::Vector2i(int initX, int initY):
  x(initX),
  y(initY)
{
}

inline Vector2i Vector2i::operator + (const Vector2i& other) const
{
  return Vector2i(x + other.x, y + other.y);
}

inline Vector2i Vector2i::operator - (const Vector2i& other) const
{
  return Vector2i(x - other.x, y - other.y);
}

inline Vector2i Vector2i::operator * (const Vector2i& other) const
{
  return Vector2i(x * other.x, y * other.y);
}

inline Vector2i Vector2i::operator / (const Vector2i& other) const
{
  return Vector2i(x / other.x, y / other.y);
}

inline Vector2i& Vector2i::operator += (const Vector2i& other)
{
  x += other.x;
  y += other.y;
  return *this;
}

inline Vector2i& Vector2i::operator -= (const Vector2i& other)
{
  x -= other.x;
  y -= other.y;
  return *this;
}

inline Vector2i& Vector2i::operator *= (const Vector2i& other)
{
  x *= other.x;
  y *= other.y;
  return *this;
}

inline Vector2i& Vector2i::operator /= (const Vector2i& other)
{
  x /= other.x;
  y /= other.y;
  return *this;
}

inline void Vector2i::setDefaults(void)
{
  x = y = 0;
}

inline void Vector2i::set(int newX, int newY)
{
  x = newX;
  y = newY;
}

///////////////////////////////////////////////////////////////////////

inline Vector3::Vector3(void)
{
}

inline Vector3::Vector3(float sx, float sy, float sz):
  x(sx),
  y(sy),
  z(sz)
{
}

inline float Vector3::length(void) const
{
  return sqrtf(x * x + y * y + z * z);
}

inline float Vector3::lengthSquared(void) const
{
  return x * x + y * y + z * z;
}

inline float Vector3::dotProduct(const Vector3& vector) const
{
  return x * vector.x + y * vector.y + z * vector.z;
}

inline Vector3 Vector3::crossProduct(const Vector3& vector) const
{
  return Vector3(vector.z * y - vector.y * z,
                 vector.x * z - vector.z * x,
		 vector.y * x - vector.x * y);
}

inline Vector3 Vector3::interpolateTo(float t, const Vector3& other) const
{
  return Vector3(x * (1.f - t) + other.x * t,
                 y * (1.f - t) + other.y * t,
                 z * (1.f - t) + other.z * t);
}

inline Vector3 Vector3::absolute(void) const
{
  return Vector3(fabsf(x), fabsf(y), fabsf(z));
}

inline Vector3& Vector3::scaleBy(float factor)
{
  x *= factor;
  y *= factor;
  z *= factor;
  return *this;
}

inline Vector3& Vector3::scaleTo(float len)
{
  const float scale = len / length();
  
  x *= scale;
  y *= scale;
  z *= scale;
  return *this;
}

inline Vector3& Vector3::mirrorBy(const Vector3& vector)
{
  const float scale = 2.f * (x * vector.x + y * vector.y + z * vector.z);
  
  x = vector.x * scale - x;
  y = vector.y * scale - y;
  z = vector.z * scale - z;
  return *this;
}

inline Vector3& Vector3::negate(void)
{
  x = -x;
  y = -y;
  z = -z;
  return *this;
}

inline Vector3& Vector3::normalize(void)
{
  const float scale = 1.f / length();
  
  x *= scale;
  y *= scale;
  z *= scale;
  return *this;
}

inline Vector3::operator float* (void)
{
  return &(x);
}

inline Vector3::operator const float* (void) const
{
  return &(x);
}

inline Vector3 Vector3::operator - (void) const
{
  return Vector3(-x, -y, -z);
}

inline Vector3 Vector3::operator + (float value) const
{
  return Vector3(x + value, y + value, z + value);
}

inline Vector3 Vector3::operator - (float value) const
{
  return Vector3(x - value, y - value, z - value);
}

inline Vector3 Vector3::operator * (float value) const
{
  return Vector3(x * value, y * value, z * value);
}

inline Vector3 Vector3::operator / (float value) const
{
  return Vector3(x / value, y / value, z / value);
}

inline Vector3 Vector3::operator += (float value)
{
  x += value;
  y += value;
  z += value;
  return *this;
}

inline Vector3& Vector3::operator -= (float value)
{
  x -= value;
  y -= value;
  z -= value;
  return *this;
}

inline Vector3& Vector3::operator *= (float value)
{
  x *= value;
  y *= value;
  z *= value;
  return *this;
}

inline Vector3& Vector3::operator /= (float value)
{
  x /= value;
  y /= value;
  z /= value;
  return *this;
}

inline Vector3 Vector3::operator + (const Vector3& vector) const
{
  return Vector3(x + vector.x, y + vector.y, z + vector.z);
}

inline Vector3 Vector3::operator - (const Vector3& vector) const
{
  return Vector3(x - vector.x, y - vector.y, z - vector.z);
}

inline Vector3 Vector3::operator * (const Vector3& vector) const
{
  return Vector3(x * vector.x, y * vector.y, z * vector.z);
}

inline Vector3 Vector3::operator / (const Vector3& vector) const
{
  return Vector3(x / vector.x, y / vector.y, z / vector.z);
}

inline Vector3 Vector3::operator += (const Vector3& vector)
{
  x += vector.x;
  y += vector.y;
  z += vector.z;
  return *this;
}

inline Vector3& Vector3::operator -= (const Vector3& vector)
{
  x -= vector.x;
  y -= vector.y;
  z -= vector.z;
  return *this;
}

inline Vector3& Vector3::operator *= (const Vector3& vector)
{
  x *= vector.x;
  y *= vector.y;
  z *= vector.z;
  return *this;
}

inline Vector3& Vector3::operator /= (const Vector3& vector)
{
  x /= vector.x;
  y /= vector.y;
  z /= vector.z;
  return *this;
}

inline bool Vector3::operator == (const Vector3& vector) const
{
  return x == vector.x && y == vector.y && z == vector.z;
}

inline bool Vector3::operator != (const Vector3& vector) const
{
  return x != vector.x || y != vector.y || z != vector.z;
}

inline void Vector3::setDefaults(void)
{
  x = 0.f;
  y = 0.f;
  z = 0.f;
}

inline void Vector3::set(float sx, float sy, float sz)
{
  x = sx;
  y = sy;
  z = sz;
}	

///////////////////////////////////////////////////////////////////////

inline Vector4::Vector4(void)
{
}

inline Vector4::Vector4(float sx, float sy, float sz, float sw):
  x(sx),
  y(sy),
  z(sz),
  w(sw)
{
}

inline Vector4::Vector4(const Vector3& xyz, float sw):
  x(xyz.x),
  y(xyz.y),
  z(xyz.z),
  w(sw)
{
}

inline float Vector4::length(void) const
{
  return sqrtf(x * x + y * y + z * z + w * w);
}

inline float Vector4::lengthSquared(void) const
{
  return x * x + y * y + z * z + w * w;
}

inline Vector4 Vector4::interpolateTo(float t, const Vector4& other) const
{
  return Vector4(x * (1.f - t) + other.x * t,
                 y * (1.f - t) + other.y * t,
                 z * (1.f - t) + other.z * t,
                 w * (1.f - t) + other.w * t);
}

inline Vector4 Vector4::absolute(void) const
{
  return Vector4(fabsf(x), fabsf(y), fabsf(z), fabsf(w));
}

inline Vector4& Vector4::scaleTo(float len)
{
  const float scale = len / length();

  x *= scale;
  y *= scale;
  z *= scale;
  w *= scale;
  return *this;
}

inline Vector4& Vector4::normalize(void)
{
  const float scale = 1.f / length();

  x *= scale;
  y *= scale;
  z *= scale;
  w *= scale;
  return *this;
}

inline Vector4::operator float* (void)
{
  return &(x);
}

inline Vector4::operator const float* (void) const
{
  return &(x);
}

inline Vector4 Vector4::operator - (void) const
{
  return Vector4(-x, -y, -z, -w);
}

inline Vector4 Vector4::operator + (float value) const
{
  return Vector4(x + value, y + value, z + value, w + value);
}

inline Vector4 Vector4::operator - (float value) const
{
  return Vector4(x - value, y - value, z - value, w - value);
}

inline Vector4 Vector4::operator * (float value) const
{
  return Vector4(x * value, y * value, z * value, w * value);
}

inline Vector4 Vector4::operator / (float value) const
{
  value = 1.f / value;

  return Vector4(x * value, y * value, z * value, w * value);
}

inline Vector4 Vector4::operator += (float value)
{
  x += value;
  y += value;
  z += value;
  w += value;
  return *this;
}

inline Vector4& Vector4::operator -= (float value)
{
  x -= value;
  y -= value;
  z -= value;
  w -= value;
  return *this;
}

inline Vector4& Vector4::operator *= (float value)
{
  x *= value;
  y *= value;
  z *= value;
  w *= value;
  return *this;
}

inline Vector4& Vector4::operator /= (float value)
{
  const float scale = 1.f / value;

  x *= scale;
  y *= scale;
  z *= scale;
  w *= scale;
  return *this;
}

inline Vector4 Vector4::operator + (const Vector4& vector) const
{
  return Vector4(x + vector.x, y + vector.y, z + vector.z, w + vector.w);
}

inline Vector4 Vector4::operator - (const Vector4& vector) const
{
  return Vector4(x - vector.x, y - vector.y, z - vector.z, w - vector.w);
}

inline Vector4 Vector4::operator * (const Vector4& vector) const
{
  return Vector4(x * vector.x, y * vector.y, z * vector.z, w * vector.w);
}

inline Vector4 Vector4::operator / (const Vector4& vector) const
{
  return Vector4(x / vector.x, y / vector.y, z / vector.z, w / vector.w);
}

inline Vector4 Vector4::operator += (const Vector4& vector)
{
  x += vector.x;
  y += vector.y;
  z += vector.z;
  w += vector.w;
  return *this;
}

inline Vector4& Vector4::operator -= (const Vector4& vector)
{
  x -= vector.x;
  y -= vector.y;
  z -= vector.z;
  w -= vector.w;
  return *this;
}

inline Vector4& Vector4::operator *= (const Vector4& vector)
{
  x *= vector.x;
  y *= vector.y;
  z *= vector.z;
  w *= vector.w;
  return *this;
}

inline Vector4& Vector4::operator /= (const Vector4& vector)
{
  x /= vector.x;
  y /= vector.y;
  z /= vector.z;
  w /= vector.w;
  return *this;
}

inline bool Vector4::operator == (const Vector4& vector) const
{
  return x == vector.x && y == vector.y && z == vector.z && w == vector.w;
}

inline bool Vector4::operator != (const Vector4& vector) const
{
  return x != vector.x || y != vector.y || z != vector.z || w != vector.w;
}

inline void Vector4::setDefaults(void)
{
  x = 0.f;
  y = 0.f;
  z = 0.f;
  w = 0.f;
}

inline void Vector4::set(float sx, float sy, float sz, float sw)
{
  x = sx;
  y = sy;
  z = sz;
  w = sw;
}

inline void Vector4::set(const Vector3& xyz, float sw)
{
  x = xyz.x;
  y = xyz.y;
  z = xyz.z;
  w = sw;
}

///////////////////////////////////////////////////////////////////////

template <typename T>
void VectorTrack<T>::evaluate(float t, T& result) const
{
  if (points.size() == 0)
  {
    result.setDefaults();
    return;
  }

  if (t <= 0.f || points.size() == 1)
  {
    result = points.front();
    return;
  }

  if (t >= 1.f)
  {
    result = points.back();
    return;
  }

  unsigned int source = (unsigned int) (t * (points.size() - 1));
  unsigned int target = source + 1;

  const float tt = (t * (points.size() - 1)) - (float) source;

  result = points[source].interpolateTo(tt, points[target]);
}

template <typename T>
T VectorTrack<T>::operator () (float t) const
{
  T result;
  evaluate(t, result);
  return result;
}

///////////////////////////////////////////////////////////////////////

} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
#endif /*MOIRA_VECTOR_H*/
///////////////////////////////////////////////////////////////////////
