///////////////////////////////////////////////////////////////////////
// Moira library
// Copyright (c) 2004 Camilla Berglund <elmindreda@home.se>
//
// This software is provided 'as-is', without any express or implied
// warranty. In no event will the authors be held liable for any
// damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any
// purpose, including commercial applications, and to alter it and
// redistribute it freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you
//     must not claim that you wrote the original software. If you use
//     this software in a product, an acknowledgment in the product
//     documentation would be appreciated but is not required.
//
//  2. Altered source versions must be plainly marked as such, and
//     must not be misrepresented as being the original software.
//
//  3. This notice may not be removed or altered from any source
//     distribution.
//
///////////////////////////////////////////////////////////////////////
#ifndef MOIRA_SOCKET_H
#define MOIRA_SOCKET_H
///////////////////////////////////////////////////////////////////////

namespace moira
{
  
///////////////////////////////////////////////////////////////////////

/*! @brief TCP connection.
 */
class Connection
{
  friend class ClientSocket;
  friend class ServerSocket;
public:
  /*! @return The client side port of this connection.
   */
  unsigned short getClientPort(void) const;
  /*! @return The server side port of this connection.
   */
  unsigned short getServerPort(void) const;
  /*! @return The host name of the connecting client.
   */
  const String& getClientHostName(void) const;
  /*! @return The host name of the connected server.
   */
  const String& getServerHostName(void) const;
  /*! @return The data stream for this connection.
   */
  Stream& getStream(void);
private:
  Connection(void);
  unsigned short clientPort;
  unsigned short serverPort;
  String clientHostName;
  String serverHostName;
  Ptr<Stream> stream;
};

///////////////////////////////////////////////////////////////////////

/*! @brief TCP client connection factory.
 *
 *  Creates TCP data streams.
 */
class ClientSocket
{
public:
  /*! Default constructor.
   */
  ClientSocket(void);
  /*! Destructor.
   */
  virtual ~ClientSocket(void);
  /*! Connects to the specified host and returns its data stream.
   *  @param[in] hostName The name of the host to connect to.
   *  @param[in] port The port to connect to.
   *  @return A Connection object describing the established connection, or @c
   *  NULL if the connection failed.
   */
  Connection* connect(const String& hostName, unsigned short port);
  /*! @return @c true if the next stream will be blocking, otherwise @c false.
   */
  bool isBlocking(void) const;
  /*! Sets whether the next stream will be blocking.
   *  @param[in] enabled@c true to enable blocking, or @c false to disable it.
   */
  void setBlocking(bool enabled);
protected:
  bool blocking;
};

///////////////////////////////////////////////////////////////////////

/*! @brief TCP server socket.
 */
class ServerSocket
{
public:
  /*! Destructor.
   */
  virtual ~ServerSocket(void);
  /*! Accepts a connection and returns the resulting data stream.
   *  @return The data stream for this connection.
   */
  Connection* accept(void);
  /*! @return @c true if this socket blocks when accepting connections,
   *  otherwise @c false.
   */
  bool isBlocking(void) const;
  /*! Sets whether this socket blocks when accepting connections.
   *  @param[in] enabled @c true to enable blocking, otherwise, @c false.
   */
  void setBlocking(bool enabled);
  /*! @return The port on which this socket is listening.
   */
  unsigned short getPort(void) const;
  /*! Creates a server socket listening on the specified port.
   *  @param[in] port The port to listen on.
   *  @return The newly created socket, or @c NULL.
   */
  static ServerSocket* createInstance(unsigned short port);
private:
  ServerSocket(void);
  bool init(unsigned short port);
  int socket;
  bool blocking;
  unsigned short port;
};

///////////////////////////////////////////////////////////////////////

/*! @brief Socket stream.
 */
class SocketStream : public PosixStream
{
public:
  /*! Destructor. Closes the connection.
   */
  ~SocketStream(void);
  /*! @return @c false.
   */
  bool isSeekable(void) const;
  /*! Helper method. Creates a file stream on the specified file descriptor.
   *  @param[in] fd The file descriptor to use.
   *  @param[in] flags The mode flags to use.
   *  @return The newly created stream, or @c NULL if an error occurs.
   */
  static SocketStream* createInstance(int fd, unsigned int flags); 
private:
  SocketStream(void);
  bool init(int fd, unsigned int flags);
};

///////////////////////////////////////////////////////////////////////
  
} /*namespace moira*/

///////////////////////////////////////////////////////////////////////
#endif /*MOIRA_SOCKET_H*/
///////////////////////////////////////////////////////////////////////
