//-----------------------------------------------------------------------------
// File: terraindetail.fx
//
// Matthew Cotton
// mlcot (201163835)
// Waurn Ponds, On Campus
//
// Copyright (c) Matthew Cotton.
//-----------------------------------------------------------------------------

//--------------------------------------------------------------------------------------
// Global variables
//--------------------------------------------------------------------------------------
float4x4	g_WorldMatrix;			// World matrix for object
float4x4	g_ViewMatrix;			// View matrix
float4x4	g_WorldViewProjection;	// World * View * Projection matrix
float3		g_fvLightDirection;		// direction of the light
float4		g_fvAmbientColour;		// colour and power of ambient light
float3		g_fvEyePosition;		// position of the eye/camera in world
float		g_fMaxDetailDistance;	// max distance that the detail is shown
texture		g_tTexture;				// texture for the terrain;
texture		g_tGrassTexture;		// grass detail texture
texture		g_tRocksTexture;		// rocks detail texture
texture		g_tRockMaskTexture;		// rock mask texture
float		g_fFogStart;				// fog start
float		g_fFogEnd;				// fog end

sampler2D g_sTerrain =
sampler_state
{
	Texture = <g_tTexture>;
	MinFilter = None;
	MagFilter = Linear;
};

sampler2D g_sGrassDetail =
sampler_state
{
	Texture = <g_tGrassTexture>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
};

sampler2D g_sRocksDetail =
sampler_state
{
	Texture = <g_tRocksTexture>;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
};

sampler2D g_sRockMask =
sampler_state
{
	Texture = <g_tRockMaskTexture>;
	MinFilter = None;
	MagFilter = Linear;
	MipFilter = None;
};

struct VS_INPUT
{
	float4 Position   : POSITION;
	float3 Normal     : NORMAL;
	float2 TexCoords1 : TEXCOORD0;
	float2 TexCoords2 : TEXCOORD1;
};

struct VS_OUTPUT
{
	float4 Position       : POSITION;
	float2 TexCoords1     : TEXCOORD0;
	float2 TexCoords2     : TEXCOORD1;
	float3 Normal         : TEXCOORD2;
	float  Distance       : TEXCOORD3;
	float  Fog			  : FOG;
};

struct PS_INPUT
{
	float2 TexCoords1    : TEXCOORD0;
	float2 TexCoords2    : TEXCOORD1;
	float3 Normal        : TEXCOORD2;
	float  Distance      : TEXCOORD3;
};

VS_OUTPUT vs_main(VS_INPUT Input)
{
	VS_OUTPUT Output;
	
	// basic information
	Output.Position = mul( Input.Position, g_WorldViewProjection );
	Output.TexCoords1 = Input.TexCoords1;
	Output.TexCoords2 = Input.TexCoords2;
	Output.Normal = Input.Normal;
	
	// calculate distance to eye
	float3 fvEyeVector = mul( Input.Position, g_WorldMatrix );
	fvEyeVector = g_fvEyePosition - fvEyeVector;
	
	Output.Distance = length(fvEyeVector);
	
	float fFogRange = g_fFogEnd - g_fFogStart;
	float fVertDist = g_fFogEnd - Output.Distance;
	float f = clamp((fVertDist/fFogRange), 0.0f, 1.0f);
	
	Output.Fog = f;
	
	return ( Output );
}

float4 ps_main(PS_INPUT Input) : COLOR
{
	// sample the terrain tex
	float4 fvTerrain = tex2D( g_sTerrain, Input.TexCoords1 );
	
	// sample the grass detail
	float4 fvGrassDetail = tex2D( g_sGrassDetail, Input.TexCoords2 );
	
	// sample the rocks detail
	float4 fvRocksDetail = tex2D( g_sRocksDetail, Input.TexCoords2 );
	
	// sample the rock mask
	float4 fvRockMask = tex2D( g_sRockMask, Input.TexCoords1 );
	
	// the more white in the mask the more rocks to output
	// the more black in the mask the more grass to output
	float4 fvRocksColour = fvRocksDetail * fvRockMask;
	float4 fvGrassColour = fvGrassDetail * (1 - fvRockMask.r); // grayscale means we can use any field
	float4 fvDetailColour = fvRocksColour + fvGrassColour; // add the two
	
	// wash out distant detail
	fvDetailColour += Input.Distance / g_fMaxDetailDistance;
	fvDetailColour = clamp(0,1,fvDetailColour);
	//fvDetailColour.a = 1;
	
	// modulate the terrain and details
	float4 fvTexColour = fvTerrain * fvDetailColour;
	//fvTexColour.a = 1;
	
	// calculate the diffuse power
	float fDiffuse = max( 0, dot( normalize( -g_fvLightDirection ), normalize( Input.Normal ) ) );
	
	// result
	return float4( (g_fvAmbientColour * fvTexColour) + (fDiffuse * fvTexColour) );
}


//--------------------------------------------------------------------------------------
// Techniques
//--------------------------------------------------------------------------------------
technique RenderScene
{
    pass P0
    {
		VertexShader = compile vs_2_0 vs_main();
		PixelShader = compile ps_2_0 ps_main();
    }
}
