#include "turret.h"

#include "constants.h"
#include "data.h"
#include "entity.h"
#include "entityarchtype.h"
#include "weaponslot.h"

//############################################################################
// Turret construction #######################################################
//############################################################################

/** Default constructor.
 * @param op_pos Position relative to host.
 * @param op_zpos Z position of turret.
 * @param mina Minimum angle.
 * @parma maxa Maximum angle.
 * @param op_dir Direction.
 * @param archtype Archtype to use for physical data.
 */
Turret::Turret(const libfhi::Vector2 &op_pos, float op_zpos, int mina,
    int maxa, int op_dir, EntityArchtype *archtype) :
  EntityBase(0, 0), base_class(archtype), pos(op_pos), zpos(op_zpos),
  min_dir(mina), max_dir(maxa), dir(op_dir)
{
  // Do nothing.
}

/** Copy constructor.
 * @param op Source.
 * @param h Host entity.
 */
Turret::Turret(Turret *op, Entity *h) :
  EntityBase(h->get_faction(), op->base_class->get_life_maximum()),
  base_class(op->base_class), host(h), pos(op->pos), zpos(op->zpos),
  min_dir(op->min_dir), max_dir(op->max_dir), dir(op->dir), charge_now(0),
  charge_last(0), ai_target(NULL), ai_last_select(-AI_TURRET_RESELECT_DELAY)
{
  std::vector<CollisionMap*> *base_maps =
    this->base_class->get_collisionmaps();
  for(std::vector<CollisionMap*>::iterator i = base_maps->begin(),
      e = base_maps->end(); (i != e); ++i)
  {
    this->add(*i);
  }

  std::vector<libfhi::Mesh*> *base_meshes =
    this->base_class->get_meshes();
  for(std::vector<libfhi::Mesh*>::iterator i = base_meshes->begin(),
      e = base_meshes->end(); (i != e); ++i)
  {
    this->add(*i);
  }

  std::vector<WeaponSlot*> *base_weaponslots =
    this->base_class->get_weaponslots();
  for(std::vector<WeaponSlot*>::iterator i = base_weaponslots->begin(),
      e = base_weaponslots->end(); (i != e); ++i)
  {
    this->add(*i);
  }
}

/** Default destructor.
 */
Turret::~Turret()
{
  for(std::vector<CollisionMap*>::iterator i = this->maps.begin(),
      e = this->maps.end(); (i != e); ++i)
  {
    delete *i;
  }

  for(std::vector<libfhi::PostModel*>::iterator i = this->models.begin(),
      e = this->models.end(); (i != e); ++i)
  {
    delete *i;
  }

  for(std::vector<WeaponSlot*>::iterator i = this->weaponslots.begin(),
      e = this->weaponslots.end(); (i != e); ++i)
  {
    delete *i;
  }
}

//############################################################################
// Turret methods ############################################################
//############################################################################

/** Copy one collision map to a collision map in this.
 * @param op Collision map source.
 */
void Turret::add(CollisionMap *op)
{
  if(op == NULL)
  {
    std::cerr <<
      "Warning: trying to add an empty collision map to a turret.\n";
  }

  this->maps.push_back(new CollisionMap(op, this));
}

/** Copy one mesh to a model in this.
 * @param op Mesh source.
 */
void Turret::add(libfhi::Mesh *op)
{
  if(op == NULL)
  {
    std::cerr <<
      "Warning: trying to add an empty mesh to a turret.\n";
  }

  this->models.push_back(new libfhi::PostModel(op));
}

/** Copy one weaponslot to a weaponslot in this.
 * @param op Weapon slot source.
 */
void Turret::add(WeaponSlot *op)
{
  if(op == NULL)
  {
    std::cerr <<
      "Warning: trying to add an empty weapon slot to a turret.\n";
  }

  this->weaponslots.push_back(new WeaponSlot(op));
}

/** Artificial intelligence of turrets is a simple one. If they don't have a
 * target, they select something close and track it until it leaves area or
 * is destroyed or random number generator instructs to select a new one.
 * @param hpos Host position.
 * @param hmov Host movement (per tick).
 * @param hdir Host direction.
 */
void Turret::ai_turret(const libfhi::Vector2 &hpos,
    const libfhi::Vector2 &hmov, uint16_t hdir)
{
  // Do not do anything if no life (dead).
  if(this->life <= 0)
  {
    return;
  }

  // Get terrain and some other stuff.
  Terrain *terrain = Terrain::instance;
  int frame_number = Game::get_frame_number();
  float pi = libfhi::uint2pi(hdir),
	cr = cosf(pi),
	sr = sinf(pi);

  // Get absolute position.
  libfhi::Vector2 apos;
  apos.rotate(this->pos, cr, sr);
  apos += hpos;

  // If the ai target has perished, nullify it. Please note, that there is a
  // very, very slight chanse that the target has perished but another one has
  // appeared with identical address. Should this happen, let us just say
  // there is a glitch in the turret artificial intelligence. Hey, it's the
  // truth, ain't it?
  if(!Game::instance->entity_exists(this->ai_target))
  {
    this->ai_target = NULL;
  }

  // Try to select a new target if our current is not a good one.
  if((frame_number - this->ai_last_select > AI_TURRET_RESELECT_DELAY) &&
      ((!this->ai_target) ||
       (terrain->get_relative_bounded_position(this->ai_target->get_pos(),
					       apos).length_sqr() >
	AI_TURRET_TRACK_DISTANCE_SQR) ||
       (rand() % AI_TURRET_RESELECT == 0)))
  {
    // We can select again in the given number of frames.
    this->ai_last_select = frame_number;

    std::vector<std::pair<float, Entity*> > *targets =
      Entity::ai_get_proximity_targets(this->faction, 1, apos,
	  AI_TURRET_TRACK_DISTANCE_SQR);

    // If no targets, do nothing.
    if(targets->empty())
    {
      this->ai_target = NULL;
    }
    // Otherwise pick first.
    else
    {
      this->ai_target = targets->front().second;
    }

    // Delete the target list, it's useless now.
    delete targets;
  }

  // We aren't firing without a reason.
  bool fire_permission = false;

  // If we still have a target, try to shoot it.
  if(this->ai_target)
  {
    float wspeed;

    if(this->weaponslots.empty())
    {
      wspeed = 1.0f;
    }
    else
    {
      wspeed = this->weaponslots.front()->get_speed();
    }

    // Get lookahead (assuming we want to hit).
    float lookahead = Entity::ai_aim_lookahead(apos, hmov,
	this->ai_target->get_pos(), wspeed);

    if(lookahead != FLT_MAX)
    {
      // lookahead now contains the absolute angle we want to look at if
      // we are willing to hit. Get the position this turret exactly would
      // need to aim at by substracting host direction. Note that AND with 16
      // lower bits equals capping to 16-bit.
      int lookat_int = (libfhi::pi2int(lookahead) - hdir) & 0x0000FFFF;

      this->dir = Entity::ai_try_rotate(this->dir, lookat_int,
	  this->base_class->get_rotspeed());

      // Clamp direction.
      if(this->min_dir != this->max_dir)
      {
	int
	  dir_distance_min = (this->dir & 0x0000FFFF) -
	    (this->min_dir & 0x0000FFFF),
	  dir_distance_max = (this->dir & 0x0000FFFF) -
	    (this->max_dir & 0x0000FFFF);

	if(dir_distance_min > 32768)
	{
	  dir_distance_min -= 65536;
	}
	else if(dir_distance_min < -32768)
	{
	  dir_distance_min += 65536;
	}
	if(dir_distance_max > 32768)
	{
	  dir_distance_max -= 65536;
	}
	else if(dir_distance_max < -32768)
	{
	  dir_distance_max += 65536;
	}

	if(abs(dir_distance_min) < abs(dir_distance_max))
	{
	  if(dir_distance_min < 0)
	  {
	    this->dir = this->min_dir;
	  }
	}
	else
	{
	  if(dir_distance_max > 0)
	  {
	    this->dir = this->max_dir;
	  }
	}
      }
      /*// Clamp direction.
	this->dir =
	libfhi::bound(this->dir % 65536, this->min_dir, this->max_dir);*/

      // Turrets try to fire unless the difference is way too big.
      fire_permission = ((abs(this->dir - lookat_int) % 65536) <=
	  AI_TURRET_FIRE_ARC);
    }
  }

  // Get fire decision from AI.
  this->charge_now = Entity::ai_fire_decision(fire_permission,
      this->charge_now, &(this->weaponslots));

  // Iterate through all weapons.
  for(std::vector<WeaponSlot*>::iterator i = this->weaponslots.begin(),
      e = this->weaponslots.end(); (i != e); ++i)
  {
    (*i)->fire(this->faction, apos, hmov, this->dir + hdir, this->charge_now,
	this->charge_last);
    (*i)->tick();
  }

  // Update charge.
  this->charge_last = this->charge_now;
}

/** Draw this turret.
 * @param relpos Relative position (already bounded) of host.
 * @param rot Rotation of host.
 */
void Turret::draw(const libfhi::Vector2 &relpos, uint16_t rot)
{
  // Do not draw if no life (dead).
  if(this->life <= 0)
  {
    return;
  }

  // Get rotations.
  float pi = libfhi::uint2pi(rot);

  // Get absolute position, then rotate it.
  libfhi::Vector2 apos;
  apos.rotate_translate(this->pos, cosf(pi), sinf(pi), relpos);

  for(std::vector<libfhi::PostModel*>::iterator i = this->models.begin(),
      e = this->models.end(); (i != e); ++i)
  {
    (*i)->set_pos(apos.xf, apos.yf, this->zpos);
    (*i)->set_angles(0, 0, this->dir + rot);
    (*i)->tick();
    libfhi::Surface::draw_model(*i);
  }
}

/** Insert collision maps of this turret onto the world.
 * @param hpos Host position.
 * @param cr Cosine of host rotation.
 * @param sr Sine of host rotation.
 * @param angle Host direction.
 */
void Turret::insert_collision_maps(const libfhi::Vector2 &hpos,
    float cr, float sr, uint16_t angle)
{
  // Dead turrets take no collision shite.
  if(this->life <= 0)
  {
    return;
  }

  Terrain *terrain = Terrain::instance;

  float pi = libfhi::uint2pi(static_cast<uint16_t>(angle + this->dir));

  // Rotate local center.
  libfhi::Vector2 tcenter;
  tcenter.rotate_translate(this->pos, cr, sr, hpos);

  // Get real-world cosine and sine.
  cr = cosf(pi);
  sr = sinf(pi);

  // Iterate and insert, note slightly different approach as operations are
  // actually done to the host of this turret thingy instead.
  for(std::vector<CollisionMap*>::iterator i = this->maps.begin(),
      e = this->maps.end(); (i != e); ++i)
  {
    (*i)->transform(tcenter, cr, sr);

    // Get the square and possible first colliding object.
    CollisionSquare *sq = terrain->get_collision_square(*i);
    CollisionMap *hit = sq->collides_freeform(*i);

    // If a hit happened
    if((hit) && (hit->get_host()->get_entity() != this->host))
    {
      this->host->collide_with(hit->get_host()->get_entity());
    }
    else
    {
      CollisionMap::CollisionLine *line =
	sq->collides_terrain_freeform(*i);
      if(line)
      {
	this->host->collide_with(line);
      }
    }

    // Insert into the square found.
    sq->insert(*i);
  }
}

/** Invoke death of this turret.
 */
void Turret::invoke_death()
{
  Effect *die = this->base_class->get_die();

  if(die)
  {
    int cnt = this->base_class->get_die_multitude();

    libfhi::Vector2 hpos = this->host->get_pos();
    uint16_t hang = this->host->get_angle(),
	     tang = this->dir + hang;
    float pi = libfhi::uint2pi(hang);

    libfhi::Vector2 tpos;
    tpos.rotate_translate(this->pos, cosf(pi), sinf(pi), hpos);

    while(cnt--)
    {
      this->base_class->invoke_sparks(die, tpos, tang);
    }
  }
}

//############################################################################
// Virtual methods ###########################################################
//############################################################################

/** Return the entity of this, in this case, the host.
 */
Entity* Turret::get_entity()
{
  return this->host;
}

/** Take damage, if reach zero or less, die instantly.
 * @param op Damage amount.
 */
void Turret::take_damage(int op)
{
  this->life -= op;

  if(this->life <= 0)
  {
    this->invoke_death();
  }
}

//############################################################################
// End #######################################################################
//############################################################################

