#include "sound.h"

#include "game.h"
#include "terrain.h"

//############################################################################
// Channel construction ######################################################
//############################################################################

/** Default constructor.
 * @param ch Channel id.
 */
Channel::Channel(int ch) :
  channel(ch)
{
  // Do nothing.
}

/** Default destructor.
 */
Channel::~Channel()
{
  // Do nothing.
}

//############################################################################
// Channel methods ###########################################################
//############################################################################

/** Pause the channel.
 */
void Channel::Pause() const
{
  Mix_Pause(channel);
}

/** Resume playing on channel.
 */
void Channel::Resume() const
{
  Mix_Resume(channel);
}

/** Stop playing on channel.
 */
void Channel::Stop() const
{
  Mix_HaltChannel(channel);
}

/** Fade the channel out to silence.
 * @param t Fadeout time.
 */
void Channel::FadeOut(int t) const
{
  Mix_FadeOutChannel(channel, t);
}

/** Tell if the channel is paused.
 * @return True if yes, false if no.
 */
bool Channel::IsPaused() const
{
  return Mix_Paused(channel)==1;
}

/** Tell if the channel if playing.
 * @return True if yes, false if no.
 */
bool Channel::IsPlaying() const
{
  return Mix_Playing(channel)==1;
}

//############################################################################
// Sample construction #######################################################
//############################################################################

/** Load a new sample from a file.
 * @param filename File to load from.
 */
Sample::Sample(const char *filename) :
  last_play_frame(0)
{
  sample = Mix_LoadWAV(filename);
}

/** Destroy this sample.
 */
Sample::~Sample()
{
  Mix_FreeChunk(sample);
}

//############################################################################
// Sample methods ############################################################
//############################################################################

/** Play a sample with a volume relevant to the distance between the source
 * and the viewport.
 * @param pos Sound source position.
 * @return Channel id the sample is played in or -1 if not.
 */
int Sample::Play(const libfhi::Vector2 &pos)
{
  // Fetch viewport data from Game.
  libfhi::Vector2 tpos = Terrain::instance->get_relative_bounded_position(
      Game::instance->get_viewport(), pos);

  // Get distance from viewport.
  float len = tpos.length();

  if(len < SOUND_FADE_THRESHOLD)
  {
    // Play with default.
    this->Play();
  }
  else if(len < SOUND_FADE_THRESHOLD + SOUND_FADE_AREA)
  {
    this->Play(lrintf(static_cast<float>(MIX_MAX_VOLUME) *
	  (len - SOUND_FADE_THRESHOLD) / SOUND_FADE_AREA));
  }
  // Otherwise no play, too far.
  return -1;
}

/** Play this sample in a channel.
 * @param vol Volume to play with.
 * @param loops How many times this sample is looped.
 * @return Channel id the sample is played in or negative if not.
 */
int Sample::Play(int vol, int loops)
{
  int currframe = Game::get_frame_number();

  // If there is too little time since last play, wait instead of flooding
  // the channels. This is pretty ugly, but also pretty much necessary.
  if((last_play_frame <= currframe) &&
      (last_play_frame + SOUND_SAMPLE_WAIT > currframe))
  {
    return -1;
  }

  int ch = Mix_PlayChannel(-1, sample, loops);
  Mix_Volume(ch, vol);

  // Set that this sample was last played on the given frame.
  this->last_play_frame = currframe;

  // Return the channel id.
  return ch;
}

/** Fade in the sample.
 * @param t Fade in time.
 * @param loops How many times this sample is looped.
 * @return Channel id the sample is played in.
 */
int Sample::FadeIn(int t, int loops) const
{
  int ch = Mix_FadeInChannel(-1, sample, loops, t);
  return ch;
}

//############################################################################
// End #######################################################################
//############################################################################

