#include "particlesystem.h"

#include "globals.h"
#include "terrain.h"

//############################################################################
// Construction ##############################################################
//############################################################################

/** Empty constructor.
 */
ParticleSystem::ParticleSystem()
  : msh(NULL), model(NULL), visible_count(0)
{
  // Do nothing.
}

/** Default constructor.
 * @param filename Texture filename to use with the particles.
 */
ParticleSystem::ParticleSystem(const char *filename)
  : msh(NULL), model(NULL), visible_count(0)
{
  libfhi::ConfigFile filu(filename);

  if(!filu.is_ok())
  {
    std::cerr << "Error: Could not open particle system: \"" << filename <<
      "\"\n";
  }

  // These need not be saved into a field.
  float particle_size = 1.0f;
  std::string texture_file;

  while(filu.advance())
  {
    if(filu.has_id_arg("count", 1))
    {
      this->visible_count = filu.get_int(0);
    }
    else if(filu.has_id_arg("size", 1))
    {
      particle_size = filu.get_float(0) *
	static_cast<float>(g_screen->get_h());
    }
    else if(filu.has_id_arg("texture", 1))
    {
      texture_file = filu.get_string(0);
    }
    else
    {
      filu.warn_empty();
    }
  }

  // Construct the mesh in somewhat dummy way. Generate a premodel, add all
  // the things to it, then writeout. This will generate a ready-to-go mesh
  // without any of the hassle of actually knowing what happens inside the
  // mesh structure.
  libfhi::PreModel *pre = new libfhi::PreModel();

  // Fill with vertices and points pointing at those vertices.
  size_t i = this->visible_count;
  do {
    pre->add_face(pre->add_vertex(
	  libfhi::Vector3(0.0f, 0.0f, 0.0f),
	  libfhi::Color4(1.0f, 1.0f, 1.0f, 1.0f),
	  libfhi::Vector2(0.0f, 0.0f)
	  )->get_idx());
  } while(--i);

  // Set the texture file to use.
  pre->set_texture_file(texture_file.c_str());

  // Compile into mesh of this.
  this->msh = pre->compile(NULL);
  this->msh->set_point_size(particle_size);

  // Create a new model to handle particles.
  this->model = new libfhi::PostModel(msh);

  // Delete temporary premodel while exiting.
  delete pre;

  // Final echo.
  std::cout << "Loaded particle system \"" << filename << "\" with S(" <<
    particle_size << ") C(" << static_cast<uint32_t>(this->visible_count) <<
    ") T(\"" << texture_file << "\")\n";
}

/** Default destructor.
 */
ParticleSystem::~ParticleSystem()
{
  // Delete all particles.
  this->clear_particles();

  // Get rid of the model and mesh.
  delete this->model;
  delete this->msh;
}

//############################################################################
// ParticleSystem ############################################################
//############################################################################

/** Add one particle to this system.
 * @param c Color.
 * @param p Position.
 * @param s Speed.
 * @param d Direction.
 * @param l Lifetime.
 */
void ParticleSystem::add(const libfhi::Color4 &c, const libfhi::Vector2 &p,
    float s, uint16_t d, int l)
{
  float pi = libfhi::uint2pi(d);

  // Movement derieved from speed and direction.
  libfhi::Vector2 mov(cosf(pi) * s,
      sinf(pi) * s);

  // Call basic constructor.
  this->particles.push_back(new Particle(c, p, mov, l));
}

/** Add one particle to this system.
 * @param c Color.
 * @param p Position.
 * @param d Direction vector.
 * @param l Lifetime.
 */
void ParticleSystem::add(const libfhi::Color4 &c, const libfhi::Vector2 &p,
    const libfhi::Vector2 &d, int l)
{
  // Now this is just too simple.
  this->particles.push_back(new Particle(c, p, d, l));
}

/** Clear all particles in this system.
 */
void ParticleSystem::clear_particles()
{
  // Delete all particles.
  for(std::list<Particle*>::iterator i = this->particles.begin(),
      e = this->particles.end(); (i != e); ++i)
  {
    delete (*i);
  }

  // Clear.
  this->particles.clear();
}

/** Draw the whole particle system to a viewport.
 * @param view Camera position.
 */
void ParticleSystem::draw(const libfhi::Vector2 &view)
{
  libfhi::Vector3 *vdst = this->msh->get_array_vertex();
  libfhi::Color4 *cdst = this->msh->get_array_color();

  // Get the number of particles to be drawn. If 0, abort.
  size_t drawcnt = stdmin(this->visible_count, this->particles.size());

  if(drawcnt <= 0)
  {
    return;
  }

  // Set the drawing thingies to the mesh.
  this->msh->set_num_vertex(drawcnt);
  this->msh->set_num_elem(libfhi::MESH_POINTS,
      static_cast<MESH_TYPEDEF_INDEX>(drawcnt));

  Terrain *terrain = Terrain::instance;

  // Iterate through all particles.
  std::list<Particle*>::iterator iter = this->particles.begin();
  do {
    // Get vertex and bound it.
    libfhi::Vector2 tpos = terrain->get_relative_bounded_position(view,
	(*iter)->get_pos());

    // Save to vertex destination.
    vdst->xf = tpos.xf;
    vdst->yf = tpos.yf;
    vdst->zf = 0.0f;

    // Plain copy color to color destination.
    *cdst = (*iter)->get_col();

    // Increment pointers.
    ++vdst;
    ++cdst;
    ++iter;
  } while(--drawcnt);

  libfhi::Surface::draw_model(this->model);
}

/** Tick all particles and remove dead ones.
 */
void ParticleSystem::tick()
{
  for(std::list<Particle*>::iterator iter = this->particles.begin();
      (iter != this->particles.end());)
  {
    std::list<Particle*>::iterator curr = iter;
    ++iter;

    // Tick particle and remove it if need be.
    if(!(*curr)->tick())
    {
      delete *curr;
      this->particles.erase(curr);
    }
  }
}

//############################################################################
// End #######################################################################
//############################################################################

