#include "gamechallenge.h"

#include "challenge.h"
#include "commands.h"
#include "globals.h"
#include "levelitem.h"
#include "stringinput.h"
#include "weapon.h"

#include <sstream>

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 * @param filename Filename to load the parameters from.
 */
GameChallenge::GameChallenge(const char *filename)
  : Game(filename)
{
  // No need to set slowdown here, it is already being set upon initiating
  // challenge mode.
}

/** Default destructor.
 */
GameChallenge::~GameChallenge()
{
  // Do nothing.
}

//############################################################################
// Virtual methods ###########################################################
//############################################################################

/** Only draw the backplane, leave the drawing method itself to the base
 * method.
 * @param screen Screen to output to.
 */
void GameChallenge::draw(libfhi::Surface *screen)
{
  // Clear only the depth buffer, since all will be drawn.
  screen->clear(libfhi::Boundary::ZBUFFER_MAX);

  // Use host drawing method for objects.
  this->Game::draw(screen);

  // Draw score.
  this->Game::draw_score(screen);
}

/** Tick method.
 */
void GameChallenge::tick()
{
  this->Game::tick();

  // End game if reached correct frame.
  if((Game::frame_end) && (Game::frame_number >= Game::frame_end))
  {
    if(this->string_input)
    {
      // Check for update.
      LevelItem *it = Challenge::instance->get_current_level();

      if(it->need_update_time(Game::frame_number))
      {
	it->set_best_time(Game::frame_number,
	    this->string_input->get_input());
      }

      if(it->need_update_score(this->score))
      {
	it->set_best_score(this->score,
	    this->string_input->get_input());
      }
    }

    command_challenge();
  }
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Delete an existing entity. Also remove it from the entity table.
 * @param iter Iterator to the entity list in this.
 * @param allow_scoring Allow score accumulation, default to false.
 */
void GameChallenge::entity_delete(std::list<Entity*>::iterator iter,
    bool allow_scoring)
{
  // Actually delete via fallback.
  Game::entity_delete(iter, allow_scoring);

  // If there is only one left (highlander-style), start a new phase.
  if((this->entities.size() == 1) && (this->player))
  {
    this->challenge_end();
  }
}

/** End the challenge. This will check if the accumulated score was greater
 * or time shorter than previous values.
 */
void GameChallenge::challenge_end()
{
  LevelItem *it = Challenge::instance->get_current_level();

  bool lowertime = it->need_update_time(Game::frame_number),
       higherscore = it->need_update_score(this->score);

  // If no higher score, just end with delay.
  if(!higherscore && !lowertime)
  {
    Game::frame_end = Game::frame_number + GAME_END_DELAY;
  }
  else
  {
    std::stringstream header;
    this->pause = true;

    header << "You have achieved " <<
      ((higherscore && lowertime) ? "both " : " ") <<
      ((higherscore) ? "higher score " : " ") <<
      ((higherscore && lowertime) ? "and " : " ") <<
      ((lowertime) ? "lower time " : " ") <<
      "than previous record. Please enter your name:";

    this->string_input = new StringInput(header.str());
    memset(this->keys_now, 0, sizeof(int) * COMMAND_TOTAL);
  }
}

//############################################################################
// End #######################################################################
//############################################################################

