#include "game.h"

#include "commands.h"
#include "data.h"
#include "effectdb.h"
#include "entityarchtype.h"
#include "menu.h"
#include "music.h"
#include "options.h"
#include "particlesystem.h"
#include "stringinput.h"
#include "textonplayfield.h"
#include "weapon.h"
#include "weaponslot.h"

#include "SDL.h"

#include <sstream>

//############################################################################
// Variables #################################################################
//############################################################################

int Game::frame_number = 0,
    Game::frame_end = 0;

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 */
Game::Game() :
  entity_number(0), follow(NULL), viewport(0.0f, 0.0f), player(NULL),
  camera(NULL), light(NULL), score(0),
  destroy_tick(-GAME_TIME_BONUS_AREA * GAME_TIME_BONUS_GRANULARITY),
  string_input(NULL), pause(false)
{
  // Do nothing.
}

/** Load constructor. This is an immensely large method.
 * @param filename Filename to load the parameters from.
 */
Game::Game(const char *filename) :
  entity_number(0), follow(NULL), viewport(0.0f, 0.0f), player(NULL),
  camera(NULL), light(NULL), score(0),
  destroy_tick(-GAME_TIME_BONUS_AREA * GAME_TIME_BONUS_GRANULARITY),
  string_input(NULL), pause(false)
{
  libfhi::ConfigFile filu(filename);

  if(!filu.is_ok())
  {
    std::cerr << "Error: Could not open level: \"" << filename << "\"\n";
    return;
  }

  // Create entity archtypes. New entities will be created by copying from
  // these. Must be in same order as the entity enum.
  this->add_entity_archtype("entity/b-1a3.ent");
  this->add_entity_archtype("entity/b-1d3.ent");
  this->add_entity_archtype("entity/bx-2.ent");
  this->add_entity_archtype("entity/genon.ent");
  this->add_entity_archtype("entity/interdimensional_battleship.ent");
  this->add_entity_archtype("entity/lightbydo.ent");
  this->add_entity_archtype("entity/nulaylaf.ent");
  this->add_entity_archtype("entity/nulayragom.ent");
  this->add_entity_archtype("entity/psyvariar_boss.ent");
  this->add_entity_archtype("entity/r-11a.ent");
  this->add_entity_archtype("entity/r-11s2.ent");
  this->add_entity_archtype("entity/r-13a.ent");
  this->add_entity_archtype("entity/r-13b.ent");
  this->add_entity_archtype("entity/r-9a.ent");
  this->add_entity_archtype("entity/r-9b2.ent");
  this->add_entity_archtype("entity/r-9d.ent");
  this->add_entity_archtype("entity/r-9d2.ent");
  this->add_entity_archtype("entity/r-9dh2.ent");
  this->add_entity_archtype("entity/r-9dv2.ent");
  this->add_entity_archtype("entity/r-9k.ent");
  this->add_entity_archtype("entity/r-9o.ent");
  this->add_entity_archtype("entity/rva-818.ent");
  this->add_entity_archtype("entity/rx-10.ent");
  this->add_entity_archtype("entity/surge.ent");
  this->add_entity_archtype("entity/tx-t.ent");

  // Create all particle structures.
  this->add_particlesystem("effect/spark.part");
  this->add_particlesystem("effect/puff.part");
  this->add_particlesystem("effect/explosion.part");
  this->add_particlesystem("effect/hex.part");
  this->add_particlesystem("effect/boom.part");

  // Clear particles.
  for(std::vector<ParticleSystem*>::iterator
      i = this->particlesystems.begin(), e = this->particlesystems.end();
      (i != e);
      ++i)
  {
    (*i)->clear_particles();
  }

  // Create all weapon structures. Hardcoding, sorry. Must be at the exact
  // same order as the weapon identifiers in the enum.
  this->add_weapon("weapon/blaster_bydo.weap");
  this->add_weapon("weapon/blaster_normal.weap");
  this->add_weapon("weapon/bullet_bydo.weap");
  this->add_weapon("weapon/gemini.weap");
  this->add_weapon("weapon/missile_eyeball.weap");
  this->add_weapon("weapon/missile_lrm.weap");
  this->add_weapon("weapon/missile_nulaylaf.weap");
  this->add_weapon("weapon/missile_photon.weap");
  this->add_weapon("weapon/missile_srm.weap");
  this->add_weapon("weapon/pulsar.weap");
  this->add_weapon("weapon/taito_laser.weap");
  this->add_weapon("weapon/wave_cannon_devil_wave.weap");
  this->add_weapon("weapon/wave_cannon_mini_burst.weap");
  this->add_weapon("weapon/wave_cannon_photon_belt.weap");
  this->add_weapon("weapon/wave_cannon_shock.weap");
  this->add_weapon("weapon/wave_cannon_standard.weap");

  // Clear weapons.
  for(std::vector<Weapon*>::iterator i = this->weapons.begin(),
      e = this->weapons.end(); (i != e); ++i)
  {
    (*i)->clear_munitions();
  }

  // Load the effect database and empty it (this will initialize it if it is
  // not already done).
  this->effectdb = EffectDB::instance_get();

  // Clear effects.
  this->effectdb->clear_effects();

  // Front camera.
  this->camera = new libfhi::Camera();
  this->camera->set_angles(0, 0, 0);
  this->camera->set_rotation(libfhi::Orientation::ROTATION_NO);
  this->camera->set_pos(0.0f, 0.0f, VIEW_CAMERA_PLANE);

  // Front light.
  if(!this->light)
  {
    this->light = new libfhi::LightDirectional();
    this->light->set_ambient(0.8f, 0.8f, 0.8f);
    this->light->set_diffuse(0.2f, 0.2f, 0.2f);
    this->light->set_angles(0, 800, 1589);
  }

  // Empty keys.
  memset(this->keys_now, 0, sizeof(int) * COMMAND_TOTAL);

  // Create difficulty groups. Here and only here, one at a time.
  this->add_scoring_rule(ENTITY_GENON, 1);
  this->add_scoring_rule(ENTITY_SURGE, 1);
  this->add_scoring_rule(ENTITY_PSYVARIAR_BOSS, 2);
  this->add_scoring_rule(ENTITY_R_9A, 2);
  this->add_scoring_rule(ENTITY_R_9DV2, 2);
  this->add_scoring_rule(ENTITY_RX_10, 3);
  this->add_scoring_rule(ENTITY_R_13A, 3);
  this->add_scoring_rule(ENTITY_RVA_818, 3);
  this->add_scoring_rule(ENTITY_R_9D, 4);
  this->add_scoring_rule(ENTITY_INTERDIMENSIONAL_BATTLESHIP, 5);
  this->add_scoring_rule(ENTITY_NULAYRAGOM, 5);

  // Later entity insertions list.
  std::list<Entity*> later_entity_insertion;

  // Loop the file.
  while(filu.advance())
  {
    // Set lighting.
    if(filu.has_id_arg("light", 9))
    {
      if(!this->light)
      {
	this->light = new libfhi::LightDirectional();
      }

      this->light->set_ambient(filu.get_float(0),
	  filu.get_float(1),
	  filu.get_float(2));
      this->light->set_diffuse(filu.get_float(3),
	  filu.get_float(4),
	  filu.get_float(5));
      this->light->set_angles(filu.get_int(6),
	  filu.get_int(7),
	  filu.get_int(8));
    }
    // Load terrain. This should be after the mechanics and dimensions
    // definitions. Failing to do this, those will be set to default values.
    else if(filu.has_id_arg("terrain", 1))
    {
      Terrain::instance_delete();
      Terrain::instance_new(new Terrain(filu.get_string(0).c_str()));
    }
    // Objects are loaded in two phases. First, objects whose coordinates are
    // specified explicitly. Objects specified with random position will be
    // pushed onto the
    else if(filu.has_id("entity"))
    {
      // Syntax is either x y dir faction name
      // or
      // faction name
      // Latter produces random location.
      switch(filu.get_num_arg())
      {
	case 5:
	  {
	    Entity *ent = this->entity_new(
		filu.get_string(4).c_str(),
		filu.get_int(3));
	    this->entity_insert(ent,
		libfhi::Vector2(filu.get_float(0), filu.get_float(1)),
		filu.get_int(2),
		false);
	  }
	  break;

	// Random inserts are delayed since we don't want them conflicting
	// with the actual inserts.
	case 2:
	  later_entity_insertion.push_back(this->entity_new(
		filu.get_string(1).c_str(),
		filu.get_int(0)));
	  break;

	default:
	  filu.warn("needs 2 or 5 arguments");
	  break;
      }
    }
    // Lock player to last seen entity.
    else if(filu.has_id("player"))
    {
      std::cout << "Setting player.\n";
      this->player = this->entities.back();
    }
    // Lock player to last seen entity.
    else if(filu.has_id("follow"))
    {
      std::cout << "Setting follow.\n";
      this->follow = this->entities.back();
    }
    // TODO: Implement cube loading.
    else
    {
      filu.warn_empty();
    }
  }

  // Terrain has now been reserved.
  Terrain *terrain = Terrain::instance;

  // Perform the random entity insertions that were scheduled for later.
  for(std::list<Entity*>::iterator i = later_entity_insertion.begin(),
      e = later_entity_insertion.end(); (i != e); ++i)
  {
    bool inserted = false;

    // Try to insert in random spot.
    for(int cnt = ENTITY_INSERTION_TRY_COUNT; (--cnt);)
    {
      libfhi::Vector2 ipos = terrain->get_random_position();
      uint16_t idir = rand() % 65536;

      if(this->entity_insert(*i, ipos, idir, true, GAME_SAFE_DISTANCE))
      {
	inserted = true;
	break;
      }
    }

    // Did not work, just delete.
    if(!inserted)
    {
      this->entity_delete(*i);
      std::cout << "Warning: Could not insert an entity.\n";
    }
  }

  // Inform loading is complete.
  std::cout << "Loaded terrain \"" << filename << "\"\n";

  // Start playing the game music.
  Music *musa = Music::instance_get();
  musa->Init("music/Avangstgarden_-_Chinchilla_Goes_Sinister.ogg");

  // Set the static frame number variable to 0. This will be incremented
  // continuously.
  Game::frame_number = 0;
  Game::frame_end = 0;
}

/** Default destructor. Note that any archtypes are NOT destroyed. Entity
 * archtypes, particlesystems and weapons stay in the cache.
 */
Game::~Game()
{
  // Delete front viewing constructs.
  delete this->camera;
  delete this->light;

  // Delete all entities.
  for(std::list<Entity*>::iterator i = this->entities.begin(),
      e = this->entities.end(); (i != e); ++i)
  {
    delete *i;
  }

  // Delete the score groups.
  for(HASHMAP<int, std::vector<EntityEnum>*>::iterator
      i = this->score_entity_by_difficulty.begin(),
      e = this->score_entity_by_difficulty.end();
      (i != e); ++i)
  {
    delete (*i).second;
  }

  // Delete string input thingy if it exists.
  delete this->string_input;
}

//############################################################################
// Inherited methods #########################################################
//############################################################################

#ifdef DEBUG
extern libfhi::PostModel *pathmodel;
#endif

/** The drawing method of Game only draws the entities. The iheriting drawing
 * methods must take care of clearing the screens and drawing the background.
 * @param screen Screen to draw to.
 */
void Game::draw(libfhi::Surface *screen)
{
  Terrain *terrain = Terrain::instance;

  float backplane = (VIEW_CAMERA_PLANE + terrain->get_bottom_depth()) + 0.1f;
  int x1 = 0,
      y1 = 0,
      x2 = screen->get_h() - 1,
      y2 = screen->get_h() - 1;


  // If we have something to follow, update the position here. Otherwise
  // viewport is left in the same position it was in the last frame.
  if(this->follow)
  {
    viewport = this->follow->get_pos();
  }

  // Switch to foreground drawing mode. Note that the nearplane is set so,
  // that the camera plane is located at the one third of the depth buffer.
  screen->set_boundary(x1, y1, x2, y2, VIEW_RATIO,
      VIEW_CAMERA_PLANE * 0.5f,
      backplane);

  // Enable the foreground camera and light.
  screen->set_camera(this->camera);
  screen->set_light(this->light);
  screen->select_3d();

  // Draw terrain if it exists.
  Terrain::instance->draw(viewport);

  // Draw all entities.
  for(std::list<Entity*>::iterator i = this->entities.begin(),
      e = this->entities.end(); (i != e); ++i)
  {
    (*i)->draw(viewport);
  }

#ifdef DEBUG
  // Draw path.
  if(pathmodel)
  {
    pathmodel->set_pos(-viewport.xf, -viewport.yf, 0.0f);
    pathmodel->tick();
    libfhi::Surface::draw_model(pathmodel);
  }
#endif

  // Bullets may have alpha channels, me want to make sure they are drawn on
  // top of everything, that is, slightly in front of vehicles but only
  // slightly.
  screen->set_boundary(x1, y1, x2, y2, VIEW_RATIO,
      VIEW_CAMERA_PLANE * 0.7f,
      backplane);
  screen->select_3d();

  // Draw all weapons.
  for(std::vector<Weapon*>::iterator i = this->weapons.begin(),
      e = this->weapons.end(); (i != e); ++i)
  {
    (*i)->draw(viewport);
  }

  // Draw all particles.
  for(std::vector<ParticleSystem*>::iterator
      i = this->particlesystems.begin(), e = this->particlesystems.end();
      (i != e);
      ++i)
  {
    (*i)->draw(viewport);
  }

  // Switch to 2d drawing mode for the 2d text thingies.
  screen->set_boundary(x1, y1, x2, y2);
  screen->select_2d();

  float area_float = static_cast<float>(screen->get_h()) /
    (VIEW_RATIO * VIEW_CAMERA_PLANE * 2.0f);

  for(std::list<TextOnPlayfield*>::iterator i = this->texts.begin(),
      e = this->texts.end(); (i != e); ++i)
  {
    // Draw this.
    libfhi::Vector2 tpos = terrain->get_relative_bounded_position(
	viewport, (*i)->get_pos());

    // Get positions. This is very unwieldy, perhaps I should think of some
    // kind of unproject() to libfhi?
    int
      screen_x = lrintf(tpos.xf * area_float + screen->get_xcenter()),
      screen_y = lrintf(screen->get_ycenter() - tpos.yf * area_float);

    screen->draw_text(screen_x, screen_y, (*i)->get_color(),
	Menu::get_font_desc(), (*i)->get_text());
  }

  // Switch to 2d drawing mode and clear the right side of the screen.
  x1 = screen->get_h();
  y1 = 0;
  x2 = screen->get_w() - 1;
  y2 = screen->get_h() - 1;

  screen->set_boundary(x1, y1, x2, y2);
  screen->select_2d();
  // Draw separator bar, it gets gobbled by point sptrites anyway (it should
  // not, but it does).
  libfhi::Surface::nc_rect(x1, y1, x1 + 1, y2, Menu::get_color_border());

  // Increment x1 and blank the rest.
  x1 += 2;
  libfhi::Surface::nc_rect(x1, y1, x2, y2,
      libfhi::Surface::makecol3(0, 0, 0));

  // Display all weapon gauges.
  if(this->player)
  {
    int y = Menu::draw_map(x1, x2, viewport, &(this->entities)),
	charge = this->player->get_charge();
    float percent = static_cast<float>(this->player->get_life()) /
      static_cast<float>(this->player->get_base_class()->get_life_maximum());

    // Life gauge.
    y = Menu::draw_charge_bar(y,
	stdmin(percent, 1.0f),
	libfhi::Surface::makecol3(160, 40, 40),
	"Life",
	screen);

    // Draw absorb charge at the top if need be,
    if(this->player->get_base_class()->has_absorb())
    {
      int absorb = this->player->get_absorb_state();

      if(absorb < 0)
      {
	absorb = -absorb;
      }

      // Normal gauge.
      percent = static_cast<float>(stdmin(absorb, ABSORB_WAIT_THRESHOLD)) /
	static_cast<float>(ABSORB_WAIT_THRESHOLD);

      Menu::draw_charge_bar(y,
	  percent,
	  libfhi::Surface::makecol3(160, 110, 185),
	  "Absorb",
	  screen);

      // Wait gauge.
      percent = static_cast<float>(stdmax(0, absorb - ABSORB_WAIT_THRESHOLD))
	/ static_cast<float>(ABSORB_TIME_MAX - ABSORB_WAIT_THRESHOLD);

      y = Menu::draw_charge_bar(y,
	  percent,
	  libfhi::Surface::makecol3(210, 25, 100),
	  "Absorb",
	  screen);

      // Iterate through weapon types and display absorbtion counts.
      int col = libfhi::Surface::makecol3(135, 190, 160);
      for(int i = 0; (i < static_cast<int>(WEAPON_COUNT)); ++i)
      {
	WeaponEnum weaponid = static_cast<WeaponEnum>(i);
	Weapon *weap = this->weapons[i];

	// Only weapons that can be absorbed are displayed.
	if(weap->get_absorb() > 0)
	{
	  percent = static_cast<float>(
	      this->player->get_absorb_ammo(weaponid)) /
	    static_cast<float>(weap->get_absorb());

	  y = Menu::draw_charge_bar(y, percent, col,
	      this->weapons[i]->get_name(), screen);
	}
      }
    }
    else
    {
      // Iterate through the weapon slots and draw them.
      std::vector<WeaponSlot*> *slots = this->player->get_weaponslots();

      for(std::vector<WeaponSlot*>::iterator i = slots->begin(),
	  e = slots->end(); (i != e); ++i)
      {
	y = (*i)->draw(y, charge, screen);
      }
    }
  }

  // If string input exists, draw it.
  if(this->string_input)
  {
    this->string_input->draw(VIEW_STRING_INPUT, screen);
  }
}

/** Game keydown command.
 * @param key Pressed key.
 */
void Game::keydown(int key)
{
  // If have string input, instead redirect there.
  if(this->string_input)
  {
    if(this->string_input->keydown(key))
    {
      this->set_end_time();
    }
    return;
  }

  if(key == SDLK_ESCAPE)
  {
    command_ingame_menu();
  }
  else if((key == SDLK_RIGHT) || (key == Options::key_right))
  {
    keys_now[COMMAND_RIGHT]++;
  }
  else if((key == SDLK_UP) || (key == Options::key_up))
  {
    keys_now[COMMAND_FORWARD]++;
  }
  else if((key == SDLK_LEFT) || (key == Options::key_left))
  {
    keys_now[COMMAND_LEFT]++;
  }
  else if((key == SDLK_DOWN) || (key == Options::key_down))
  {
    keys_now[COMMAND_BACKWARD]++;
  }
  else if(key == Options::key_absorb)
  {
    keys_now[COMMAND_ABSORB]++;
  }
  else if(key == Options::key_target)
  {
    keys_now[COMMAND_TARGET]++;
  }
  else if(key == Options::key_fire)
  {
    keys_now[COMMAND_FIRE]++;
  }
}

/** Game keyup command.
 * @param key Released key.
 */
void Game::keyup(int key)
{
  if(this->string_input)
  {
    this->string_input->keyup(key);
    return;
  }

  if((key == SDLK_RIGHT) || (key == Options::key_right))
  {
    keys_now[COMMAND_RIGHT] = 0;
  }
  else if((key == SDLK_UP) || (key == Options::key_up))
  {
    keys_now[COMMAND_FORWARD] = 0;
  }
  else if((key == SDLK_LEFT) || (key == Options::key_left))
  {
    keys_now[COMMAND_LEFT] = 0;
  }
  else if((key == SDLK_DOWN) || (key == Options::key_down))
  {
    keys_now[COMMAND_BACKWARD] = 0;
  }
  else if(key == Options::key_absorb)
  {
    keys_now[COMMAND_ABSORB] = 0;
  }
  else if(key == Options::key_target)
  {
    keys_now[COMMAND_TARGET] = 0;
  }
  else if(key == Options::key_fire)
  {
    keys_now[COMMAND_FIRE] = 0;
  }
}

/** Ticker for game does a lot of stuff.
 */
void Game::tick()
{
  // Iterate the die list, delete all entities from there, then remove them
  // from actual die list. In the end this is cheaper than iterating through
  // the list in normal manner and checking for deaths.
  while(!this->die_list.empty())
  {
    Entity *dying = die_list.back();

    dying->invoke_death_sparks();
    dying->clear_from_terrain();

    // Search the entity list, remove from there.
    for(std::list<Entity*>::iterator iter = this->entities.begin();
	(iter != this->entities.end()); ++iter)
    {
      if((*iter) == dying)
      {
	this->entity_delete(iter, true);
	break;
      }
    }

    // Shorten the die list.
    die_list.pop_back();
  }

  // Tick the entities, no end caching since may be altered.
  for(std::list<Entity*>::iterator iter = this->entities.begin();
      (iter != this->entities.end());)
  {
    std::list<Entity*>::iterator curr = iter;
    ++iter;

    // Either send commands or call AI. Do nothing if ai / commands are
    // disabled.
    if(!this->pause)
    {
      if(this->player == *curr)
      {
	(*curr)->input_commands(this->keys_now);
      }
      else
      {
	(*curr)->ai();
      }
    }

    // Check that still exists after iteration.
    if(!(*curr)->tick())
    {
      // Push the thingy to the die list.
      this->die_list.push_back(*curr);
    }
  }

  // Tick all weapons.
  for(std::vector<Weapon*>::iterator i = this->weapons.begin(),
      e = this->weapons.end(); (i != e); ++i)
  {
    (*i)->tick();
  }

  // Tick all particles.
  for(std::vector<ParticleSystem*>::iterator
      i = this->particlesystems.begin(), e = this->particlesystems.end();
      (i != e);
      ++i)
  {
    (*i)->tick();
  }

  // Tick text thingies (mainly check for lifetime over etc.
  for(std::list<TextOnPlayfield*>::iterator iter = this->texts.begin();
      iter != this->texts.end();)
  {
    std::list<TextOnPlayfield*>::iterator curr = iter;
    ++iter;

    if(!(*curr)->tick())
    {
      delete (*curr);
      this->texts.erase(curr);
    }
  }

  // Move the keys to store.
  memcpy(keys_last, keys_now, sizeof(int) * COMMAND_TOTAL);

  int *keys = this->keys_now;
  int cmdcnt = COMMAND_TOTAL;
  do {
    if((*keys) > 0)
    {
      ++(*keys);
    }
    ++keys;
  } while(--cmdcnt);

  // Increment the frame number.
  if(!this->pause)
  {
    ++Game::frame_number;
  }
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Add one entity archtype and set it's type to a given number specified by
 * it's order in the loading process.
 * @param filename Name of the file to load from.
 */
void Game::add_entity_archtype(const char *filename)
{
  EntityArchtype *ent = Data::load_ent(filename);

  ent->set_type(static_cast<EntityEnum>(this->entity_archtypes.size()));
  this->entity_archtypes.push_back(ent);
}

/** Add a particle system.
 * @param name Particle filename.
 */
void Game::add_particlesystem(const char *filename)
{
  this->particlesystems.push_back(Data::load_part(filename));
}

/** Adds a scoring rule.
 * @param ent Entity type.
 * @param difficulty Difficulty of this entity.
 */
void Game::add_scoring_rule(EntityEnum ent, int difficulty)
{
  // Add the other way.
  this->score_difficulty_by_entity[static_cast<int>(ent)] = difficulty;

  // If there is no difficulty group, create it.
  if(score_entity_by_difficulty.find(difficulty) ==
      score_entity_by_difficulty.end())
  {
    this->score_entity_by_difficulty[difficulty] =
      new std::vector<EntityEnum>();
  }

  this->score_entity_by_difficulty[difficulty]->push_back(ent);
}

/** Add one text to be displayed on the playfield each frame.
 * @param pos Position to put the text.
 * @param text Textual content.
 */
void Game::add_text_on_playfield(const libfhi::Vector2 &pos,
    const std::string &text)
{
  this->texts.push_back(new TextOnPlayfield(pos, text));
}

/** Add a weapon construct.
 * @param filename Weapon filename.
 */
void Game::add_weapon(const char *filename)
{
  Weapon *weap = Data::load_weap(filename);
  
  weap->set_type(static_cast<WeaponEnum>(this->weapons.size()));
  this->weapons.push_back(weap);
}

/** Render the score and  score multiplier.
 * @param screen Screen to render to.
 */
void Game::draw_score(libfhi::Surface *screen) const
{
  // After the game drawing the boundary and draw methods will still be in the
  // phase where we can draw the score & such.
  libfhi::Font *font = Menu::get_font();
  int fontsize = Menu::get_fontsize(),
      x1 = Menu::get_text_x(),
      y2 = Menu::get_text_bottom_y(screen),
      y1 = y2 - fontsize,
      x2 = screen->get_w() - 5;

  // Write score and time multiplier.
  std::stringstream scoretext;
  scoretext << this->score;
  libfhi::Surface::draw_text(
      x2 - font->get_string_width(scoretext.str().c_str()),
      y1,
      Menu::get_color_text(),
      font,
      scoretext.str().c_str());

  std::stringstream multext;
  multext << this->get_time_bonus_multiplier() << "x";
  libfhi::Surface::draw_text(
      x1,
      y1 + fontsize,
      Menu::get_color_text(),
      font,
      multext.str().c_str());
}

/** Fetch the list iterator for given entity and delete it.
 * @param op Entity to delete.
 * @return True if entity found, false otherwise.
 */
bool Game::entity_delete(Entity *op)
{
  for(std::list<Entity*>::iterator i = this->entities.begin(),
      e = this->entities.end(); (i != e); ++i)
  {
    if((*i) == op)
    {
      this->entity_delete(i);
      return true;
    }
  }

  return false;
}

/** Delete an existing entity. Also remove it from the entity table.
 * @param iter Iterator to the entity list in this.
 * @param allow_scoring Defaults to false
 */
void Game::entity_delete(std::list<Entity*>::iterator iter,
    bool allow_scoring)
{
  // Check whether to score.
  if(allow_scoring && this->player &&
      ((*iter)->get_faction() != this->player->get_faction()))
  {
    EntityEnum type = (*iter)->get_base_class()->get_type();
    int difficulty = this->get_difficulty(type),
	scoreadd = difficulty * this->get_time_bonus_multiplier();
    this->destroy_tick = Game::frame_number;

    // Add score number on the playfield, then actually increase score by it.
    std::stringstream textadd;
    textadd << scoreadd;
    this->add_text_on_playfield((*iter)->get_pos(), textadd.str());
    this->score += scoreadd;
  }

  // Get integer representation and fetch from hashmap.
  size_t integer_representation = reinterpret_cast<size_t>(*iter);
  HASHMAP<size_t, bool>::iterator table_iter =
    this->entity_table.find(integer_representation);

  // If found, erase from everywhere.
  if(table_iter != this->entity_table.end())
  {
    // Check if we lost something.
    if((*iter) == this->player)
    {
      this->player = NULL;

      // Only set game to end if we are not inputting the stuff right now.
      if(!this->string_input)
      {
	Game::frame_end = Game::frame_number + GAME_END_DELAY;
      }
    }
    if((*iter) == this->follow)
    {
      this->follow = NULL;
    }

    this->entity_table.erase(table_iter);
    delete (*iter);
    this->entities.erase(iter);
  }
  // Otherwise something is wrong. BAD.
  else
  {
    std::cerr <<
      "Error: Unable to delete entity, not found in exists table!\n";
  }
}

/** Insert entity to given position. The entity is not inserted or removed
 * from any possible lists, it is not a job for this method.
 * @param ent Entity.
 * @param pos Position to insert in.
 * @param dir Initial direction.
 * @param care Does the insert method care about anything beneath.
 * @param safe_distance Set at least this far from any entity.
 * @return True if was inserted, false if not.
 */
bool Game::entity_insert(Entity *ent, const libfhi::Vector2 &pos,
    uint16_t dir, bool care, float safe_distance)
{
  ent->set_pos(pos);
  ent->set_angle(dir);

  // Care about safe distance?
  if((safe_distance > 0.0f) && (this->player))
  {
    Terrain *terrain = Terrain::instance;

    libfhi::Vector2 rel = terrain->get_relative_bounded_position(pos,
	this->player->get_pos());

    // Abort if too close to player
    if(rel.length() <= safe_distance)
    {
      return false;
    }
  }

  // To care or not to care?
  if(care)
  {
    return ent->try_insert();
  }

  // Not.
  ent->insert();
  return true;
}

/** Creates a new entity, pushes it in the back of entity list, and returns
 * it.
 * @param archtype Entity archtype to copy
 * @param faction Faction to create the entity under.
 * @return Pointer to newly allocated entity.
 */
Entity* Game::entity_new(EntityArchtype *archtype, int faction)
{
  // Create a new entity and initialize it from the archtypes.
  Entity *ret = new Entity(archtype,
      faction | (this->entity_number << FACTION_SHIFT));

  // Push the entity to our list, then increment number of entities created.
  this->entities.push_back(ret);
  ++this->entity_number;

  // Add the entity to our table.
  size_t integer_representation = reinterpret_cast<size_t>(ret);
  this->entity_table[integer_representation] = true;

  // Return.
  return ret;
}

/** Wrapped version.
 * @param type Entity type enum.
 * @param faction Faction to create the entity under.
 * @return Pointer to newly allocated entity.
 */
Entity* Game::entity_new(EntityEnum type, int faction)
{
  return this->entity_new(this->entity_archtypes[type], faction);
}

/** Wrapped version.
 * @param name Entity type name.
 * @param faction Faction to create the entity under.
 * @return Pointer to newly allocated entity.
 */
Entity* Game::entity_new(const char *name, int faction)
{
  return this->entity_new(Data::load_ent(name), faction);
}

/** Get difficulty of an entity.
 * @param op Entity type.
 */
int Game::get_difficulty(EntityEnum op)
{
  return this->score_difficulty_by_entity[static_cast<int>(op)];
}

/** Get a random entity by difficulty.
 * @param op Difficulty value.
 */
EntityEnum Game::get_random_entity(int op)
{
  std::vector<EntityEnum> *vec = this->score_entity_by_difficulty[op];

  return (*vec)[rand() % vec->size()];
}

/** Gets the current time bonus multiplier.
 * @return Current multiplier.
 */
int Game::get_time_bonus_multiplier() const
{
  int timediff = Game::frame_number - this->destroy_tick;

  return stdmax(GAME_TIME_BONUS_AREA - (timediff /
	GAME_TIME_BONUS_GRANULARITY), 1);
}

//############################################################################
// End #######################################################################
//############################################################################

