#ifndef effectdb_h
#define effectdb_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "effect.h"

//############################################################################
// Class #####################################################################
//############################################################################

/** One effect instance. This is barely a combination of effect and it's para-
 * meters.
 */
class EffectInstance
{
  private:
    /** Vector parameter 1. */
    libfhi::Vector2 param1;

    /** Vector parameter 2. */
    libfhi::Vector2 param2;

    /** The effect used. */
    const Effect *effect;

    /** Lifetime of the instance. */
    int lifetime;

  public:
    inline EffectInstance(const libfhi::Vector2*, const libfhi::Vector2*,
	const Effect*, int);
    inline ~EffectInstance();

    // Methods.
  public:
    inline bool tick();
};

/** The effect database, holds all the effects currently active, removes them
 * when their lifetime runs out.
 */
class EffectDB :
  public libfhi::Singleton<EffectDB>
{
  private:
    /** List of all effect instances currently active. */
    std::list<EffectInstance*> effects;

  public:
    EffectDB();
    ~EffectDB();

  public:
    void clear_effects();
    void tick();

    // Inline methods.
  public:
    inline void add(EffectInstance*);
};

//############################################################################
// EffectInstance inline #####################################################
//############################################################################

/** Default constructor.
 * @param op1 The first parameter.
 * @param op2 The second parameter.
 * @param efu The effect.
 * @param l Lifetime.
 */
EffectInstance::EffectInstance(const libfhi::Vector2 *op1,
    const libfhi::Vector2 *op2, const Effect *efu, int l)
  : effect(efu), lifetime(l)
{
  // It is theoretically possible I shall add effects with no
  // parameters
  if(op1)
  {
    this->param1 = (*op1);
  }
  // This, however, might me null many times.
  if(op2)
  {
    this->param2 = (*op2);
  }
}

/** Default destructor.
 */
EffectInstance::~EffectInstance()
{
  // Do nothing.
}

/** Tick the effect instance one tick.
 */
bool EffectInstance::tick()
{
  if((this->lifetime--) <= 0)
  {
    return false;
  }

  this->effect->execute_now(&(this->param1), &(this->param2));
  return true;
}

//############################################################################
// EffectDB inline ###########################################################
//############################################################################

/** Add one effect instance here.
 * @param op Effect instance to add.
 */
void EffectDB::add(EffectInstance *op)
{
  this->effects.push_back(op);
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

