#ifndef collisionmap_h
#define collisionmap_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi.h"

#include "enums.h"

#include <float.h>

//############################################################################
// Define ####################################################################
//############################################################################

// Forward declaration.
class CollisionSquare;
class Entity;
class EntityBase;
class Terrain;

//############################################################################
// Class #####################################################################
//############################################################################

/** CollisionMap describes one set of collision constructs with a maximum
 * largest radius of half the collision square size. Larger constructs are
 * possible, but will not work correctly. There is no need for distinct
 * indicators for the center position of CollisionMap, since upon
 * constructing objects, the relative_x and relative_y coordinates are set
 * thus they already point to the center of the map.
 *
 * To clarify, the algorithm is like this.
 * 1) Transform Collision map from object space to global space. This is done
 *    by calling transform(dx, dy, rot), which rotates the collision map over
 *    it's origin (which is the object's origin), then translates it by dx and
 *    dy. The resulting map may or may not need bounding.
 * 2) Insert the map into the collision tree. This is done by the absolute
 *    position of the lower left coordinate. Then, transform the object
 *    coordinates into relative coordinates inside this single square.
 * 3) Now the collision coordinates lie in the relative coordinates inside a
 *    square, when testing for collisions, the difference in square
 *    coordinates is added to the testing algorithms.
 */
class CollisionMap :
  public libfhi::Attributable,
  public libfhi::CanonicalDB<CollisionMap>
{
  public:
    /** True if this is included in the normal mode collisions. */
    static const uint8_t ATTR_NORMAL = 0x01;

    /** True if this is included in the absorb mode collisions. */
    static const uint8_t ATTR_ABSORB = 0x02;

    /** Default attributes. */
    static const uint8_t DEFAULT_ATTR = 0x00;

  public:
    /** A line collision element. */
    typedef struct
    {
      /** First point. */
      libfhi::Vector2 p1;

      /** Second point. */
      libfhi::Vector2 p2;

      /** Direction (unit vector). */
      libfhi::Vector2 dir;

      /** Radius. */
      float r;

      /** Transformed first point. */
      libfhi::Vector2 tp1;

      /** Transformed second point. */
      libfhi::Vector2 tp2;

      /** Transformed direction. */
      libfhi::Vector2 tdir;

      /** Transformed normal. */
      libfhi::Vector2 tnor;

      /** Length of this line. */
      float len;
    } CollisionLine;

    /** A point collision element. */
    typedef struct
    {
      /** Center point. */
      libfhi::Vector2 p;

      /** Radius. */
      float r;

      /** Transformed center point. */
      libfhi::Vector2 tp;
    } CollisionDot;

  private:
    /** Difference between two colliding dots. */
    static libfhi::Vector2 dot_difference;

    /** Lhs comparator line. */
    static CollisionLine *line_lhs;

    /** Rhs comparator line. */
    static CollisionLine *line_rhs;

    /** Collision position for lhs-line. */
    static libfhi::Vector2 collision_position;

  private:
    /** What CollisionSquare this map is currently inside. */
    CollisionSquare *square;

    /** What multimodel this belongs in. */
    EntityBase *host;

    /** Allegiance. */
    int faction;

    /** Current center. */
    libfhi::Vector2 center;

    /** Positio relative to the object center. */
    libfhi::Vector2 relative;

    /** The longest radius within. */
    float longest_radius;

    /** Array of all dot collision elements. */
    CollisionDot *array_dots;

    /** Number of dots in this map. */
    int num_dots;

    /** Array of all line collision elements. */
    CollisionLine *array_lines;

    /** Number of lines in this map. */
    int num_lines;
    
  public:
    CollisionMap();
    CollisionMap(const char*);
    CollisionMap(CollisionMap*, EntityBase*);
    ~CollisionMap();

  public:
    void add_dot(const libfhi::Vector2&, float);
    void add_line(const libfhi::Vector2&, const libfhi::Vector2&, float);
    void clear_from_terrain();
    void precalculate();
    void remove_from_terrain();
    void section(std::vector<CollisionMap*>*);
    void transform(const libfhi::Vector2&, float, float);
    void translate(const libfhi::Vector2&);

    // Static methods.
  public:
    static bool CollisionMap::collision_batch_dot_dots(CollisionDot*,
	const libfhi::Vector2&, CollisionMap*);
    static bool CollisionMap::collision_batch_dot_lines(CollisionDot*,
	const libfhi::Vector2&, CollisionMap*);
    static bool CollisionMap::collision_batch_line_dots(CollisionLine*,
	const libfhi::Vector2&, CollisionMap*);
    static bool CollisionMap::collision_batch_line_lines(CollisionLine*,
	const libfhi::Vector2&, CollisionMap*);
    static bool collides_freeform_freeform(CollisionMap*,
	const libfhi::Vector2&, CollisionMap*);
    static CollisionLine* collides_lines_freeform(CollisionMap*,
	CollisionMap*);
    static CollisionLine* collides_lines_single_dot(CollisionMap*,
	CollisionMap*);
    static CollisionLine* collides_lines_single_line(CollisionMap*,
	CollisionMap*);
    static bool collides_single_dot_freeform(CollisionMap*,
	const libfhi::Vector2&, CollisionMap*);
    static bool collides_single_line_freeform(CollisionMap*,
	const libfhi::Vector2&, CollisionMap*);

    // Inline methods.
  public:
    inline int get_faction() const;
    inline EntityBase* get_host();
    inline CollisionSquare* get_square();
    inline libfhi::Vector2 get_lowerleft() const;
    inline CollisionDot* get_single_dot();
    inline CollisionLine* get_single_line();
    inline bool is_empty() const;
    inline void set_center(const libfhi::Vector2&);
    inline void set_faction(int);
    inline void set_longest_radius(float op);
    inline void set_square(CollisionSquare*);
    inline void transform_single_dot(const libfhi::Vector2&);

    // Static inline template methods.
  public:
    template <class Type1, class Type2>
      static inline bool collides_template(Type1*, const libfhi::Vector2&,
	Type2*);
    template <class Type1, class Type2>
      static inline bool collides_template(Type1*, Type2*);

    // Static inline methods.
  public:
    static inline bool collides_line_line(CollisionLine*, CollisionLine*);
    static inline const libfhi::Vector2& get_collision_position();
    static inline const libfhi::Vector2& get_dot_difference();
    static inline CollisionLine* get_line_lhs();
    static inline CollisionLine* get_line_rhs();
};

//############################################################################
// Inline methods ############################################################
//############################################################################

/** Get the faction of this.
 * @return Faction of this.
 */
int CollisionMap::get_faction() const
{
  return this->faction;
}

/** Get the host model of this collision map.
 * @return Host as a pointer.
 */
EntityBase* CollisionMap::get_host()
{
  return this->host;
}

/** Get the tree square this is in.
 * @return Pointer to the square or NULL.
 */
CollisionSquare* CollisionMap::get_square()
{
  return this->square;
}

/** Get the lower left coordinate of collision map (the one it is set into
 * square upon).
 * @return Current, transformed coordinate.
 */
libfhi::Vector2 CollisionMap::get_lowerleft() const
{
  return libfhi::Vector2(this->center.xf - longest_radius,
      this->center.yf - longest_radius);
}

/** Return the single dot that makes this collision map. Potentially
 * dangerous.
 * @return Pointer to a dot.
 */
CollisionMap::CollisionDot* CollisionMap::get_single_dot()
{
  return this->array_dots;
}

/** Return the single line that makes this collision map. Potentially
 * dangerous.
 * @return Pointer to a line.
 */
CollisionMap::CollisionLine* CollisionMap::get_single_line()
{
  return this->array_lines;
}

/** Tell if this collisionmap is empty.
 * @return True if yes, false if no.
 */
bool CollisionMap::is_empty() const
{
  return (!this->array_dots && !this->array_lines);
}

/** Set the center of this.
 * @param op New center.
 */
void CollisionMap::set_center(const libfhi::Vector2 &op)
{
  this->center = op;
}

/** Set the faction of this.
 * @param op Faction enum.
 */
void CollisionMap::set_faction(int op)
{
  this->faction = op;
}

/** Set the longest radius of this.
 * @param op New longest radius.
 */
void CollisionMap::set_longest_radius(float op)
{
  this->longest_radius = op;
}

/** Set the square of this.
 * @param op Current collision square.
 */
void CollisionMap::set_square(CollisionSquare *op)
{
  this->square = op;
}

/** Extremely simple transformation, assumes one dot exactly at the center of
 * the map, transforms it by translation, nothing else.
 * @param trans Translation.
 */
void CollisionMap::transform_single_dot(const libfhi::Vector2 &trans)
{
  this->center = trans;

  // Array dots already points at the first dot.
  this->array_dots->tp = trans;
}

//############################################################################
// Static inline #############################################################
//############################################################################

/** Get the position of last collision. The position is only updated when a
 * line as a left-hand-side component of collision detection hits another
 * line. Basically, it is used to determine the intersection and collision
 * points when calculating the hits of instant weapons.
 * @return Reference to static collision position vector.
 */
const libfhi::Vector2& CollisionMap::get_collision_position()
{
  return CollisionMap::collision_position;
}

/** Get the right-hand-side collision dot.
 * @return Pointer to last right-hand collision line.
 */
const libfhi::Vector2& CollisionMap::get_dot_difference()
{
  return CollisionMap::dot_difference;
}

/** Get the left-hand-side collision line.
 * @return Pointer to last left-hand collision line.
 */
CollisionMap::CollisionLine* CollisionMap::get_line_lhs()
{
  return CollisionMap::line_lhs;
}

/** Get the right-hand-side collision line.
 * @return Pointer to last right-hand collision line.
 */
CollisionMap::CollisionLine* CollisionMap::get_line_rhs()
{
  return CollisionMap::line_rhs;
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

