#include "libfhi_texture.h"

#include "libfhi_misc.h"
#include "libfhi_glsurface.h"

#ifdef LIBFHI_OPENGL
#include LIBFHI_GLU_H
#endif

#ifdef LIBFHI_PNG
#include "libfhi_png.h"
#include "png.h"
#endif

namespace libfhi {

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 */
Texture::Texture() :
#ifdef LIBFHI_OPENGL
  texture_name(0),
#endif
#ifdef LIBFHI_PNG
  data(NULL),
#endif
  width(0), height(0), bpp(0)
{
  // Do nothing.
}

/** Load constructor.
 * @param filename Image file to open the texture from.
 */
Texture::Texture(const char *filename) :
#ifdef LIBFHI_OPENGL
  texture_name(0),
#endif
#ifdef LIBFHI_PNG
  data(NULL),
#endif
  width(0), height(0), bpp(0)
{
  this->load(filename);
}

/** Default destructor.
 */
Texture::~Texture()
{
#ifdef LIBFHI_OPENGL
  if(this->texture_name)
  {
    glDeleteTextures(1, &(this->texture_name));
  }
#endif
#ifdef LIBFHI_PNG
  delete this->data;
#endif
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Loader. Please note, that by default operation, the texture does nto save
 * the loaded bitmap data and tries to create an openGL image.
 * @param filename File to to load the texture from.
 * @param load_flags Or'd flags determining what to do.
 * @return True on success, false on failure.
 */
bool Texture::load(const char *filename, uint8_t load_flags)
{
#ifdef LIBFHI_PNG
  // Read data.
  this->data = read_png(filename);

  // Abort if unavailable.
  if(this->data == NULL)
  {
    std::cerr << "Error: could read image file \"" << filename << "\"\n";
    return false;
  }

  // Read data.
  this->bpp = get_last_png_b();
  this->height = get_last_png_h();
  this->width = get_last_png_w();

#ifdef LIBFHI_OPENGL
  if(load_flags & KEEP_OPENGL)
  {
    if(!glSurface::is_opengl_initialized())
    {
      std::cerr << "Warning: Not in OpenGL mode, skipping texture \"" <<
	filename << "\"\n";
      // Exiting, delete data if requested.
      if(!(load_flags & KEEP_DATA))
      {
        zdel_array(this->data);
      }
      return false;
    }

    GLenum format;
    switch(this->bpp)
    {
      case 8:
	format = GL_LUMINANCE8;
	break;
      case 24:
	format = GL_RGB;
	break;
      case 32:
	format = GL_RGBA;
	break;
      default:
	std::cerr << "Error: Illegal texture bit depth: " << this->bpp <<
	  " (allowed: 8, 24, 32).\n";
	// Exiting, delete data if requested.
	if(!(load_flags & KEEP_DATA))
	{
	  zdel_array(this->data);
	}
	return false;
    }

    // Generate new texture and bind it.
    glEnable(GL_TEXTURE_2D);
    glGenTextures(1, &(this->texture_name));
    glBindTexture(GL_TEXTURE_2D, this->texture_name);

    glPixelStorei(GL_PACK_ALIGNMENT, 1);
    glPixelStorei(GL_PACK_ROW_LENGTH, this->width);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    glPixelStorei(GL_UNPACK_ROW_LENGTH, this->width);

    // Set mipmapping.
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
	GL_LINEAR_MIPMAP_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
	GL_LINEAR_MIPMAP_LINEAR);

    // Create texture from the data.
    gluBuild2DMipmaps(GL_TEXTURE_2D, 4, this->width, this->height, format,
	GL_UNSIGNED_BYTE, this->data);

    // Disable textures again.
    glDisable(GL_TEXTURE_2D);
  }
#endif

  // Delete data if requested.
  if(!(load_flags & KEEP_DATA))
  {
    zdel_array(this->data);
  }

  std::cout << "Loaded texture \"" << filename << "\" with D(" <<
    this->width << "x" << this->height << ")\n";

  return true;

#else
  std::cout << "Error: no libpng support, texture load aborted.\n";

  // Fallback to returning false.
  return false;
#endif
}

//############################################################################
// Functions #################################################################
//############################################################################

/** Load a new texture from a file.
 * @param filename File to to load the texture from.
 * @return New texture object (in the canonical database) or NULL on error.
 */
Texture* texture_new(const char *filename)
{
#ifdef LIBFHI_OPENGL
  Texture *ret = Texture::canon_get(filename);

  if(!ret)
  {
    ret = Texture::canon_load(filename);
  }

  // If it got a legal texture name it is ok, return it.
  if(ret->get_texture_name())
  {
    return ret;
  }

  // Delete the reserved texture. Then fallback to returning NULL.
  Texture::canon_delete(filename);
#endif

  return NULL;
}

//############################################################################
// End #######################################################################
//############################################################################

}

