#ifndef libfhi_surface_include
#define libfhi_surface_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_boundary.h"
#include "libfhi_font.h"
#include "libfhi_light.h"
#include "libfhi_point.h"

namespace libfhi {

//############################################################################
// Definet ###################################################################
//############################################################################

// Clip specific heap values.
#define LIBFHI_HEAP_POINTDATA_C	(LIBFHI_HEAP_POLY_C + 3 * 4)
#define LIBFHI_HEAP_POLYDATA_C	(LIBFHI_HEAP_POLY_C + 3 * 2)

// Types for the surfaces. It appeared signed values were faster (?).
#define SURFACE_TYPEDEF_8	int8_t
#define SURFACE_TYPEDEF_16	int16_t
#define SURFACE_TYPEDEF_32	int32_t

// Forward declaration.
class Camera;
class Face;
class Glyph;
class PostModel;
class Texture;

//############################################################################
// Luokka ####################################################################
//############################################################################

/** The base class for all surface classes.
 */
class Surface
{
  public:
    /** Flag for no bitmap in this surface. */
    static const uint8_t FLAG_NOBITMAP = 0x1;

    /** Flag for enabled z-buffer in this surface. */
    static const uint8_t FLAG_ZBUFFER = 0x2;

    /** Flag for setting the video mode (window) with this surface. */
    static const uint8_t FLAG_SET_MODE = 0x4;

    /** Default flags for surfaces. */
    static const uint8_t DEFAULT_FLAGS = 0x0;

  protected:
    /** Clipping boundary. */
    Boundary boundary;

    /** Z-buffer pointer. */
    uint16_t *zbuf;

    /** Color buffer pointer. */
    void *cbuf;

    /** Size of this surface's bitmap in bytes. */
    int size_bitmap;
    
    /** Size of this surface's bitmap in pixels. */
    int size_pixels;
    
    /** Size of this surface's z-buffer in bytes. */
    int size_zbuffer;
    
    /** Width of this surface. */
    int w;

    /** Height of this surface. */
    int h;
    
    /** Bits per pixel of this surface. */
    int bpp;

    /** Attributes of this surface. */
    uint8_t attributes;

  protected:
    /** State machine boundary pointer. */
    static Boundary *bound;

    /** State machine camera pointer. */
    static Camera *camera;

    /** State machine light pointer. */
    static Light *light;

    /** Point data array for clipping and state machine. */
    static Point pointdata[LIBFHI_HEAP_POINTDATA_C];
    
    /** Pointer to the first point. */
    static Point *first;

    /** Pointer to the next point to be filled. */
    static Point *next;

    /** Static z-data pointer for drawing primitives. */
    static uint16_t *zdata;

    /** Static color data pointer for drawing primitives. */
    static void *cdata;

    /** Corner count for drawing primitives. */
    static int corners;
    
    /** X multiplier for drawing primitives. */
    static int xmul;

    /** AND mask of all clipping masks. */
    static uint8_t andmask;
    
    /** OR mask of all clipping masks. */
    static uint8_t ormask;    

  protected:
    // Static template functions.
    template <class Type>
      static void nc_setpixel_template(int, int, int);
    template <class Type>
      static void nc_hline_template(int, int, int, int);
    template <class Type>
      static void nc_vline_template(int, int, int, int);
    template <class Type>
      static void nc_line_template(int, int, int, int, int);
    template <class Type>
      static void nc_rect_template(int, int, int, int, int);
    template <class Type>
      static void nc_rect_contour_template(int, int, int, int, int);
    template <class Type>
      static void nc_triangle_flat_template(int);
    template <class Type>
      static void nc_triangle_gouraud_template();
    template <class Type>
      static void nc_poly_flat_template(int);
    template <class Type>
      static void nc_poly_gouraud_template();
    template <class Type>
      static void nc_zbuf_spot_template(int);
    template <class Type>
      static void nc_zbuf_line_flat_template(int);
    template <class Type>
      static void nc_zbuf_line_gouraud_template();
    template <class Type>
      static void nc_zbuf_triangle_flat_template(int);
    template <class Type>
      static void nc_zbuf_triangle_gouraud_template();
    template <class Type>
      static void nc_zbuf_poly_flat_template(int);
    template <class Type>
      static void nc_zbuf_poly_gouraud_template();
    template <class Type>
      static void select_template();

    // Static template functions (always clipped).
    template <class Type>
      static void draw_glyph_template(int, int, int, Glyph*);

    // Templates of templates.
    template <class Type1, class Type2>
      static void nc_triangle_template();
    template <class Type1, class Type2>
      static void nc_poly_template();
    template <class Type1, class Type2>
      static void nc_zbuf_line_template();
    template <class Type1, class Type2>
      static void nc_zbuf_triangle_template();
    template <class Type1, class Type2>
      static void nc_zbuf_poly_template();

    // Tilakoneen leikkausfunktiot
    static void clip_poly_pass(uint8_t, float,
	void (*)(Point*, const Point*, const Point*, float));
    static bool point_clip_spot_zyx();
    template <class Type>
      static bool point_clip_line_zyx();
    template <class Type>
      static bool point_clip_poly_yx();
    template <class Type>
      static bool point_clip_poly_zyx();

    // Clipped primitives.
    static void c_setpixel(int, int, int);
    static void c_hline(int, int, int, int);
    static void c_vline(int, int, int, int);
    static void c_line(int, int, int, int, int);
    static void c_rect(int, int, int, int, int);
    static void c_rect_contour(int, int, int, int, int);
    static void c_poly_flat(int);
    static void c_poly_gouraud();
    static void c_zbuf_line_flat(int);
    static void c_zbuf_line_gouraud();
    static void c_zbuf_poly_flat(int);
    static void c_zbuf_poly_gouraud();

    // High-level operations on surfaces.
    static int c_draw_text(int, int, int, Font*, const char*);
    static void c_draw_model(PostModel*);
    static void c_draw_texture(int, int, int, int, Texture*);

  public:
    // Constructor and destructor.
    Surface();
    virtual ~Surface();

    // Not clipped function pointers.
    static int (*makecol3)(int, int, int);
    static int (*makecol4)(int, int, int, int);
    static void (*nc_setpixel)(int, int, int);
    static void (*nc_hline)(int, int, int, int);
    static void (*nc_vline)(int, int, int, int);
    static void (*nc_line)(int, int, int, int, int);
    static void (*nc_rect)(int, int, int, int, int);
    static void (*nc_rect_contour)(int, int, int, int, int);
    static void (*nc_triangle_flat)(int);
    static void (*nc_triangle_gouraud)();
    static void (*nc_poly_flat)(int);
    static void (*nc_poly_gouraud)();
    static void (*nc_zbuf_spot)(int);
    static void (*nc_zbuf_line_flat)(int);
    static void (*nc_zbuf_line_gouraud)();
    static void (*nc_zbuf_triangle_flat)(int);
    static void (*nc_zbuf_triangle_gouraud)();
    static void (*nc_zbuf_poly_flat)(int);
    static void (*nc_zbuf_poly_gouraud)();

    // Clipped function pointers.
    static void (*setpixel)(int, int, int);
    static void (*hline)(int, int, int, int);
    static void (*vline)(int, int, int, int);
    static void (*line)(int, int, int, int, int);
    static void (*rect)(int, int, int, int, int);
    static void (*rect_contour)(int, int, int, int, int);
    static void (*poly_flat)(int);
    static void (*poly_gouraud)();
    static void (*zbuf_line_flat)(int);
    static void (*zbuf_line_gouraud)();
    static void (*zbuf_poly_flat)(int);
    static void (*zbuf_poly_gouraud)();

    // Always clipped function pointers.
    static void (*draw_glyph)(int, int, int, Glyph*);

    // High-level pointers
    static int (*draw_text)(int, int, int, Font*, const char*);
    static void (*draw_model)(PostModel*);
    static void (*draw_texture)(int, int, int, int, Texture*);

  public:
    // Polygonitilakone public.
    static void nc_point_begin();
    static void point_begin();
    static void nc_point_add(int, int);
    static void nc_point_add(int, int, int, int, int, int);
    static void nc_point_add(int, int, int);
    static void nc_point_add(int, int, int, int, int, int, int);
    static void point_add(float, float);
    static void point_add(float, float, const Color4&);
    static void point_add(const Vector3&);
    static void point_add(const Vector3&, const Color4&);

  protected:
    // Polygon state machine private. Point addition with 3 color
    // components is protected so it can not be accessed from outside
    // class resulting in uncertain state of the alpha component.
    // TODO: Add some kind of friend method?
    static void nc_point_inc();
    static void point_inc();
    static bool point_bfc();
    static void point_add(const Vector3&, float, float, float);

  public:
    // Common utility
    void clear();
    int reserve_zbuffer(uint8_t);
    void set_boundary();
    void set_boundary(int, int, int, int);
    void set_boundary(float, float = Boundary::DEF_CLIP_Z_NEAR,
	float = Boundary::DEF_CLIP_Z_FAR);
    void set_boundary(int, int, int, int, float,
	float = Boundary::DEF_CLIP_Z_NEAR, float = Boundary::DEF_CLIP_Z_FAR);

    // Virtual methods
    virtual void clear(int);
    virtual void clear(uint16_t);
    virtual void clear(int, uint16_t);
    virtual void flip() = 0;
    virtual void lock();
    virtual void null();
    virtual void select();
    virtual void unlock();
    virtual void unreserve();

    // 3D-utility (virtual and static)
    virtual void select_2d();
    virtual void select_3d();
    virtual void set_camera(Camera*);
    virtual void set_light(Light*);

    // Save methods.
#ifdef LIBFHI_PNG
    virtual bool save_png(const char*);
#endif

    // Inline
    static inline void feed_vertex_flat(CVUnit*, int);
    static inline float* feed_vertex_flat_with_color(CVUnit*, int);
    static inline void feed_vertex_gouraud(CVUnit*, int);
    inline Boundary* get_boundary();
    inline int get_bpp() const;
    static inline Camera* get_camera();
    inline void* get_cbuf();
    inline int get_h() const;
    static inline Light* get_light();
    inline int get_w() const;
    inline float get_xcenter() const;
    inline float get_ycenter() const;
};

//############################################################################
// Inline ####################################################################
//############################################################################

/** Feed given vertex from vertex array into the state machine.
 * Flat version.
 * @param array C3F_V3F array.
 * @param idx Index of the vertex.
 */
void Surface::feed_vertex_flat(CVUnit *array, int idx)
{
  CVUnit *ref = array + idx;

  Surface::point_add(ref->ver);
}

/** Feed given vertex from vertex array into the state machine.
 * Flat version with color return.
 * @param array C3F_V3F array.
 * @param idx Index of the vertex.
 * @return Pointer to color of the face.
 */
float* Surface::feed_vertex_flat_with_color(CVUnit *array, int idx)
{
  CVUnit *ref = array + idx;

  Surface::point_add(ref->ver);

  return ref->col;
}

/** Feed given vertex from vertex array into the state machine.
 * Gouraud version.
 * @param array C3F_V3F array.
 * @param idx Index of the vertex.
 */
void Surface::feed_vertex_gouraud(CVUnit *array, int idx)
{
  CVUnit *ref = array + idx;

  Surface::point_add(ref->ver, ref->col[0], ref->col[1], ref->col[2]);
}

/** Get the boundary associated with this.
 * @return Pointer to Boundary.
 */
Boundary* Surface::get_boundary()
{
  return &boundary;
}

/** Get the bit depth of this.
 * @return Bit depth.
 */
int Surface::get_bpp() const
{
  return this->bpp;
}

/** Get the camera associated with this.
 * @return Pointer to the camera.
 */
Camera* Surface::get_camera()
{
  return camera;
}

/** Get the color buffer. Potentially dangerous.
 * @return Void pointer to color buffer.
 */
void* Surface::get_cbuf()
{
  return this->cbuf;
}

/** Get the height of this.
 * @return Height as an integer.
 */
int Surface::get_h() const
{
  return h;
}

/** Get the light associated with this.
 * @return Pointer to the light.
 */
Light* Surface::get_light()
{
  return light;
}

/** Get the width of this.
 * @return Width as an integer.
 */
int Surface::get_w() const
{
  return w;
}

/** Get the x center of this.
 * @return X center as a float.
 */
float Surface::get_xcenter() const
{
  return boundary.get_xcenter();
}

/** Get the y center of this.
 * @return Y center as a float.
 */
float Surface::get_ycenter() const
{
  return boundary.get_ycenter();
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif

