#ifndef libfhi_mesh_include
#define libfhi_mesh_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_attributable.h"
#include "libfhi_canonicaldb.h"
#include "libfhi_color.h"
#include "libfhi_matvec.h"

#include <float.h>

#ifdef LIBFHI_OPENGL
#include LIBFHI_GLEW_H
#endif

namespace libfhi {

//############################################################################
// Definet ###################################################################
//############################################################################

// Forward declaration
class PreModel;
class Texture;

/** ElementType specifies the drawn elements in a mesh. */
enum ElementType
{
  MESH_POINTS = 0, // Points must be the first, don't modify.
  MESH_LINES,
  MESH_TRIANGLES,
  MESH_QUADS,
  MESH_FLAT_LINES,
  MESH_FLAT_TRIANGLES,
  MESH_FLAT_QUADS,
  MESH_ELEMENT_COUNT
};

#ifdef LIBFHI_OPENGL
#define MESH_TYPEDEF_INDEX GLuint
#else
#define MESH_TYPEDEF_INDEX uint32_t
#endif

//############################################################################
// Luokka ####################################################################
//############################################################################

/** Mesh holds one vertex array and one skin.
 * Please note that the vertex array and skin are inseparable due to the order
 * in which they are fed to OpenGL (software implementation reflects this).
 * If there is need for several different skins for the same model, they need
 * to be saved distinctively to different meshes.
 */
class Mesh :
  public Attributable,
  public CanonicalDB<Mesh>
{
  public:
#ifdef LIBFHI_DEBUG
    DEBUG_PRINT_PROTOTYPE(Mesh)
#endif

  public:
#ifdef LIBFHI_OPENGL
    GLuint display_list;
#endif

  public:
    /** Tell if this is a point sprite (particle) system or a normal mesh. */
    static const uint8_t ATTR_POINT_SPRITES = 0x01;

    /** Set this to allow caching in OpenGL (display list). */
    static const uint8_t ATTR_ALLOW_CACHE = 0x02;

    /** Reserved for user applications. */
    static const uint8_t ATTR_USER_1 = 0x04;

    /** Reserved for user applications. */
    static const uint8_t ATTR_USER_2 = 0x08;

    /** Reserved for user applications. */
    static const uint8_t ATTR_USER_3 = 0x10;

    /** Reserved for user applications. */
    static const uint8_t ATTR_USER_4 = 0x20;

    /** Reserved for user applications. */
    static const uint8_t ATTR_USER_5 = 0x40;

    /** Reserved for user applications. */
    static const uint8_t ATTR_USER_6 = 0x80;

    /** Default attributes. */
    static const uint8_t DEFAULT_ATTR = 0x00;

  private:
    /** Array of color data for the vertices. */
    Color4 *color;

    /** Array of coordinate data for the vertices. */
    Vector3 *vertex;

    /** Array of normal data for the vertices. */
    Vector3 *vnormal;

    /** Array of texture coordinates. */
    Vector2 *texcoord;

    /** Point size for "voxels" (read: point sprites). */
    float point_size;
    
    /** Pointer to skeleton object. */
    void *skel;

    /** Number of vertices. */
    size_t num_vertex;

    /** Arrays of different elements. */
    MESH_TYPEDEF_INDEX *elem[MESH_ELEMENT_COUNT];

    /** Texture. */
    Texture *texture;

  public:      
    Mesh();
    Mesh(Mesh*);
    ~Mesh();

  public:
    void generate_frame(int, int, float);
    float get_maxdist() const;
    float get_maximum_in_axis(int) const;
    float get_minimum_in_axis(int) const;
    void null();
    void reserve_vertex(size_t);
    void reserve_face(MESH_TYPEDEF_INDEX[MESH_ELEMENT_COUNT]);
    void reserve_multiple(Mesh*, size_t);
    void set_point_size(float);

#ifdef LIBFHI_OPENGL
  public:
    GLuint reserve_display_list();

  public:
    inline GLuint get_display_list() const; 
#endif

    // Inline.
  public:
    inline Color4* get_array_color();
    inline MESH_TYPEDEF_INDEX* get_array_elem(ElementType);
    inline Vector2* get_array_texcoord();
    inline Vector3* get_array_vnormal();
    inline Vector3* get_array_vertex();
    inline size_t get_num_vertex() const;
    inline float get_point_size() const;
    inline void* get_skeleton();
    inline Texture* get_texture();
    inline bool has_cache() const;
    inline bool has_point_sprites() const;
    inline bool has_texture() const;
    inline bool has_skeleton() const;
    inline void set_cache();
    inline void set_num_vertex(size_t);
    inline void set_num_elem(ElementType, MESH_TYPEDEF_INDEX);
    inline void set_texture(Texture*);
};

//############################################################################
// Inline ####################################################################
//############################################################################

/** Return the color array.
 * @return Pointer to the beginning of the color array.
 */
Color4* Mesh::get_array_color()
{
  return color;
}

/** Return the texture coordinate array.
 * @return Pointer to the beginning of the texture coordinate array.
 */
Vector2* Mesh::get_array_texcoord()
{
  return this->texcoord;
}

/** Return the normal array.
 * @return Pointer to the beginning of the normal array.
 */
Vector3* Mesh::get_array_vnormal()
{
  if(has_skeleton())
  {
    return vnormal; // TODO: Return the array from the skeleton
  }
  return vnormal;
}

/** Return the vertex array.
 * @return Pointer to the beginning of the vertex array.
 */
Vector3* Mesh::get_array_vertex()
{
  if(has_skeleton())
  {
    return vertex; // TODO: Return the array from the skeleton
  }
  return vertex;
}

#ifdef LIBFHI_OPENGL
/** Get the display list used by this.
 * @return Display list index or 0 if none.
 */
inline GLuint Mesh::get_display_list() const
{
  return this->display_list;
}
#endif

/** Get the number of vertices.
 * @return Vertex count as integer.
 */
size_t Mesh::get_num_vertex() const
{
  return num_vertex;
}

/** Tell if this mesh has cache attribute.
 * @return True if yes, false if no.
 */
bool Mesh::has_cache() const
{
  return this->has_attr(ATTR_ALLOW_CACHE);
}

/** Tell if this mesh has point sprites.
 * @return True if yes, false if no.
 */
bool Mesh::has_point_sprites() const
{
  return this->has_attr(ATTR_POINT_SPRITES);
}

/** Tell if this mesh has a texture.
 * @return True if yes, false if no.
 */
bool Mesh::has_texture() const
{
  return (this->texture != NULL);
}

/** Tell if this mesh has a skeleton.
 * @return True if yes, false if no.
 */
bool Mesh::has_skeleton() const
{
  return (this->skel != NULL);
}

/** Return the array of elements as specified.
 * @param op Element type.
 * @return The array containing all indexes of this type.
 */
MESH_TYPEDEF_INDEX* Mesh::get_array_elem(ElementType op)
{
  return this->elem[op];
}

/** Get the point size of this mesh.
 * @return Point size as float.
 */
float Mesh::get_point_size() const
{
  return this->point_size;
}

/** Return the skeleton associated with this mesh.
 * @return A pointer to the skeleton.
 */
void* Mesh::get_skeleton()
{
  return skel;
}

/** Get the current texture of this.
 * @return Texture pointer.
 */
Texture* Mesh::get_texture()
{
  return this->texture;
}

/** Turns cache attribute on. Should only be called by programs and only if it
 * is ensured that the mesh will not be modified.
 */
void Mesh::set_cache()
{
  this->attr_or(ATTR_ALLOW_CACHE);
}

/** Set the number of vertices. DANGEROUS.
 * @param cnt New vertex count.
 */
void Mesh::set_num_vertex(size_t cnt)
{
  this->num_vertex = cnt;
}

/** Set the number of elements of given type, DANGEROUS.
 * @param type Element type.
 * @param cnt New count.
 */
void Mesh::set_num_elem(ElementType type, MESH_TYPEDEF_INDEX cnt)
{
  *(this->get_array_elem(type)) = cnt;
}

/** Set the texture of this.
 * @param op New texture pointer.
 */
void Mesh::set_texture(Texture *op)
{
  this->texture = op;
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif

