#ifndef libfhi_light_include
#define libfhi_light_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_color.h"
#include "libfhi_orientation.h"

namespace libfhi {

//############################################################################
// Define ####################################################################
//############################################################################

// Forward declaration.
class PostModel;
class Mesh;

/** One unit of Color - Vector data.
 */
typedef struct
{
  /** Color instance. */
  float col[3];

  /** Vertex instance. */
  Vector3 ver;
} CVUnit;

//############################################################################
// Class #####################################################################
//############################################################################

/** Base light class. Contains the vertual method but nothing else really.
 */
class Light
{
  public:
  protected:
    /** C3F_V3F heap. */
    static CVUnit *cvdata;

    /** Current size of the transformation array. */
    static size_t cvheap;

  public:
    Light();
    virtual ~Light();

    // Static methods.
    static CVUnit* ensure_heap(size_t);

    /** The light is actually the entity that performs transforming the vertex
     * data according to a given matrix.
     * @param cm Camera matrix.
     * @param mdl Model.
     * @param mesh Mesh.
     * @return Array of transformed vectors.
     */
    virtual CVUnit* transform(const Matrix44& cm, PostModel *mdl, Mesh *mesh)
      = 0;

    /** Same as Orientation, needs tick method.
     */
    virtual void tick() = 0;

#ifdef LIBFHI_OPENGL
  public:
    /** The OpenGL version of the transformations method. Unlike the software
     * version, the OpenGL version already has the camera transformation
     * matrix in stack, also it does not need to return a pointer, since it
     * sets color, normal and vertex arrays.
     * @param mdl Model.
     * @param mesh Mesh.
     */
    virtual void transform_gl(PostModel *mdl, Mesh *mesh) = 0;

    /** OpenGL tick method for possible other stuff.
     */
    virtual void tick_gl() = 0;
#endif

};

/** Directional light source a'la OpenGL directional light. This version only
 * supports one light. It is the only base class provided by libfhi, other
 * light sources have to be written by hand. Directional light only has
 * ambient and diffuse color values.
 */
class LightDirectional :
  public Light,
  public Orientation
{
#ifdef LIBFHI_DEBUG
  public:
    DEBUG_PRINT_PROTOTYPE(LightDirectional)
#endif

  public:
    /** Default ambient color for directional light. */
    static const Color4 DEFAULT_AMBIENT;

    /** Default diffuse color for directional light. */
    static const Color4 DEFAULT_DIFFUSE;

    /** Default direction for directional light. */
    static const Vector3 DEFAULT_DV;
    
  private:
    /** Ambient color of this light. */
    Color4 ambient;

    /** Diffuse color of this light. */
    Color4 diffuse;

    /** Transformed ambient color of this light. */
    Color4 mul_ambient;

    /** Transformed diffuse color of this light. */
    Color4 mul_diffuse;

  public:
    LightDirectional();
    LightDirectional(uint16_t, uint16_t, uint16_t);
    LightDirectional(const Vector3&);
    virtual ~LightDirectional();

    // Virtual methods.
    virtual CVUnit* transform(const Matrix44&, PostModel*, Mesh*);
    virtual void tick();
#ifdef LIBFHI_OPENGL
    virtual void transform_gl(PostModel*, Mesh*);
    virtual void tick_gl();
#endif

    // Inline methods.
    inline const Color4& get_ambient() const;
    inline const Color4& get_diffuse() const;
    inline void set_ambient(float, float, float, float = 1.0f);
    inline void set_diffuse(float, float, float, float = 1.0f);
    inline void set_dir(const Vector3& op);
};

//############################################################################
// Inline methods ############################################################
//############################################################################

/** Get the ambient color vector of this light.
 * @return Ambient color as const vector reference.
 */
const Color4& LightDirectional::get_ambient() const
{
  return this->ambient;
}

/** Get the diffuse color vector of this light.
 * @return Diffuse color as const vector reference.
 */
const Color4& LightDirectional::get_diffuse() const
{
  return this->diffuse;
}

/** Set the ambient color.
 * @param r Red color component.
 * @param g Green color component.
 * @param b Blue color component.
 * @param a Alpha color component.
 */
void LightDirectional::set_ambient(float r, float g, float b, float a)
{
  ambient.set(r, g, b, a);
}

/** Set the diffuse color.
 * @param r Red color component.
 * @param g Green color component.
 * @param b Blue color component.
 * @param a Alpha color component.
 */
void LightDirectional::set_diffuse(float r, float g, float b, float a)
{
  diffuse.set(r, g, b, a);
}

/** Set the direction of this light exactly. That means, light is set to look
 * at given direction, then rotation disabled.
 * @param op New direction vector.
 */
void LightDirectional::set_dir(const Vector3& op)
{
  this->om.lookat(op, Vector3(0.0f, 0.0f, 0.0f),
      Vector3(0.0f, 1.0f, 0.0f));
  this->set_rotation(ROTATION_NO);
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif

