#ifndef libfhi_component_include
#define libfhi_component_include

//############################################################################
// Includet ##################################################################
//############################################################################

#include "libfhi_attributable.h"
#include "libfhi_color.h"
#include "libfhi_matvec.h"

#include <vector>

namespace libfhi {

//############################################################################
// Enumit ####################################################################
//############################################################################

/** Enum determining the type of a Component.
 */
enum ComponentType
{
  // Base types.
  COMPONENT_NONE = 0x0,
  COMPONENT_VERTEX = 0x1,
  COMPONENT_EDGE = 0x2,
  COMPONENT_VOXEL = 0x4,
  COMPONENT_LINE = 0x8,
  COMPONENT_TRIANGLE = 0x10,
  COMPONENT_QUAD = 0x20,

  // Masks.
  COMPONENT_FACE = COMPONENT_VOXEL | COMPONENT_LINE | COMPONENT_TRIANGLE |
                   COMPONENT_QUAD
};

//############################################################################
// Luokka ####################################################################
//############################################################################

/** The base class of components.
 */
class Component
{
#ifdef LIBFHI_DEBUG
  public:
    // Virtual print function.
    virtual std::ostream& print(std::ostream&, int) const = 0;

    // These as inline.
    static inline std::ostream& print_spaces(std::ostream&, int);
    inline friend std::ostream& operator<<(std::ostream&, const Component&);
#endif

  protected:
    /** Array of parents. */
    std::vector<Component*> parents;

    /** Position. */
    Vector3 pos;

    /** Index. */
    size_t idx;

    /** Type. */
    ComponentType type;

  public:
    Component(ComponentType = COMPONENT_NONE);
    Component(const Vector3&, ComponentType);
    virtual ~Component();

    // Virtual methods.
    virtual bool has_child(const Component*) const = 0;
    virtual bool has_child(size_t) const = 0;
    virtual void remove_child(Component*) = 0;
    virtual void update() = 0;

    // Inline
    inline void add_parent(Component*);
    inline size_t get_idx() const;
    inline Component* get_last_parent();
    inline Component* get_parent(size_t) const;
    inline const Vector3& get_pos() const;
    inline ComponentType get_type() const;
    inline bool has_idx(size_t) const;
    inline bool has_parents() const;
    inline bool is_face() const;
    inline size_t num_parents() const;
    inline void remove_parent(Component*);
    inline void set_idx(size_t);
    inline void set_pos(const Vector3&);
};

/** ComponentVertex represents one vertex.
 * It has a position and a color.
 */
class ComponentVertex : public Component
{
#ifdef LIBFHI_DEBUG
  public:
    std::ostream& print(std::ostream&, int) const;
#endif

  protected:
    /** The color of this vertex. */
    Color4 col;

    /** Texture coordinates of this vertex. */
    Vector2 texcoord;

    /** The normal associated with this vertex. */
    Vector3 normal;

  public:
    ComponentVertex(const Vector3&, const Color4&, const Vector2&);
    virtual ~ComponentVertex();

    void calculate_normal();

    // Virtuals
    virtual bool has_child(const Component*) const;
    virtual bool has_child(size_t) const;
    virtual void remove_child(Component*);
    virtual void update();

    // Inline
    inline const Color4& get_col() const;
    inline const Vector3& get_nor() const;
    inline const Vector2& get_tex() const;
    inline bool has_alpha() const;
};

/** ComponentEdge represents one edge.
 * It is linked to two vertices.
 */
class ComponentEdge :
  public Component,
  public Attributable
{
#ifdef LIBFHI_DEBUG
  public:
    std::ostream& print(std::ostream&, int) const;
#endif

  public:
    static const uint8_t
      ATTR_FLAT = 	0x01,
      DEFAULT_ATTR =	0x00;

  protected:
    /** Pointers to vertices. */
    ComponentVertex *vertex[2];

  public:
    ComponentEdge(ComponentVertex*, ComponentVertex*);
    virtual ~ComponentEdge();

    void calculate_attributes();

    // Virtuals
    virtual bool has_child(const Component*) const;
    virtual bool has_child(size_t) const;
    virtual void remove_child(Component*);
    virtual void update();

    // Inline
    inline ComponentVertex* get_child(int);
    inline bool has_children(const Component*, const Component*) const;
    inline bool has_flat() const;
    inline ComponentVertex* get_vertex(int);
};

/** ComponentFace represents one face component.
 * It is linked to 1-4 vertices and 1-4 faces.
 */
class ComponentFace :
  public Component,
  public Attributable
{
#ifdef LIBFHI_DEBUG
  public:
    std::ostream& print(std::ostream&, int) const;
#endif

  public:
    static const uint8_t
      ATTR_DRAW =	0x01,
      ATTR_FLAT = 	0x02,
      ATTR_TEXTURED =	0x04,
      ATTR_OFF =	0xFF - ATTR_DRAW,
      DEFAULT_ATTR =	ATTR_DRAW;
    
  protected:
    /** Vertex array. */
    ComponentVertex *vertex[4];

    /** Edge array. */
    ComponentEdge *edge[4];

    /** Normal. */
    Vector3 normal;

  public:
    ComponentFace(ComponentVertex*);
    ComponentFace(ComponentVertex*, ComponentVertex*);
    ComponentFace(ComponentVertex*, ComponentVertex*, ComponentVertex*);
    ComponentFace(ComponentVertex*, ComponentVertex*, ComponentVertex*,
	ComponentVertex*);
    virtual ~ComponentFace();

    void calculate_attributes();
    void calculate_normal();
    bool has_alpha() const;
    void set_edge(ComponentEdge*);
    void set_edge(ComponentEdge*, ComponentEdge*);
    void set_edge(ComponentEdge*, ComponentEdge*, ComponentEdge*);
    void set_edge(ComponentEdge*, ComponentEdge*, ComponentEdge*,
	ComponentEdge*);

    // Virtuals
    virtual bool has_child(const Component*) const;
    virtual bool has_child(size_t) const;
    virtual void remove_child(Component*);
    virtual void update();

    // Inline
    inline const Vector3& get_nor() const;
    inline ComponentVertex* get_vertex(int);
    inline bool has_draw() const;
    inline bool has_flat() const;
    inline bool has_gouraud() const;
    inline bool has_skip() const;
};

//############################################################################
// Component:in inline-funktiot ##############################################
//############################################################################

/** Add one parent to this component.
 * @param op [in] Pointer to parent.
 */
void Component::add_parent(Component* op)
{
  this->parents.push_back(op);
}

/** Remove one parent from this component.
 * @param op [in] Pointer to parent.
 */
void Component::remove_parent(Component *op)
{
  for(std::vector<Component*>::iterator i = this->parents.begin(),
      e = this->parents.end(); (i != e); ++i)
  {
    if(*i == op)
    {
      this->parents.erase(i);
      return;
    }
  }
}

/** Get a parent by index.
 * @param op Parent index.
 * @return Pointer to the parent or NULL if not found.
 */
Component* Component::get_parent(size_t op) const
{
  if(op < parents.size())
  {
    return parents[op];
  }
  return NULL;
}

/** Get the last parent.
 * @return Pointer to the last parent in the list.
 */
Component* Component::get_last_parent()
{
  return this->parents.empty() ? NULL : this->parents.back();
}

/** Get the index of this.
 * @return Index as integer.
 */
size_t Component::get_idx() const
{
  return idx;
}

/** Get the position of this.
 * @return Position as const Vector3 reference.
 */
const Vector3& Component::get_pos() const
{
  return pos;
}

/** Get the type of this component.
 * @return Type enum.
 */
ComponentType Component::get_type() const
{
  return this->type;
}

/** Tell if this component has the given idx.
 * @param op Index to test for.
 * @return True if yes, false if no.
 */
bool Component::has_idx(size_t op) const
{
  return (this->idx == op);
}

/** Tell if this component has parents.
 * @return True if yes, false if no.
 */
bool Component::has_parents() const
{
  return (!this->parents.empty());
}

/** Tells if this component is a face.
 * @return True if yes, false if no.
 */
bool Component::is_face() const
{
  return ((this->type & COMPONENT_FACE) != 0);
}

/** Returns the number of parents on a component.
 * @return The number as an integer.
 */
size_t Component::num_parents() const
{
  return parents.size();
}

/** Set the index of this.
 * @param op [in] The new index.
 */
void Component::set_idx(size_t op)
{
  idx = op;
}

/** Set the position of this.
 * @param op [in] The new position.
 */
void Component::set_pos(const Vector3& op)
{
  pos = op;
}

//############################################################################
// ComponentVertex inline ####################################################
//############################################################################

/** Get the color of this vertex.
 * @return Color as const reference.
 */
const Color4& ComponentVertex::get_col() const
{
  return this->col;
}

/** Get the normal of this vertex.
 * @return Normal as const reference.
 */
const Vector3& ComponentVertex::get_nor() const
{
  return this->normal;
}

/** Get the texture coordinates of this vertex.
 * @return Texture coordinate as const reference.
 */
const Vector2& ComponentVertex::get_tex() const
{
  return this->texcoord;
}

/** Tell if this vertex has an alpha channel (alpha component of color is not
 * equal to or greater than 1.0f).
 * @return True if yes, false if no.
 */
bool ComponentVertex::has_alpha() const
{
  return (this->col.af < 1.0f);
}

//############################################################################
// ComponentEdge inline ######################################################
//############################################################################

/** Return either child of this.
 * @param op Index of the child, zero for first, nonzero for second.
 * @return Pointer to the first or second child.
 */
ComponentVertex* ComponentEdge::get_child(int op)
{
  return (op == 0) ? this->vertex[0] : this->vertex[1];
}

/** Tells if the this edge has it's children as the given components
 * (vertices). Assumes index of op1 is smaller than op2. Also assumes storage
 * likewise.
 * @param op1 [in] First component to check.
 * @param op2 [in] Second component to check.
 * @return True if this is the edge between those vertices, false if not.
 */
bool ComponentEdge::has_children(const Component *op1,
    const Component *op2) const
{
  return ((vertex[0] == op1) && (vertex[1] == op2));
}

/** Tell if this face has flat shading.
 * @return True if yes, false if no.
 */
bool ComponentEdge::has_flat() const
{
  return (attributes & ATTR_FLAT);
}

/** Get the vertex by index.
 * @param op [in] Vertex index.
 * @return Pointer to vertex.
 */
ComponentVertex* ComponentEdge::get_vertex(int op)
{
  return vertex[op];
}

//############################################################################
// ComponentFace inline ######################################################
//############################################################################

/** Get the normal of this face.
 * @return Normal as const reference.
 */
const Vector3& ComponentFace::get_nor() const
{
  return this->normal;
}

/** Return the vertex by index.
 * @param op Vertex index.
 * @return The vertex associated by given index or NULL if not found.
 */
ComponentVertex* ComponentFace::get_vertex(int op)
{
  if((op < 0) || (op > 3))
  {
    return NULL;
  }
  return this->vertex[op];
}

/** Tell if this face is drawn.
 * @return True if yes, false if no.
 */
bool ComponentFace::has_draw() const
{
  return this->has_attr(ATTR_DRAW);
}

/** Tell if this face has flat shading.
 * @return True if yes, false if no.
 */
bool ComponentFace::has_flat() const
{
  return this->has_attr(ATTR_FLAT);
}

/** Tell if this face has gouraud shading.
 * @return True if yes, false if no.
 */
bool ComponentFace::has_gouraud() const
{
  return !this->has_attr(ATTR_FLAT);
}

/** Tell if this face is skipped (i.e. not drawn).
 * @return True if yes, false if no.
 */
bool ComponentFace::has_skip() const
{
  return !this->has_attr(ATTR_DRAW);
}

//############################################################################
// Debug #####################################################################
//############################################################################

#ifdef LIBFHI_DEBUG

/** Print a number of spaces to given stream.
 * @param s [in] Stream to output to.
 * @param cnt [in] Number of spaces.
 * @return Modified stream.
 */
std::ostream& Component::print_spaces(std::ostream &s, int cnt)
{
  for(; cnt > 0; --cnt)
    s << " ";
  return s;
}

/** Output a Component to a given stream (friend operator function).
 * @param s [in] Stream to output to.
 * @param op [in] Component.
 * @return Modified stream.
 */
std::ostream& operator<<(std::ostream &s, const Component &op)
{
  return op.print(s, 0);
}

#endif

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif

