#include "libfhi_glsurface.h"
#include "libfhi_camera.h"
#include "libfhi_font.h"
#include "libfhi_light.h"
#include "libfhi_mesh.h"
#include "libfhi_misc.h"
#include "libfhi_postmodel.h"
#include "libfhi_texture.h"

#include <iostream>

#include LIBFHI_GLEW_H

#include "SDL.h"

namespace libfhi {

//############################################################################
// Variables #################################################################
//############################################################################

bool glSurface::opengl_initialized = false;

//############################################################################
// Helper functions ##########################################################
//############################################################################

/** Check all GL extensions we need.
 */
void gl_get_extensions()
{
  GLenum err = glewInit();

  if(GLEW_OK == err)
  {
    if(!glewIsSupported("GL_ARB_point_sprite GL_ARB_point_parameters"))
    {
      std::cout << "Warning: No point sprite support.\n";
    }
  }
  else
  {
    std::cout << "Warning: GLEW init failed: \"" <<glewGetErrorString(err) <<
      "\"\n";
  }
}

//############################################################################
// Construction ##############################################################
//############################################################################

glSurface::glSurface()
{
  null();
}

glSurface::glSurface(int x, int y, int b, uint32_t sdl_flags)
{
  null();
  reserve(x, y, b, sdl_flags);
}

glSurface::~glSurface()
{
  unreserve();
}

void glSurface::null()
{
  Surface::null();
  sdls = NULL;
}

//############################################################################
// Reservation ###############################################################
//############################################################################

/** Create a new GL context to this surface via SDL.
 */
void glSurface::reserve(int x, int y, int b, uint32_t sdl_flags)
{
  // Remove old values
  unreserve();

  // Check for approperiate new values
  w = x;
  h = y;
  if((w <= 0) || (h <= 0))
  {
    std::cerr << "Error: Could not create surface: Invalid size " << w <<
      "x" << h << ".\n";
    unreserve();
  }

  // Set other values
  bpp = b;
  size_pixels = w * h;
  size_bitmap = size_pixels * (bpp / 8);

  // Call the reservation functions
  sdl_flags |= SDL_OPENGL;
  SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
  sdls = SDL_SetVideoMode(w, h, bpp, sdl_flags);

  // Set the boundary, works just like normal surfaces
  set_boundary();

  // Set the default attributes
  glDisable(GL_LIGHTING);
  glDisable(GL_BLEND);
  glCullFace(GL_BACK);
  glDepthFunc(GL_LEQUAL);

  // Tell to the texture that loading is allowed now that OpenGL is on.
  // The method will return true if it is called for the first time, which
  if(!glSurface::opengl_initialized)
  {
    // Call helper function here.
    gl_get_extensions();

    // Mark that initialization is complete.
    glSurface::opengl_initialized = true;
  }

#ifdef LIBFHI_DEBUG
  std::cout << "glSurface desired: ";
  print_sdls_flags(std::cout, sdl_flags);
  std::cout << "\nglSurface acquired: ";
  print_sdls_flags(std::cout, static_cast<SDL_Surface*>(sdls)->flags);
  std::cout << "\n";
#endif

  // Does not have normal blitting operations enabled!
  this->cbuf = NULL;
  this->zbuf = NULL;
}

/** Free the GL surface.
 */
void glSurface::unreserve()
{
  if(sdls != NULL)
  {
    SDL_FreeSurface(static_cast<SDL_Surface*>(sdls));
    sdls = NULL;
    cbuf = NULL;
  }
  Surface::unreserve();
}

//############################################################################
// Inline utility functions ##################################################
//############################################################################

/** Extract the red color component from an integer.
 * @param Floating point value between 0.0f and 1.0f.
 */
static inline float extract_r(uint32_t col)
{
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  return static_cast<float>(col >> 24) / 255.0f;
#else
  return static_cast<float>(col & 0xFF) / 255.0f;
#endif
}

/** Extract the green color component from an integer.
 * @param Floating point value between 0.0f and 1.0f.
 */
static inline float extract_g(uint32_t col)
{
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  return static_cast<float>((col >> 16) & 0xFF) / 255.0f;
#else
  return static_cast<float>((col >> 8) & 0xFF) / 255.0f;
#endif
}

/** Extract the blue color component from an integer.
 * @param Floating point value between 0.0f and 1.0f.
 */
static inline float extract_b(uint32_t col)
{
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  return static_cast<float>((col >> 8) & 0xFF) / 255.0f;
#else
  return static_cast<float>((col >> 16) & 0xFF) / 255.0f;
#endif
}

/** Extract the alpha color component from an integer.
 * @param Floating point value between 0.0f and 1.0f.
 */
static inline float extract_a(uint32_t col)
{
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  return static_cast<float>(col & 0xFF) / 255.0f;
#else
  return static_cast<float>(col >> 24) / 255.0f;
#endif
}

/** OpenGL makecol with alpha.
 * @param r R color component.
 * @param g G color component.
 * @param b B color component.
 * @param a A color component.
 * @return Color in 4ubv compatible format.
 */
static inline int gl_makecol4(int r, int g, int b, int a)
{
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  return (r << 24) | (g << 16) | (b << 8) | a;
#else
  return (a << 24) | (b << 16) | (g << 8) | r;
#endif
}

/** Wrapper for OpenGL makecol.
 * @param r R color component.
 * @param g G color component.
 * @param b B color component.
 * @return Color in 4ubv compatible format.
 */
static inline int gl_makecol3(int r, int g, int b)
{
  return gl_makecol4(r, g, b, 0xFF);
}

/** Feed color from Vector3 (int).
 * @param op Source vector.
 */
static inline void gl_color4i(const Color4& op)
{
  int32_t c = gl_makecol4(static_cast<int>(op.ri), static_cast<int>(op.gi),
      static_cast<int>(op.bi), static_cast<int>(op.ai));
  glColor4ubv(reinterpret_cast<GLubyte*>(&c));
}

/** Feed color from Vector3 (float).
 * @param op Source vector.
 */
static inline void gl_color4f(const Color4& op)
{
  glColor4f(op.rf / 255.0f, op.gf / 255.0f, op.bf / 255.0f, op.af / 255.0f);
}

/** Feed 2d coordinate from Vector3 (int).
 * @param op Source vector.
 */
static inline void gl_vertex2i(const Vector3& op)
{
  glVertex2i(static_cast<int>(op.xi), static_cast<int>(op.yi));
}

/** Feed 2d coordinate from Vector3 (float).
 * @param op Source vector.
 */
static inline void gl_vertex2f(const Vector3& op)
{
  glVertex2f(op.xf, op.yf);
}

/** Feed 3d coordinate from Vector3 (int).
 * @param op Source vector.
 */
static inline void gl_vertex3i(const Vector3& op)
{
  glVertex3i(static_cast<int>(op.xi), static_cast<int>(op.yi),
      static_cast<int>(op.zi));
}

/** Feed 3d coordinate from Vector3 (float).
 * @param op Source vector.
 */
static inline void gl_vertex3f(const Vector3& op)
{
  glVertex3fv(op.get_fv());
}

//############################################################################
// Utility functions #########################################################
//############################################################################

/** Clear this OpenGL surface to given color.
 * @param color Color.
 */
void glSurface::clear(int color)
{
  glClearColor(extract_r(color),
      extract_g(color),
      extract_b(color),
      extract_a(color));

  glClear(GL_COLOR_BUFFER_BIT);
}

/** Clear this OpenGL surface to given depth.
 * @param depth Depth.
 */
void glSurface::clear(uint16_t depth)
{
  // Set clear depth.
  glClearDepth(static_cast<double>(depth) /
      static_cast<double>(Boundary::ZBUFFER_MAX));

  glClear(GL_DEPTH_BUFFER_BIT);
}

/** Virtual function to clear GL drawing buffer
 * @param color RGBA color to clear into.
 * @param depth Clear depth, defaults to max uint16_t.
 */
void glSurface::clear(int color, uint16_t depth)
{
  glClearColor(extract_r(color),
      extract_g(color),
      extract_b(color),
      extract_a(color));

  // Set clear depth
  glClearDepth(static_cast<double>(depth) /
      static_cast<double>(Boundary::ZBUFFER_MAX));

  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

/** Virtual function to flip GL contents to screen.
 */
void glSurface::flip()
{
  SDL_GL_SwapBuffers();
}

//############################################################################
// 3D GL control functions ###################################################
//############################################################################

/** Set the projection matrix for 2d viewing.
 * Acutally, just set the absense of real projection, the methods provided by
 * glOrtho and the like are insufficent, since the OpenGL viewport is mapped
 * from -1,1 to 1,-1 on the screen. Any orthagonal projection functions will
 * produce a matrix that translates the coordinates by the same ratio as it
 * scales them, which will lead to rounding problems. The following is a
 * kludge, but a surprisingly working one, it produces a matrix that maps each
 * coordinate directly to the same pixel on the screen.
 */
void glSurface::select_2d()
{
  double mat[16] =
  {
    2.0 / static_cast<double>(boundary.get_width()), 0.0, 0.0, 0.0,
    0.0, -2.0 / static_cast<double>(boundary.get_heigth()), 0.0, 0.0,
    0.0, 0.0, -1.0, 0.0,
    -0.99999, 0.99999, 0.0, 1.0
  };

  // Viewport.
  glViewport(boundary.gl_vpx(), boundary.gl_vpy(this->h),
      boundary.get_width(), boundary.get_heigth());
  // Rules.
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glDisable(GL_COLOR_MATERIAL);
  glDisable(GL_CULL_FACE);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_LIGHTING);
  glDisable(GL_TEXTURE_2D);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
  // Projection.
  glMatrixMode(GL_PROJECTION);
  glLoadMatrixd(mat);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslatef(-boundary.xmin_f, -boundary.ymin_f, 0.0f);
  Surface::select_2d();
}

/** Set the projection matrix for 3d viewing.
 * Note that with GL viewports, it is not suppoerted to place the center of
 * view outside the center of screen (this is just an additional software
 * feature.
 */
void glSurface::select_3d()
{
  /* This is quite delicate, determine the ratio points on the screen. */
  float glunear = boundary.glu_near(),
	glufar = boundary.glu_far(),
	xc = boundary.xcenter,
	yc = boundary.ycenter,
	pers = boundary.perspective,
    	rt = (boundary.xmax_f - xc) / pers * glunear,
	lt = (boundary.xmin_f - xc) / pers * glunear,
	up = (boundary.ymin_f - yc) / pers * glunear,
	dn = (boundary.ymax_f - yc) / pers * glunear;

  // Viewport.
  glViewport(boundary.gl_vpx(), boundary.gl_vpy(this->h),
      boundary.get_width(), boundary.get_heigth());
  // Rules.
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_CULL_FACE);
  glEnable(GL_DEPTH_TEST);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
  // Projection.
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glDepthFunc(GL_LEQUAL);
  glFrustum(lt, rt, up, dn, glunear, glufar);
  glMatrixMode(GL_MODELVIEW);
  Surface::select_3d();
}

/** Set the camera transformation (the only transformation done in GL).
 * @param op The camera to use.
 */
void glSurface::set_camera(Camera *op)
{
  Matrix44 cm;

  Surface::camera = op;
  op->tick();

  /* We are in the modelview already, so we can load the camera
   * transformation straigth off the matrix of the orientation. */
  cm.view_matrix(op->get_om());
  glLoadMatrixf(cm.get_glfv());
}

/** Sets the light to the OpenGL surface.
 * @param op The light to use.
 */
void glSurface::set_light(Light *op)
{
  Surface::light = op;
  op->tick_gl();
}

/** Selects this OpenGL surface as the drawing context.
 * Note that the surface still needs to be selected for 2d or 3d drawing
 * before anything can really be done.
 */
void glSurface::select()
{
  // Color creators.
  makecol3 = gl_makecol3;
  makecol4 = gl_makecol4;

  // Not clipped (although OpenGL actually clips them).
  nc_setpixel = gl_setpixel;
  nc_hline = gl_hline;
  nc_vline = gl_vline;
  nc_line = gl_line;
  nc_rect = gl_rect;
  nc_rect_contour = gl_rect_contour;
  nc_triangle_flat = gl_nc_triangle_flat;
  nc_triangle_gouraud = gl_nc_triangle_gouraud;
  nc_poly_flat = gl_nc_poly_flat;
  nc_poly_gouraud = gl_nc_poly_gouraud;
  nc_zbuf_spot = gl_nc_zbuf_spot;
  nc_zbuf_line_flat = gl_nc_zbuf_line_flat;
  nc_zbuf_line_gouraud = gl_nc_zbuf_line_gouraud;
  nc_zbuf_poly_flat = gl_nc_zbuf_poly_flat;
  nc_zbuf_poly_gouraud = gl_nc_zbuf_poly_gouraud;

  // Clipped.
  setpixel = gl_setpixel;
  hline = gl_hline;
  vline = gl_vline;
  line = gl_line;
  rect = gl_rect;
  rect_contour = gl_rect_contour;
  poly_flat = gl_poly_flat;
  poly_gouraud = gl_poly_gouraud;
  zbuf_line_flat = gl_zbuf_line_flat;
  zbuf_line_gouraud = gl_zbuf_line_gouraud;
  zbuf_poly_flat = gl_zbuf_poly_flat;
  zbuf_poly_gouraud = gl_zbuf_poly_gouraud;

  // Dummies (set to NULL so if something uses them, we know it from
  // segfault).
  draw_glyph = NULL;

  // High-level.
  draw_text = gl_draw_text;
  draw_model = gl_draw_model;
  draw_texture = gl_draw_texture;

  // GL surface takes these differently.
  bound = &boundary;
  cdata = NULL;
  zdata = NULL;
  xmul = w;
}

//############################################################################
// GL-variations of primitive drawing methods ################################
//############################################################################

/** OpenGL setpizel.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @param col Color.
 */
void glSurface::gl_setpixel(int x, int y, int col)
{
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_POINTS);
  glVertex2i(x, y);
  glEnd();
}

/** OpenGL horizontal line.
 * @param x1 First x coordinate.
 * @param x2 Second x coordinate.
 * @param y Y coordinate.
 * @param col Color.
 */
void glSurface::gl_hline(int x1, int x2, int y, int col)
{
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_LINES);
  glVertex2i(x1, y);
  glVertex2i(x2, y);
  glEnd();
}

/** OpenGL vertical line.
 * @param y1 First y coordinate.
 * @param y2 Second y coordinate.
 * @param x X coordinate.
 * @param col Color.
 */
void glSurface::gl_vline(int y1, int y2, int x, int col)
{
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_LINES);
  glVertex2i(x, y1);
  glVertex2i(x, y2);
  glEnd();
}

/** OpenGL line.
 * @param x1 First x coordinate.
 * @param y1 First y coordinate.
 * @param x2 Second x coordinate.
 * @param y2 Second y coordinate.
 * @param col Color.
 */
void glSurface::gl_line(int x1, int y1, int x2, int y2, int col)
{
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_LINES);
  glVertex2i(x1, y1);
  glVertex2i(x2, y2);
  glEnd();
}

/** OpenGL rectangle.
 * @param x1 First x coordinate.
 * @param y1 First y coordinate.
 * @param x2 Second x coordinate.
 * @param y2 Second y coordinate.
 * @param col Color.
 */
void glSurface::gl_rect(int x1, int y1, int x2, int y2, int col)
{
  /* NOTE: OpenGL renders the rectangles in such a manner that the quad is
   * only rendered up to, not including the latter coordinates. This is
   * obvious, since rendering in excess would just be harmful when polygons
   * need not overlap. However, to get pixel-perfect representation, a
   * kludge is neccessary. Sorry. */
  if(x2 < x1)
  {
    int temp = x1 + 1;
    x1 = x2;
    x2 = temp;
  }
  else
  {
    x2++;
  }
  if(y2 < y1)
  {
    int temp = y1 + 1;
    y1 = y2;
    y2 = temp;
  }
  else
  {
    y2++;
  }
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glRecti(x1, y1, x2, y2);
}

/** OpenGL contour rectangle.
 * @param x1 First x coordinate.
 * @param y1 First y coordinate.
 * @param x2 Second x coordinate.
 * @param y2 Second y coordinate.
 * @param col Color.
 */
void glSurface::gl_rect_contour(int x1, int y1, int x2, int y2, int col)
{
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_LINE_LOOP);
  glVertex2i(x1, y1);
  glVertex2i(x1, y2);
  glVertex2i(x2, y2);
  glVertex2i(x2, y1);
  glEnd();
}

/** OpenGL 2d flat triangle from non-clipped data.
 * @param col Color.
 */
void glSurface::gl_nc_triangle_flat(int col)
{
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_TRIANGLES);

  // Loop
  Point *pnt = Surface::first;
  int cnt = 3;
  do {
    gl_vertex2i(pnt->get_pos());
    pnt = pnt->get_next();
  } while(--cnt);

  glEnd();
}

/** OpenGL 2d flat triangle from clipped data.
 * @param col Color.
 */
void glSurface::gl_triangle_flat(int col)
{
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_TRIANGLES);

  // Loop
  Point *pnt = Surface::first;
  int cnt = 3;
  do {
    gl_vertex2f(pnt->get_pos());
    pnt = pnt->get_next();
  } while(--cnt);

  glEnd();
}

/** OpenGL 2d gouraud triangle from non-clipped data.
 */
void glSurface::gl_nc_triangle_gouraud()
{
  glBegin(GL_TRIANGLES);

  // Loop
  Point *pnt = Surface::first;
  int cnt = 3;
  do {
    gl_color4i(pnt->get_col());
    gl_vertex2i(pnt->get_pos());
    pnt = pnt->get_next();
  } while(--cnt);

  glEnd();
}

/** OpenGL 2d gouraud triangle from clipped data.
 */
void glSurface::gl_triangle_gouraud()
{
  glBegin(GL_TRIANGLES);

  // Loop
  Point *pnt = Surface::first;
  int cnt = 3;
  do {
    gl_color4f(pnt->get_col());
    gl_vertex2f(pnt->get_pos());
    pnt = pnt->get_next();
  } while(--cnt);

  glEnd();
}

/** OpenGL 2d flat polygon from non-clipped data.
 * @param col Color.
 */
void glSurface::gl_nc_poly_flat(int col)
{
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_POLYGON);

  // Loop.
  Point *pnt = Surface::first;
  do {
    gl_vertex2i(pnt->get_pos());
    pnt = pnt->get_next();
  } while(pnt != Surface::first);

  glEnd();
}

/** OpenGL 2d flat polygon from clipped data.
 * @param col Color.
 */
void glSurface::gl_poly_flat(int col)
{
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_POLYGON);

  // Loop.
  Point *pnt = Surface::first;
  do {
    gl_vertex2f(pnt->get_pos());
    pnt = pnt->get_next();
  } while(pnt != Surface::first);

  glEnd();
}

/** OpenGL 2d gouraud polygon from non-clipped data.
 */
void glSurface::gl_nc_poly_gouraud()
{
  glBegin(GL_POLYGON);

  // Loop.
  Point *pnt = Surface::first;
  do {
    gl_color4i(pnt->get_col());
    gl_vertex2i(pnt->get_pos());
    pnt = pnt->get_next();
  } while(pnt != Surface::first);

  glEnd();
}

/** OpenGL 2d gouraud polygon from clipped data.
 */
void glSurface::gl_poly_gouraud()
{
  glBegin(GL_POLYGON);

  // Loop.
  Point *pnt = Surface::first;
  do {
    gl_color4f(pnt->get_col());
    gl_vertex2f(pnt->get_pos());
    pnt = pnt->get_next();
  } while(pnt != Surface::first);

  glEnd();
}

/** OpenGL 3d spot from non-clipped data.
 * @param col Color.
 */
void glSurface::gl_nc_zbuf_spot(int col)
{
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_POINTS);
  gl_vertex3i(Surface::first->get_pos());
  glEnd();
}

/** OpenGL 3d flat line from non-clipped data.
 * @param col Color.
 */
void glSurface::gl_nc_zbuf_line_flat(int col)
{
  Point *p1 = pointdata + 0,
	*p2 = pointdata + 1;

  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_LINE);
  gl_vertex3i(p1->get_pos());
  gl_vertex3i(p2->get_pos());
  glEnd();
}

/** OpenGL 3d flat line from clipped data.
 * @param col Color.
 */
void glSurface::gl_zbuf_line_flat(int col)
{
  Point *p1 = pointdata + 0,
	*p2 = pointdata + 1;

  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  glBegin(GL_LINE);
  gl_vertex3f(p1->get_pos());
  gl_vertex3f(p2->get_pos());
  glEnd();
}

/** OpenGL 3d gouraud line from non-clipped data.
 */
void glSurface::gl_nc_zbuf_line_gouraud()
{
  Point *p1 = pointdata + 0,
	*p2 = pointdata + 1;

  glBegin(GL_LINE);
  gl_color4i(p1->get_col());
  gl_vertex3i(p1->get_pos());
  gl_color4i(p2->get_col());
  gl_vertex3i(p2->get_pos());
  glEnd();
}

/** OpenGL 3d gouraud line from clipped data.
 */
void glSurface::gl_zbuf_line_gouraud()
{
  Point *p1 = pointdata + 0,
	*p2 = pointdata + 1;

  glBegin(GL_LINE);
  gl_color4f(p1->get_col());
  gl_vertex3f(p1->get_pos());
  gl_color4f(p2->get_col());
  gl_vertex3f(p2->get_pos());
  glEnd();
}

/** OpenGL 3d flat polygon from non-clipped data.
 * @param col Color.
 */
void glSurface::gl_nc_zbuf_poly_flat(int col)
{
  glBegin(GL_POLYGON);
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));

  // Loop
  Point *pnt = Surface::first;
  do {
    gl_vertex3i(pnt->get_pos());
    pnt = pnt->get_next();
  } while(pnt != Surface::first);

  glEnd();
}

/** OpenGL 3d flat polygon from clipped data.
 * @param col Color.
 */
void glSurface::gl_zbuf_poly_flat(int col)
{
  glBegin(GL_POLYGON);
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));

  // Loop
  Point *pnt = Surface::first;
  do {
    gl_vertex3f(pnt->get_pos());
    pnt = pnt->get_next();
  } while(pnt != Surface::first);

  glEnd();
}

/** OpenGL 3d gouraud polygon from clipped data.
 */
void glSurface::gl_nc_zbuf_poly_gouraud()
{
  glBegin(GL_POLYGON);

  // Loop
  Point *pnt = Surface::first;
  do {
    gl_color4i(pnt->get_col());
    gl_vertex3i(pnt->get_pos());
    pnt = pnt->get_next();
  } while(pnt != Surface::first);

  glEnd();
}

/** OpenGL 3d gouraud polygon from clipped data.
 */
void glSurface::gl_zbuf_poly_gouraud()
{
  glBegin(GL_POLYGON);

  // Loop
  Point *pnt = Surface::first;
  do {
    gl_color4f(pnt->get_col());
    gl_vertex3f(pnt->get_pos());
    pnt = pnt->get_next();
  } while(pnt != Surface::first);

  glEnd();
}

//############################################################################
// Helper functions ##########################################################
//############################################################################

/** Draw one Glyph in OpenGL.
 * @param x X position of lower left coordinate.
 * @param y Y position of lower left coordinate.
 * @param glyph Glyph structure to draw.
 */
inline void gl_draw_glyph(int x1, int y1, Glyph *glyph)
{
  GLuint texture = glyph->get_texture();
  int x2 = x1 + glyph->get_w(),
      y2 = y1 + glyph->get_h();

  // If 0, generate new.
  if(texture == 0)
  {
    texture = glyph->generate_texture();
  }

  // Bind texture.
  glBindTexture(GL_TEXTURE_2D, texture);

  // Draw one quad (containing the bitmap).
  glBegin(GL_QUADS);
  glTexCoord2f(0.0f, 0.0f);
  glVertex2i(x1, y1);
  glTexCoord2f(0.0f, glyph->get_tmy());
  glVertex2i(x1, y2);
  glTexCoord2f(glyph->get_tmx(), glyph->get_tmy());
  glVertex2i(x2, y2);
  glTexCoord2f(glyph->get_tmx(), 0.0f);
  glVertex2i(x2, y1);
  glEnd();
}

//############################################################################
// High-level drawing ########################################################
//############################################################################

/** Inline helper function to execute mesh call code.
 * @param mesh Mesh to execute.
 */
inline void gl_mesh_execute_drawelements(Mesh *mesh)
{
  MESH_TYPEDEF_INDEX *elem;

  // Point sprites are a distinct thingy.
  if(mesh->has_point_sprites())
  {
    elem = mesh->get_array_elem(MESH_POINTS);

    // If there is anything to draw... this will of course crash on null
    // pointers, but please do not use particle systems with null pointers,
    // please.
    if(*elem)
    {
      // Current use of point sprites is limited to linear attenuation with no
      // fade. Specify point parameters here.
      static float paramfv[3] =
      {
	1.0f,
	0.0f,
	0.0f
      };
      float psize = mesh->get_point_size();

      glEnable(GL_POINT_SPRITE_ARB);

      glPointParameterfvARB(GL_POINT_DISTANCE_ATTENUATION_ARB, paramfv);
      glTexEnvf(GL_POINT_SPRITE_ARB, GL_COORD_REPLACE_ARB, GL_TRUE);

      //glPointParameterfARB(GL_POINT_SIZE_MIN_ARB, psize);
      //glPointParameterfARB(GL_POINT_SIZE_MAX_ARB, psize);
      glPointSize(psize);

      glBlendFunc(GL_SRC_ALPHA, GL_ONE);

      glDrawElements(GL_POINTS, *elem, GL_UNSIGNED_INT, elem + 1);

      glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

      glDisable(GL_POINT_SPRITE_ARB);
    }
  }
  // If not a point sprite mesh, iterate through different element types and
  // execute the correct one.
  else
  {
    if((elem = mesh->get_array_elem(MESH_POINTS)) != NULL)
    {
      glDrawElements(GL_POINTS, *elem, GL_UNSIGNED_INT, elem + 1);
    }
    if((elem = mesh->get_array_elem(MESH_LINES)) != NULL)
    {
      glDrawElements(GL_LINES, *elem, GL_UNSIGNED_INT, elem + 1);
    }
    if((elem = mesh->get_array_elem(MESH_FLAT_LINES)) != NULL)
    {
      glDrawElements(GL_LINES, *elem, GL_UNSIGNED_INT, elem + 1);
    }
    if((elem = mesh->get_array_elem(MESH_TRIANGLES)) != NULL)
    {
      glDrawElements(GL_TRIANGLES, *elem, GL_UNSIGNED_INT, elem + 1);
    }
    if((elem = mesh->get_array_elem(MESH_FLAT_TRIANGLES)) != NULL)
    {
      glDrawElements(GL_TRIANGLES, *elem, GL_UNSIGNED_INT, elem + 1);
    }
    if((elem = mesh->get_array_elem(MESH_QUADS)) != NULL)
    {
      glDrawElements(GL_QUADS, *elem, GL_UNSIGNED_INT, elem + 1);
    }
    if((elem = mesh->get_array_elem(MESH_FLAT_QUADS)) != NULL)
    {
      glDrawElements(GL_QUADS, *elem, GL_UNSIGNED_INT, elem + 1);
    }
  }
}

/** OpenGL textout.
 * @param x The left x coordinate.
 * @param y The lower y coordinate of the first line.
 * @param col Color to draw with.
 * @param font Font pointer.
 * @param text UTF-8 text string.
 * @return The width of outputted text in pixels.
 */
int glSurface::gl_draw_text(int x, int y, int col, Font *font,
    const char *text)
{
  char *str = const_cast<char*>(text);
  int cx = x,
      cy = y,
      ret = 0;

  // Enable texturing for the duration of this method.
  glEnable(GL_TEXTURE_2D);
  glColor4ubv(reinterpret_cast<GLubyte*>(&col));
  
  while(*str)
  {
    int unicode = strdecodeutf8(str);
    Glyph *glyph = font->get_glyph(unicode);

    if(!glyph)
    {
      continue; // Erraneous loads are skipped (no garble, just nothing).
    }

    gl_draw_glyph(cx + glyph->get_left(), cy - glyph->get_top(), glyph);

    int advance = glyph->get_advance_x();

    cx += advance;
    ret += advance;
  }

  // Revert to old state.
  glDisable(GL_TEXTURE_2D);

  return ret;
}

/** Draw a model onto the screen. Hardware renderer version.
 * @param mdl Model to draw.
 */
void glSurface::gl_draw_model(PostModel *mdl)
{
  Mesh *mesh;

  if(mdl == NULL)
  {
    return;
  }
  mesh = mdl->get_mesh();

  // Check for immediate exit.
  if((mesh == NULL) || (mesh->get_num_vertex() == 0) || (!mdl->has_draw()))
  {
    return;
  }

  // Call the OpenGL transformation stuff from light.
  glPushMatrix();
  light->transform_gl(mdl, mesh);

  // If the thing has cache allowed.
  if(mesh->has_cache())
  {
    GLuint list = mesh->get_display_list();

    if(list <= 0)
    {
      list = mesh->reserve_display_list();
      glNewList(list, GL_COMPILE_AND_EXECUTE);
      gl_mesh_execute_drawelements(mesh);
      glEndList();
    }
    else
    {
      glCallList(list);
    }
  }
  // Otherwise execute as normal.
  else
  {
    gl_mesh_execute_drawelements(mesh);
  }

  // Revert to old transformations.
  glPopMatrix();
}

/** OpenGL textured rectangle
 * @param x1 First x coordinate.
 * @param y1 First y coordinate.
 * @param x2 Second x coordinate.
 * @param y2 Second y coordinate.
 * @param col Color.
 */
void glSurface::gl_draw_texture(int x1, int y1, int x2, int y2,
    Texture *tex)
{
  // Uses the same kludge for pixel-perfect rendering as with normal
  // gl_rect, for details, see it.
  if(x2 < x1)
  {
    int temp = x1 + 1;
    x1 = x2;
    x2 = temp;
  }
  else
  {
    x2++;
  }
  if(y2 < y1)
  {
    int temp = y1 + 1;
    y1 = y2;
    y2 = temp;
  }
  else
  {
    y2++;
  }

  // Color is full white for modulate with texture.
  static const int col = 0xFFFFFFFF;
  glColor4ubv(reinterpret_cast<const GLubyte*>(&col));

  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, tex->get_texture_name());

  glBegin(GL_QUADS);
  glTexCoord2f(0.0f, 0.0f);
  glVertex2i(x1, y1);
  glTexCoord2f(0.0f, 1.0f);
  glVertex2i(x1, y2);
  glTexCoord2f(1.0f, 1.0f);
  glVertex2i(x2, y2);
  glTexCoord2f(1.0f, 0.0f);
  glVertex2i(x2, y1);
  glEnd();

  glDisable(GL_TEXTURE_2D);
}

//############################################################################
// Utility functions #########################################################
//############################################################################

/** Lock this surface.
 * Not implemented in glSurface.
 */
void glSurface::lock()
{
  return;
}

/** Unlock this surface.
 * Not implemented in glSurface.
 */
void glSurface::unlock()
{
  return;
}

//############################################################################
// Returner functions ########################################################
//############################################################################

/** Create a new glSurface.
 * @param w Width.
 * @param h Height.
 * @param sdl_flags Additional SDL flags.
 * @return A new glSurface object.
 */
glSurface* glsurface_new(int w, int h, int b, uint32_t sdl_flags)
{
  // Return null if erraneous size
  if((w <= 0) || (h <= 0) || ((b != 16) && (b != 32)))
  {
    return NULL;
  }

  // Otherwise create a new surface (not neccessarily ok)
  return new glSurface(w, h, b, sdl_flags);
}

//############################################################################
// End #######################################################################
//############################################################################

}

