#include "libfhi_configfile.h"

#include <iostream>
#include <sstream>
#include <string.h>

namespace libfhi {

//############################################################################
// Construction ##############################################################
//############################################################################

/** Default constructor.
 * @param str Filename to open.
 */
ConfigFile::ConfigFile(const char *str)
  : filename(str), line(0), filu(NULL)
{
  this->filu = fopen(str, "rt");
}

/** Default destructor.
 */
ConfigFile::~ConfigFile()
{
  // Just close the file. Everything else is bound to delete itself.
  if(this->filu)
  {
    fclose(this->filu);
  }
}

//############################################################################
// Methods ###################################################################
//############################################################################

/** Advance one line in the configuration file saving found tokens int the
 * vectors.
 * @return True if there is anything to advance anymore, false otherwise.
 */
bool ConfigFile::advance()
{
  if((this->filu == NULL) || (feof(this->filu)))
  {
    return false;
  }

  // Define read stuff.
  static const int BUFSIZE = 2048;
  static const char *delims = " \n";
  char input[BUFSIZE];

  // Empty our info.
  this->identifier = "";
  this->vfloat.clear();
  this->vint.clear();
  this->vstring.clear();
  ++line;

  bool first = true,
       lastwascontinue = false;

  do {
    // If we have already continued, remove the last comntinue token.
    if(lastwascontinue)
    {
      this->vfloat.pop_back();
      this->vint.pop_back();
      this->vstring.pop_back();
      lastwascontinue = false;
    }

    // Try to read one line from file. Abort on error.
    if(feof(this->filu) || (fgets(input, BUFSIZE - 1, this->filu) == NULL))
    {
      return false;
    }

    // Tokenize and read.
    for(char *tok = strtok(input, delims);
	(tok != NULL);
	tok = strtok(NULL, delims), first = false)
    {
      lastwascontinue = false;

      // First one is identifier, after that, others follow.
      if(first)
      {
	this->identifier = tok;
      }
      else
      {
	if(strcmp(tok, "\\") == 0)
	{
	  lastwascontinue = true;
	}

	// The float / int values may be initialized to 0 if they're not legal
	// numbers.
	this->vfloat.push_back(static_cast<float>(atof(tok)));
	this->vint.push_back(atoi(tok));
	this->vstring.push_back(std::string(tok));
      }
    }
  } while(lastwascontinue);

  // Reading a line was successful.
  return true;
}

/** Get the combined string consisting of the whole data block.
 * @return new string object.
 */
std::string ConfigFile::get_combined_string()
{
  std::stringstream text;

  // Add all the successive ones.
  for(std::vector<std::string>::iterator i = this->vstring.begin(),
      e = this->vstring.end(); (i != e); ++i)
  {
    if((*i).compare("\\n") == 0)
    {
      text << std::endl;
    }
    else if(i != this->vstring.begin())
    {
      text << " " << (*i);
    }
    else
    {
      text << (*i);
    }
  }

  // Get the returned string.
  return text.str();
}

/** Full version of has_id that checks for given id and if found, checks for
 * given number of arguments and warns if not.
 * @param str Required identifier.
 * @param cnt Required argument count.
 * @return True if correct id and right number of arguments. False otherwise.
 */
bool ConfigFile::has_id_arg(const char *str, size_t cnt) const
{
  if(!this->has_id(str))
  {
    return false;
  }

  if(this->get_num_arg() != cnt)
  {
    std::stringstream sstr;

    sstr << "needs " << static_cast<uint32_t>(cnt) << " argument" <<
      ((cnt > 1) ? "s." : ".");
    this->warn(sstr.str().c_str());

    return false;
  }

  return true;
}

/** Warns with a filename and a line.
 * @param str Freeform warningn string.
 */
void ConfigFile::warn(const char *str) const
{
  std::cerr << "Warning: \"" << this->filename << "\":" << this->line <<
    ":\"" << this->identifier << "\": " << str << "\n";
}

/** Warns about an empty definition.
 */
void ConfigFile::warn_empty()
{
  if((this->identifier.length() > 0) && (this->identifier[0] != '#'))
  {
    std::cerr << "Warning: unidentified line: \"" << this->identifier;

    for(std::vector<std::string>::iterator i = this->vstring.begin(),
	e = this->vstring.end(); (i != e); ++i)
    {
      std::cerr << " " << (*i);
    }

    std::cerr << "\"\n";
  }
}

//############################################################################
// End #######################################################################
//############################################################################

}

