#include "libfhi_misc.h"
#include "libfhi_font.h"
#include "libfhi_premodel.h"
#include "libfhi_mesh.h"
#include "libfhi_texture.h"

// Include SDL if available
#ifdef LIBFHI_SDL
#include "SDL.h"
#endif

namespace libfhi {

//############################################################################
// Initialization functions ##################################################
//############################################################################

/** Initialize libfhi.
 * @return False on failure, otherwise true;
 */
bool init_libfhi()
{
  // Initialize FreeType.
  font_init();

  // If SDL is enabled, initialize it.
#ifdef LIBFHI_SDL
  if(SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO) < 0)
  {
    std::cerr << "Error: Could not init SDL: " << SDL_GetError() << "\n";
    return false;
  }
#endif

  return true;
}

/** Quit libfhi and close the subsystems it has used.
 * @return True if no errors, false on error.
 */
bool quit_libfhi()
{
  // Delete all objects accumulated to canonical databases.
  // Remember to add here all classes utilizing the CanonicalDB.
  Mesh::canon_clear();
  PreModel::canon_clear();
  Texture::canon_clear();

  // Close FreeType (will also clear fonts).
  font_quit();

  // If SDL is enabled, close it.
#ifdef LIBFHI_SDL
  SDL_Quit();
#endif

  return true;
}

//############################################################################
// Filua #####################################################################
//############################################################################

/** Checks if a file exists.
 * @param filename [in] The name of the file.
 * @return True if found, false if not (or unable to open).
 */
bool fexists(const char *filename)
{
  FILE *filu = fopen(filename, "rt");

  if(filu == NULL)
    return false;

  fclose(filu);
  return true;
}

/** Return the length of an open file in bytes.
 * @param filu [in] Open file resource.
 * @return Length of file in bytes.
 */
int fsize(FILE *filu)
{
  int pos, len;
  
  pos = ftell(filu);
  fseek(filu, 0, SEEK_END);
  len = ftell(filu);
  fseek(filu, pos, SEEK_SET);
  return len;
}

/** Return the length of a file in bytes or -1 if the file does not exist.
 * @param filename [in] The name of the file.
 * @return Length of file in bytes.
 */
int fsize(const char *filename)
{
  FILE *filu = fopen(filename, "rb");
  int ret;

  if(filu == NULL)
    return -1;

  fseek(filu, 0, SEEK_END);
  ret = ftell(filu);
  fclose(filu);
  return ret;
}

//############################################################################
// Additional string functions ###############################################
//############################################################################

/** Calculate the length of a string in lines.
 */
int strlenl(const char *s)
{
  char *iter = const_cast<char*>(s),
       *end = const_cast<char*>(s) + strlen(s);
  int ret;

  for(ret = 0; iter != end; ++iter)
    if(*iter == '\n')
      ++ret;
  return ret;
}

/** Decode an UTF-8 sequence from a character string and update it's position
 * to the next character position.
 * @param str [in] The string pointer to read and update.
 * @return Unicode number of the character read.
 */
int strdecodeutf8(char *&str)
{
  int cmp = *str & (128 + 64 + 32 + 16 + 8);

  // One byte.
  if(cmp < 128)
  {
    return *(str++);
  }
  // Two bytes.
  if(cmp < 128 + 64 + 32)
  {
    int ret = ((*str & 31) << 6) | (*(str + 1) & 63);
    str += 2;
    return ret;
  }
  // Three bytes.
  if(cmp < 128 + 64 + 32 + 16)
  {
    int ret = ((*str & 15) << 12) | ((*(str + 1) & 63) << 6) |
      (*(str + 2) & 63);
    str += 3;
    return ret;
  }
  // Four bytes.
  int ret = ((*str & 7) << 18) | ((*(str + 1) & 63) << 12) |
    ((*(str + 2) & 63) << 6) | (*(str + 3) & 63);
  str += 4;
  return ret;
}

//############################################################################
// Stream-funktiot ###########################################################
//############################################################################

#define ASCII_BACKSPACE	static_cast<char>(8)

/** Print delay into ostream (may be handy in some cases).
 * @param op [in] The position of the delay counter.
 * @param str [out] The stream to output into.
 */
std::ostream& ostream_delay(int op, std::ostream& str)
{
  static char delaysym[4] = { '|', '/', '-', '\\' };

  str << delaysym[abs(op) % 4] << ASCII_BACKSPACE;

  return str;
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}

