#ifndef libfhi_orientation_include
#define libfhi_orientation_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_matvec.h"

namespace libfhi {

//############################################################################
// Define ####################################################################
//############################################################################

// Forward declaration
class World;

//############################################################################
// Luokka ####################################################################
//############################################################################

/** The orientation class saves it's orientation data inside one single
 * matrix that may be used to extract the information.
 */
class Orientation
{
  friend class World;

#ifdef LIBFHI_DEBUG
  public:
    DEBUG_PRINT_PROTOTYPE(Orientation)
#endif

  public:
    // Rotaatio-enum
    enum EnumRotationType
    {
      ROTATION_NO,		// Do not rotate (world).
      ROTATION_ABSORB,		// Absorb-style.
      ROTATION_QUAKE,		// Quake-style.
      ROTATION_CUMULATIVE,	// As above, but cumulative.
      ROTATION_RELATIVE,	// As above, but based on former orientation.
      ROTATION_TARGET,		// Point at target.
      ROTATION_ATTACH		// Attached to target with diff vector.
    };

    static const int NORMALIZE_DELAY = 30;
    static const EnumRotationType DEFAULT_ROTATION = ROTATION_QUAKE;

  protected:
    Matrix44 om;
    Vector3 pv, mv, attach;
    Orientation *target;
    int normalizer;
    uint16_t angles[3], da[3];
    EnumRotationType rotation;
      
  public:
    Orientation();
    virtual ~Orientation();

    void copy(Orientation*);
    void normalize_check();
    void null();

    // Virtuals
    virtual void tick();

    // Inline
    inline const Vector3 get_dv() const;
    inline const Matrix44& get_om() const;
    inline const Vector3& get_pv() const;
    inline uint16_t get_xr() const;
    inline uint16_t get_yr() const;
    inline uint16_t get_zr() const;
    inline bool has_rotation() const;
    inline void movement_add(float, float, float);
    inline void rotation_add(int, int, int);
    inline void rotation_set(int, int, int);
    inline void set_angles(short, short, short);
    inline void set_attach(const Vector3&);
    inline void set_matrix(const Matrix44&);
    inline void set_pos(const Vector3&);
    inline void set_pos(float, float, float);
    inline void set_rotation(EnumRotationType);
};

//############################################################################
// Inline methods ############################################################
//############################################################################

/** Get the direction of this.
 * @return Direction as Vector.
 */
const Vector3 Orientation::get_dv() const
{
  return om.get_fw();
}

/** Get the object matrix of this.
 * @return The matrix.
 */
const Matrix44& Orientation::get_om() const
{
  return om;
}

/** Get the position of this.
 * @return Position as Vector.
 */
const Vector3& Orientation::get_pv() const
{
  return pv;
}

/** Get the x rotation value of this.
 * @return X rotation as integer.
 */
uint16_t Orientation::get_xr() const
{
  return this->angles[0];
}

/** Get the y rotation of this.
 * @return Y rotation as integer.
 */
uint16_t Orientation::get_yr() const
{
  return this->angles[1];
}

/** Get the z rotation of this.
 * @return Z rotation as integer.
 */
uint16_t Orientation::get_zr() const
{
  return this->angles[2];
}

/** Tell if this has rotation or if it is static.
 * @return True if rotates, false if static.
 */
bool Orientation::has_rotation() const
{
  return (rotation != ROTATION_NO);
}

/** Add to the movement vector of this.
 * @param x X movement.
 * @param y Y movement.
 * @param z Z movement.
 */
void Orientation::movement_add(float x, float y, float z)
{
  mv.xf += x;
  mv.yf += y;
  mv.zf += z;
}

/** Add to the rotation of this.
 * @param x X rotation.
 * @param y Y rotation.
 * @param z Z rotation.
 */
void Orientation::rotation_add(int x, int y, int z)
{
  da[0] += x;
  da[1] += y;
  da[2] += z;
}

/** Set the exact rotation of this.
 * @param x X rotation.
 * @param y Y rotation.
 * @param z Z rotation.
 */
void Orientation::rotation_set(int x, int y, int z)
{
  da[0] = x;
  da[1] = y;
  da[2] = z;
}

/** Set the current angles of this.
 * @param xa X angle.
 * @param ya Y angle.
 * @param za Z angle.
 */
void Orientation::set_angles(short xa, short ya, short za)
{
  angles[0] = xa;
  angles[1] = ya;
  angles[2] = za;
}

/** Set the attach point (relative to target).
 * @param op The attach point.
 */
void Orientation::set_attach(const Vector3& op)
{
  this->attach = op;
}

/** Replace the matrix of this orientation.
 * @param op New matrix.
 */
void Orientation::set_matrix(const Matrix44& op)
{
  this->om = op;
}

/** Set the position of this.
 * @param op New position.
 */
void Orientation::set_pos(const Vector3& op)
{
  this->pv = op;
}

/** Set the position of this.
 * @param x X component of new position.
 * @param y Y component of new position.
 * @param z Z component of new position.
 */
void Orientation::set_pos(float x, float y, float z)
{
  this->pv.set(x, y, z);
}

/** Set the rotation model of this.
 * @param op New rotation model.
 */
void Orientation::set_rotation(EnumRotationType op)
{
  this->rotation = op;
}

//############################################################################
// Loppu #####################################################################
//############################################################################

}
#endif

