#ifndef libfhi_getopt_include
#define libfhi_getopt_include

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi_default.h"

#include <iostream>
#include <list>
#include <string>

namespace libfhi {

//############################################################################
// Define ####################################################################
//############################################################################

//############################################################################
// Class #####################################################################
//############################################################################

class GetOptOption
{
  private:
    /** Contains all names that can be given to this. */
    std::list<std::string*> names;

    /** Set to 1 if gets an argument. */
    bool argument;

    /** Value this will export if spotted. */
    int value;

  public:
    GetOptOption(const char*, bool, int, const char* = NULL);
    ~GetOptOption();

    void add_name(const char*);
    bool compare(const char*);
    bool compare_in(int);

    // Inline methods.
    inline const char* get_name();
    inline int get_val() const;
    inline bool has_arg() const;
    inline void set_arg(bool);
};

/** GetOpt is the main class that handles getopt options.
 */
class GetOpt
{
  private:
    /** All the different arguments of this. */
    std::list<GetOptOption*> options;

    /** Argc. */
    int argc;

    /** Argv. */
    char **argv;

    /** Argument index. */
    int aidx;

    /** Character index within an argument. */
    int cidx;

    /** Copy argument pointer here. */
    char *argument;

  public:
    GetOpt(int, char**);
    ~GetOpt();

    void add_option(GetOptOption*);
    void add_option(const char*, bool, int, const char* = NULL);
    int get_opt();

    // Inline.
    inline void close();
    inline const char *get_arg() const;
};

//############################################################################
// GetOptOption inline #######################################################
//############################################################################

/** Return the name of the first argument of this.
 * @return Name as const char.
 */
const char* GetOptOption::get_name()
{
  return this->names.front()->c_str();
}

/** Return the value of this.
 * @return Value (unique identifier).
 */
int GetOptOption::get_val() const
{
  return this->value;
}

/** Tell if this needs an argument.
 * @return True if yes, false if no.
 */
bool GetOptOption::has_arg() const
{
  return this->argument;
}

/** Mark the attribute requirements of this.
 * @param arg New argument dependancy.
 */
void GetOptOption::set_arg(bool arg)
{
  this->argument = arg;
}

//############################################################################
// GetOpt inline #############################################################
//############################################################################

/** Stop getting arguments.
 */
void GetOpt::close()
{
  this->aidx = this->argc;
}

/** Return the current argument.
 * @return Argument as const char.
 */
const char* GetOpt::get_arg() const
{
  return this->argument;
}

//############################################################################
// End #######################################################################
//############################################################################

}
#endif

