#include "libfhi_getopt.h"

using namespace libfhi;

//############################################################################
// GetOptOption ##############################################################
//############################################################################

/** Default constructor.
 * @param name1 Name for this.
 * @param arg Argument needed?
 * @param val Export value.
 * @param name2 Voluntary alternative name for this.
 */
GetOptOption::GetOptOption(const char *name1, bool arg, int val,
    const char *name2)
  : argument(arg), value(val)
{
  this->add_name(name1);
  this->add_name(name2);
}

/** Default destructor.
 */
GetOptOption::~GetOptOption()
{
  for(std::list<std::string*>::iterator i = this->names.begin(),
      e = this->names.end(); (i != e); ++i)
  {
    delete *i;
  }
}

/** Add a name to this.
 * @param name New name.
 */
void GetOptOption::add_name(const char *name)
{
  if((name) && (strlen(name) > 0))
  {
    this->names.push_back(new std::string(name));
  }
}

/** Tell if this equals given argument.
 * @param cmp String to compare to.
 */
bool GetOptOption::compare(const char *cmp)
{
  for(std::list<std::string*>::iterator i = this->names.begin(),
      e = this->names.end(); (i != e); ++i)
  {
    if((*i)->length() > 1)
    {
      std::string comb = std::string("--") + (**i);

      if(strcmp(cmp, comb.c_str()) == 0)
      {
	return true;
      }
    }
  }

  return false;
}

/** Tell if this is situated in a larger string.
 * @param shorthand Shorthand character value to look for.
 */
bool GetOptOption::compare_in(int shorthand)
{
  for(std::list<std::string*>::iterator i = this->names.begin(),
      e = this->names.end(); (i != e); ++i)
  {
    if((*i)->length() == 1)
    {
      if((**i)[0] == shorthand)
      {
	return true;
      }
    }
  }

  return false;
}

//############################################################################
// GetOpt ####################################################################
//############################################################################

/** Default constructor.
 * @param c Count.
 * @param v Argument vector.
 */
GetOpt::GetOpt(int c, char **v)
  : argc(c), argv(v), aidx(1), cidx(0)
{
  // Do nothing.
}

/** Default destructor. */
GetOpt::~GetOpt()
{
  for(std::list<GetOptOption*>::iterator i = this->options.begin(),
      e = this->options.end(); (i != e); ++i)
  {
    delete *i;
  }
}

/** Add one option.
 * @param opt Option.
 */
void GetOpt::add_option(GetOptOption *opt)
{
  this->options.push_back(opt);
}

/** Add option by generating a new one.
 * @param name1 Main name.
 * @param arg Needs argument?
 * @param val Option identifier.
 * @param name2 Secondary name (optional)
 */
void GetOpt::add_option(const char *name1, bool arg, int val,
    const char *name2)
{
  this->add_option(new GetOptOption(name1, arg, val, name2));
}

/** Get the next option.
 * @return Option identifier (integer) or 0 if at end or < 0 on error.
 */
int GetOpt::get_opt()
{
  // If at end, inform it.
  if(this->aidx >= this->argc)
  {
    return 0;
  }

  // Get current option.
  char *arg = this->argv[this->aidx];

  // Check if iterating through a short argument list.
  if(this->cidx > 0)
  {
    int len = static_cast<int>(strlen(arg));

    // If it is needlessly short.
    if(len == 1)
    {
      std::cerr << "Error: \"-\" is not valid argument for GetOpt.\n";
    }

    // Search for current shorthand.
    for(std::list<GetOptOption*>::iterator i = this->options.begin(),
	e = this->options.end(); (i != e); ++i)
    {
      if((*i)->compare_in(arg[this->cidx]))
      {
	// Check for used argument while in compound.
	if((*i)->has_arg())
	{
	  if(len > 2)
	  {
	    std::cerr << "Error: Option -" << arg[this->cidx] <<
	      " requiring an argument not used alone.\n";
	    return -1;
	  }
	  
	  this->argument = argv[this->aidx + 1];
	  this->aidx += 2;
	  this->cidx = 0;
	  return (*i)->get_val();
	}

	// Advance to next and fall off if neccessary.
	if(++(this->cidx) >= len)
	{
	  this->aidx++;
	  this->cidx = 0;
	}

	return (*i)->get_val();
      }
    }

    // Not found so error.
    std::cerr << "Warining: Invalid option: -" << arg[this->cidx] << "\n";
    return -1;
  }

  // Check if long option.
  if(strncmp(arg, "--", 2) == 0)
  {
    for(std::list<GetOptOption*>::iterator i = this->options.begin(),
	e = this->options.end(); (i != e); ++i)
    {
      if((*i)->compare(arg))
      {
	if((*i)->has_arg())
	{
	  if(this->aidx + 1 < this->argc)
	  {
	    this->argument = argv[this->aidx + 1];
	    this->aidx += 2;
	    return (*i)->get_val();
	  }
	  else
	  {
	    std::cerr << "Error: option " << (*i)->get_name() <<
	      " requires an argument.\n";
	    return -1;
	  }
	}

	this->aidx++;
	return (*i)->get_val();
      }
    }

    std::cerr << "Warning: Invalid option: " << arg << "\n";
    return -1;
  }
  else if(strncmp(arg, "-", 1) == 0)
  {
    this->cidx = 1;
    return this->get_opt();
  }

  std::cerr << "Warning: Invalid option: " << arg << "\n";
  return -1;
}

//############################################################################
// End #######################################################################
//############################################################################

