#include "libfhi_boundary.h"

using namespace libfhi;

//############################################################################
// Constants #################################################################
//############################################################################

const float Boundary::DEF_CLIP_Z_NEAR = 1.0f;
const float Boundary::DEF_CLIP_Z_FAR = 1000.0f;
const float Boundary::DEF_VIEW_RATIO = 1.0f;

//############################################################################
// Luokan funktiot ###########################################################
//############################################################################

/** Set the boundary.
 * @param x1 The minimum value of x allowed.
 * @param y1 The minimum value of y allowed.
 * @param x2 The maximum value of x allowed.
 * @param y2 The maximum value of y allowed.
 */
void Boundary::set(int x1, int y1, int x2, int y2)
{
  xmin_i = x1;
  xmax_i = x2;
  ymin_i = y1;
  ymax_i = y2;
  xmin_f = static_cast<float>(xmin_i);
  xmax_f = static_cast<float>(xmax_i);
  ymin_f = static_cast<float>(ymin_i);
  ymax_f = static_cast<float>(ymax_i);
}

/** Set the 3d boundary. The near clipping plane should be smaller than the
 * far clipping plane. Perspective shows best when near 1.0f.
 * @param xc Viewpoint center in x.
 * @param yc Viewpoint center in y.
 * @param pers Multiplier of coord / z divisions.
 * @param near Distance to the near clipping plane.
 * @param far Distance to the far clipping plane.
 * @param backclip [in] Back clipping plane.
 */
void Boundary::set(float xc, float yc, float pers, float znear, float zfar)
{
  this->xcenter = xc;
  this->ycenter = yc;
  this->perspective = pers;
  this->zmin = -zfar;
  this->zmax = -znear;

  // Calculate the multiplier, note that this value is always negative.
  this->zmul = static_cast<float>(ZBUFFER_MAX) / (znear - zfar);
}

//############################################################################
// Utiliteetti ###############################################################
//############################################################################

/** Return the y field of view angle to be given to gluPerspective.
 * @return GLU fov value as float.
 */
float Boundary::glu_fov() const
{
  float ydiff = ymax_f - ymin_f;

  // Ratio is the view cone ratio in a relation between full y scale and z
  // multiplier, let's take the angle (in degrees).
  return atan(ydiff / perspective) / static_cast<float>(M_PI) * 180.0f;
}

/** Return the aspect to be given to gluPerspective.
 * @return GLU aspect value as float.
 */
float Boundary::glu_aspect() const
{
  float ydiff = ymax_f - ymin_f,
	xdiff = xmax_f - xmin_f;

  return xdiff / ydiff;
}

//############################################################################
// Debug #####################################################################
//############################################################################

#ifdef LIBFHI_DEBUG

/** Output this into a stream.
 * @param s [in] Output stream.
 * @return The modified stream.
 */
std::ostream& Boundary::print(std::ostream &s) const
{
  return s << "Integer boundary: [" << xmin_i << ", " << ymin_i << "] [" <<
    xmax_i << ", " << ymax_i << "]\nFloat boundary: [" << xmin_f << ", " <<
    ymin_f << "] [" << xmax_f << ", " << ymax_f << "] [" << zmin << ", " <<
    zmax << "]\nCenter: [" << xcenter << ", " << ycenter << "]\nPerspective: "
    << perspective;
}

#endif

//############################################################################
// Loppu #####################################################################
//############################################################################
