#ifndef weapon_h
#define weapon_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "entity.h"
#include "game.h"
#include "munition.h"
#include "particle.h"
#include "terrain.h"

#include <list>

//############################################################################
// Define ####################################################################
//############################################################################

// Forward declaration.
class Sample;

//############################################################################
// Class #####################################################################
//############################################################################

/** Superclass of all weapons.
 */
class Weapon :
  public libfhi::Attributable,
  public libfhi::CanonicalDB<Weapon>
{
  public:
    /** If this is on, the weapon is instant instead of normal. */
    static const uint8_t ATTR_INSTANT = 0x01;

    /** If this is on, the weapon is homing instead of normal. */
    static const uint8_t ATTR_HOMING = 0x02;

    /** Does this leave a trail of fade instead of fading in the end. */
    static const uint8_t ATTR_TRAIL = 0x04;

    /** Can this be absorbed? */
    static const uint8_t ATTR_UNABSORBABLE = 0x08;

    /** Default attributes. */
    static const uint8_t DEFAULT_ATTR = 0x00;

  private:
    /** Slowdown variable for danmaku play. */
    static float slowdown;

  protected:
    /** The id of this weapon. */
    WeaponEnum id;

    /** The collision map is common for all bullets. */
    CollisionMap *cmap;

    /** The mesh of one bullet. */
    libfhi::Mesh *mesh;

    /** The model of all bullets. */
    libfhi::PostModel *model;

    /** All bullets. */
    std::list<Munition*> bullets;

    /** Maximum number of visible bullets. */
    size_t visible_count;

    /** The maximum possible distance from the center of the weapon. */
    float maxdist;

    /** Name of this weapon. */
    std::string name;

    /** Sample to generate upon firing. */
    Sample *sample_fire;

    /** Sound to generate on hit. */
    Sample *sample_hit;

    /** Sound to generate upon fade. */
    Sample *sample_fade;

    /** Sparks to generate upon hit. */
    Effect *hit;

    /** Sparks to create on fade. */
    Effect *fade;

    /** Sparks to generate on charge. */
    Effect *charge;

    /** Explosion radius. */
    float explosion_radius;

    /** Boolean value telling whether to use ranged explosions. */
    bool use_ranged_explosions;

    /** Damage done by this weapon. */
    int damage;

    /** Lifetime of bullets spawned by this weapon. */
    int lifetime;

    union
    {
      /** Muzzle speed of bullets. */
      float speed;

      /** Length of beam weapons. */
      float length;
    };

    /** Thurst (back) for homing weapons. */
    float thurst;

    /** Rotation speed of this for homing weapons. */
    int rotspeed;

    /** Drag coefficient for homing weapons. */
    float drag_v;

    /** Drag coefficient for homing weapons. */
    float drag_h;

    /** Mass for homing weapons. */
    float mass;

    /** How much can an absorbing entity hold a weapon of this type. */
    int absorb_cap;

  public:
    Weapon();
    Weapon(const char*);
    ~Weapon();

    // Methods.
  public:
    void add(int, const libfhi::Vector2&, const libfhi::Vector2&,
	float, float, int, int);
    void clear_munitions();
    void draw(const libfhi::Vector2&);
    void init(const char*);
    void tick();

    // Private stuff.
  private:
    void draw_bullet(const libfhi::Vector2&);
    void explode(const libfhi::Vector2&, const libfhi::Vector2&, int);
    void tick_bullet();
    void tick_instant();

    // Inline methods.
  public:
    inline int get_absorb() const;
    inline Effect* Weapon::get_charge();
    inline size_t get_count() const;
    inline int get_damage() const;
    inline int get_lifetime() const;
    inline const char* get_name() const;
    inline float get_speed() const;
    inline bool has_homing() const;
    inline bool has_instant() const;
    inline bool has_trail() const;
    inline bool has_unabsorbable() const;
    inline void set_type(WeaponEnum);

    // Static inline methods.
  public:
    static inline float get_slowdown();
    static inline void set_slowdown(float);
};

//############################################################################
// Weapon inline #############################################################
//############################################################################

/** Get the number this weapon can be absorbed.
 * @return Count as an integer.
 */
int Weapon::get_absorb() const
{
  return this->absorb_cap;
}

/** Get the charge particle of this.
 * @return Pointer to charge particle.
 */
Effect* Weapon::get_charge()
{
  return this->charge;
}

/** Get the number of weapons.
 * @return Count as an integer.
 */
size_t Weapon::get_count() const
{
  return this->bullets.size();
}

/** Get damage of this.
 * @return Damage as integer.
 */
int Weapon::get_damage() const
{
  return this->damage;
}

/** Get lifetime of this.
 * @return Lifetime as integer.
 */
int Weapon::get_lifetime() const
{
  return this->lifetime;
}

/** Get the name of this.
 * @return Name as C string.
 */
const char* Weapon::get_name() const
{
  return this->name.c_str();
}

/** Get the muzzle speed of bullets generated by this.
 * @return Speed as float.
 */
float Weapon::get_speed() const
{
  return this->speed * Weapon::slowdown;
}

/** Tell if this has instant.
 * @return True if yes, false if no.
 */
bool Weapon::has_homing() const
{
  return this->has_attr(ATTR_HOMING);
}

/** Tell if this has instant.
 * @return True if yes, false if no.
 */
bool Weapon::has_instant() const
{
  return this->has_attr(ATTR_INSTANT);
}

/** Tell if this has instant.
 * @return True if yes, false if no.
 */
bool Weapon::has_trail() const
{
  return this->has_attr(ATTR_TRAIL);
}

/** Tell if this cannot be absorbed
 * @return True if yes, false if no.
 */
bool Weapon::has_unabsorbable() const
{
  return this->has_attr(ATTR_UNABSORBABLE);
}

/** Set the type of this.
 * @param op New id.
 */
void Weapon::set_type(WeaponEnum op)
{
  this->id = op;
}

//############################################################################
// Weapon static inline ######################################################
//############################################################################

/** Get the current bullet slowdown.
 * @return Slowdown as float.
 */
float Weapon::get_slowdown()
{
  return Weapon::slowdown;
}

/** Set the bullet slowdown.
 * @param op New slowdown.
 */
void Weapon::set_slowdown(float op)
{
  Weapon::slowdown = op;
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

