#ifndef terrain_h
#define terrain_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "constants.h"
#include "collisionmap.h"
#include "game.h"

//############################################################################
// Class #####################################################################
//############################################################################

/** Every square has 8 neighbors, x / y -coordinates and an array containing
 * all the objects to be put inside it.
 *
 * Every NTree square has eight neighbors, that are indexed in this way:
 *
 * ^
 * |
 * y
 * +-+-+-+
 * |5|6|7|
 * +-+-+-+
 * |3| |4|
 * +-+-+-+
 * |0|1|2|
 * +-+-+-+x->
 *
 * Thus, relative distance to neighbor 0 is -SQUARESIZE/2, -SQUARESIZE/2 etc.
 *
 * The constructor of CollisionSquare is empty and only initializes the data,
 * actual initialization is done in set_position (because Collision squares
 * are reserved in a chunk and cannot be passed individual parameters).
 */
class CollisionSquare
{
  private:
    /** All the neighbors of this square (always 8). */
    CollisionSquare *neighbors[8];

    /** All the collision maps here. */
    std::vector<CollisionMap*> maps;

    /** The terrain collision map of this. */
    CollisionMap *terrain_map;

    /** Basically for debugging, the x coord of this square. */
    int pos_x;

    /** Basically for debugging, the y coord of this square. */
    int pos_y;

    /** For translation. The absolute coordinates. */
    libfhi::Vector2 trans;

    /** Last access frame. */
    int last_access_frame;

    /** For ai searches, this is the current heuristic value. */
    float ai_heuristic;

    /** For ai searches, this is the previous search node where we came. */
    CollisionSquare *ai_previous;

    /** The AI search id. Used for checking if square has been traversed. */
    int ai_search_id;

    /** Boolean telling if this is AI llowes square. */
    bool ai_allowed;

  public:
    CollisionSquare();
    ~CollisionSquare();

  public:
    CollisionMap* collides_single_dot(CollisionMap*);
    CollisionMap* collides_single_line(CollisionMap*);
    CollisionMap* collides_freeform(CollisionMap*);
    CollisionMap::CollisionLine* collides_terrain_single_dot(CollisionMap*);
    CollisionMap::CollisionLine* collides_terrain_single_line(CollisionMap*);
    CollisionMap::CollisionLine* collides_terrain_freeform(CollisionMap*);
    bool has_nearby_entities(const Entity*) const;
    void insert(CollisionMap*);
    void set_position(int, int, Terrain*);
    void remove(CollisionMap*);

    // Inline methods.
  private:
    inline bool is_free_for(int, const EntityBase*) const;

    // Public inline methods.
  public:
    inline void check_frame_clear();
    inline void clear_maps();
    inline float get_ai_heuristic() const;
    inline CollisionSquare* get_ai_previous();
    inline std::vector<CollisionMap*>* get_maps();
    inline CollisionSquare* get_neighbor(int);
    inline const libfhi::Vector2& get_trans() const;
    inline bool include_in_search(int);
    inline bool is_safe_to_insert();
    inline bool is_safe_to_travel(const EntityBase*) const;
    inline void set_ai_heuristic(float);
    inline void set_ai_previous(CollisionSquare*);

  private:
    inline CollisionMap* CollisionSquare::collides_meta(
	bool (*)(CollisionMap*, const libfhi::Vector2&, CollisionMap*),
	CollisionMap *cmap);
    inline CollisionMap::CollisionLine*
      CollisionSquare::collides_terrain_meta(
	  CollisionMap::CollisionLine* (*)(CollisionMap*, CollisionMap*),
	CollisionMap *cmap);
};

/** One visual subnode of terrain. Only contains the 3d model used to
 * represent it.
 *
 * VisualSquare has empty constructor for the same reasons as CollisionSquare.
 */
class VisualSquare
{
  private:
    /** Square x coordinate in subsquare space. */
    int pos_x;

    /** Square y coordinate in subsquare space. */
    int pos_y;

    /** Model representing terrain. */
    libfhi::PostModel *model;

    /** Mesh belonging to the given model. */
    libfhi::Mesh *mesh;

  public:
    VisualSquare();
    ~VisualSquare();

  public:
    void set_position(int, int, Terrain*);

    // Inline methods.
  public:
    inline void draw(float, float);
};

/** Parallax scroll is one parallax-scrolling background at given depth in the
 * playfield.
 */
class ParallaxScroll
{
  private:
    /** X scale of the scroll. */
    float scale_x;

    /** Y scale of the scroll. */
    float scale_y;

    /** Z position of the scroll. */
    float zpos;

    /** Texture filename. */
    std::string texture_file;

    /** Mesh. */
    libfhi::Mesh *mesh;

    /** Model. */
    libfhi::PostModel *model;

    /** Before or after. */
    bool phase;

  public:
    ParallaxScroll();
    ~ParallaxScroll();

    // Methods.
  public:
    void draw(const libfhi::Vector2&, bool);
    bool init(float, float, float, bool, const char*, Terrain*);
};

/** Terrain represents the map, collision data and the scenery.
 *
 * The 3d representation of terrain is divided into tiles, each of which
 * consists of a maximum of a quarter of screen. The game shows all tiles that
 * have a possibility of being visible.
 *
 * Each subtile contains a 8x8 heightfield (if present), the main play area
 * must thus be a multiple of 8 in size, having a play area with less than
 * 4 subsquares in a side could create interesting effects though.
 *
 * The collision detection is two-phase. Objects and bullets collide with
 * each other through an N-Tree structure, and the collision to terrain, that
 * is calculated after the collisions between the objects is implemented using
 * a heightfield (and map objects) that are scaled multiple times until they
 * form smooth surfaces instead of clunky ones.
 *
 * Collision squares are numbered starting from the bottom row and advancing
 * up column index first, then row index. Below is a diagram for advancement
 * for an square count of 3 per side (which would make 9 squares and is the
 * minimum area size possible).
 *
 *   ^
 *   |
 *   y
 *   +-------+-------+-------+-------+
 *   |1,3,7,9|2,1,7,8|3,2,8,9|1,3,7,9|
 * --+-------+-------+-------+-------+
 *   |7,4,6,9|8,4,5,7|9,5,6,8|7,4,6,9|
 *   +-------+-------+-------+-------+
 *   |4,1,3,6|5,1,2,4|6,2,3,5|4,1,3,6|
 *   +-------+-------+-------+-------+
 *   |1,3,7,9|2,1,7,8|3,2,8,9|1,3,7,9|
 * --+-------+-------+-------+-------+x->
 *   |                       |
 *   |                       |
 *
 * As you can see, the area is wrap-around. The rightmost column equals
 * leftmost column and top row equals bottom row, the double dashes mark
 * the wrap points. The first number in each cell represents the 'home'
 * square within it (actually, the indexing would start from 0, but I'm
 * too lazy to change the ASCII right now).
 *
 * About the scale of the game, one collision coordinate is exactly one game
 * play area coordinate in length, that is, 1.0f. And this is fixed, nothing
 * can change it.
 */
class Terrain :
  public libfhi::Singleton<Terrain>
{
  private:
    /** The number of squares in the x direction. */
    int collision_squares_x;

    /** The number of squares in the y direction. */
    int collision_squares_y;

    /** The number of visual squares in the x direction. */
    int visual_squares_x;

    /** The number of visual squares in the x direction. */
    int visual_squares_y;

    /** Size x of the playfield (in game visual units). */
    float size_x;

    /** Size y of the playfield (in game visual units). */
    float size_y;

    /** Size x divided by 2 (for some calculation speedup). */
    float size_x_div2;

    /** Size y divided by 2 (for some calculation speedup). */
    float size_y_div2;

    /** Width in heightfield pixels. */
    int field_w;

    /** Heigth in heightfield pixels. */
    int field_h;

    /** Collision plane (between 0 - 255). */
    int collision_plane;

    /** Depth of the bottom of the height map. High values == slowdown. */
    float bottom_depth;

    /** Coefficient for air density drag force. */
    float rho;

    /** The squares. */
    CollisionSquare *collision_squares;

    /** All the squares. */
    VisualSquare *visual_squares;

    /** Heightfield color. */
    libfhi::Color4 *color_map;

    /** Heightfield bitmap. */
    uint8_t *height_map;

    /** The texture used for the field. */
    std::string texture_filename;

    /** All the parallax-scrolling backgrounds. */
    std::vector<ParallaxScroll*> parallax;

  public:
    Terrain();
    Terrain(const char*);
    ~Terrain();

  private:
    bool load_terrain(const char*);

  public:
    void add_parallax(float, float, float, bool, const char*);
    void draw(const libfhi::Vector2&);
    CollisionSquare* get_collision_square(CollisionMap*);

    // Inline methods.
  public:
    inline float get_bottom_depth() const;
    inline int get_collision_plane() const;
    inline CollisionSquare* get_collision_square(int, int);
    inline int get_collision_squares_x() const;
    inline int get_collision_squares_y() const;
    inline libfhi::Color4 get_color(int, int) const;
    inline int get_field_h() const;
    inline int get_field_w() const;
    inline int get_height(int, int) const;
    inline int get_height_map_index(int, int) const;
    inline float get_height_section_horiz(int, int, int) const;
    inline float get_height_section_vert(int, int, int) const;
    inline libfhi::Vector2 get_random_position() const;
    inline libfhi::Vector2 get_relative_bounded_position(
	const libfhi::Vector2&, const libfhi::Vector2&);
    inline float get_rho() const;
    inline float get_size_x() const;
    inline float get_size_y() const;
    inline std::string get_texture_filename() const;
    inline VisualSquare* get_visual_square(int, int);
    inline float height_transform(int);
    inline float heuristic_distance(const libfhi::Vector2&,
	const libfhi::Vector2&);

    // Static inline methods.
  private:
    static inline int transform_game_visual(float);
};

//############################################################################
// CollisionSquare inline ####################################################
//############################################################################

/** Empty the collision maps.
 */
void CollisionSquare::clear_maps()
{
  this->maps.clear();
}

/** Check if need to clear the vector of this square. The square shall be
 * cleared each time it is accessed for the FIRST time in the frame, there is
 * a simple integer to check this. This method is very low-cost compared to
 * iterating through all the collision maps and manually removing them from
 * the squares. In addition, it is safe, as there is no need to worry about
 * possible destroyed entities left haunting in the map vector.
 */
void CollisionSquare::check_frame_clear()
{
  int current_frame = Game::get_frame_number();

  if(this->last_access_frame != current_frame)
  {
    this->maps.clear();
    this->last_access_frame = current_frame;
  }
}

/** Get the search heuristic value of this.
 * @return Value as float.
 */
float CollisionSquare::get_ai_heuristic() const
{
  return this->ai_heuristic;
}

/** Get the previous square. Used in ai searches.
 * @return CollisionSquare pointer.
 */
CollisionSquare* CollisionSquare::get_ai_previous()
{
  return this->ai_previous;
}

/** Return the maps from square.
 * @return All the maps in this square.
 */
std::vector<CollisionMap*>* CollisionSquare::get_maps()
{
  return &(this->maps);
}

/** Return the neighbor of given index.
 * @param idx Index of neighbor. If out of bounds, segfault.
 * @return CollisionSquare pointer.
 */
CollisionSquare* CollisionSquare::get_neighbor(int idx)
{
  return this->neighbors[idx];
}

/** Get translation vector in 2d.
 * @return Reference to the local translation vector..
 */
const libfhi::Vector2& CollisionSquare::get_trans() const
{
  return this->trans;
}

/** INclude given square in this seach of given id.
 * @param sid New search id.
 * @return True if the square already was in the search, false otherwise.
 */
bool CollisionSquare::include_in_search(int sid)
{
  if(this->ai_search_id == sid)
  {
    return false;
  }
  this->ai_search_id = sid;
  return true;
}

/** Return true if this square is free for given entity assuming given
 * reference frame.
 * @param reference_frame Should be the previous frame number.
 * @param ent Entity who's asking.
 */
bool CollisionSquare::is_free_for(int reference_frame,
    const EntityBase *ent) const
{
  return ((this->last_access_frame < reference_frame) ||
      this->maps.empty() || (this->maps.front()->get_host() == ent));
}

/** Check for safe to insert.
 * @return True if yes, false if no.
 */
bool CollisionSquare::is_safe_to_insert()
{
  // Potential clear.
  this->check_frame_clear();

  // Return true if nothing here.
  return (this->ai_allowed && this->maps.empty());
}

/** Check for safe to insert.
 * @param op Who is asking, this one is not considered a threat.
 * @return True if yes, false if no.
 */
bool CollisionSquare::is_safe_to_travel(const EntityBase *op) const
{
  // The reference frame is substracted by one, since at the time of ai path
  // finding there is nothing inserted this frame and all data is from the
  // previous one.
  int reference_frame = Game::get_frame_number() - 1;

  // There is a potential bug here. The collisionmap linked may be destroyed,
  // in which case this will just segfault.
  return (this->ai_allowed && this->is_free_for(reference_frame, op));
}

/** Set heuristic value.
 * @param op New heuristic value.
 */
void CollisionSquare::set_ai_heuristic(float op)
{
  this->ai_heuristic = op;
}

/** Set the previous.
 * @param sq New previous.
 */
void CollisionSquare::set_ai_previous(CollisionSquare *sq)
{
  this->ai_previous = sq;
}

//############################################################################
// CollisionSquare metaprogramming ###########################################
//############################################################################

/** Metaprogramming function to call when utilizing collisions of object data
 * into another object data (i.e. translating stuff).
 * @param method CollisonMap method to check the collisions with.
 * @param cmap Target CollisionMap.
 * @return Found collision.
 */
CollisionMap* CollisionSquare::collides_meta(
    bool (*func)(CollisionMap*, const libfhi::Vector2&, CollisionMap*),
    CollisionMap *cmap)
{
  static const libfhi::Vector2
    ZERO_VECTOR(0.0f, 0.0f),
    NEIGHBOR_DIFF[8] =
    {
      libfhi::Vector2(-1.0f, -1.0f),
      libfhi::Vector2(0.0f, -1.0f),
      libfhi::Vector2(1.0f, -1.0f),
      libfhi::Vector2(-1.0f, 0.0f),
      libfhi::Vector2(1.0f, 0.0f),
      libfhi::Vector2(-1.0f, 1.0f),
      libfhi::Vector2(0.0f, 1.0f),
      libfhi::Vector2(1.0f, 1.0f)
    };

  this->check_frame_clear();

  // This square.
  for(std::vector<CollisionMap*>::iterator i = this->maps.begin(),
      e = this->maps.end(); (i != e); ++i)
  {
    if(func(cmap, ZERO_VECTOR, *i))
    {
      return *i;
    }
  }

  // Hopefully this is unrolled.
  for(int j = 0; (j < 8); ++j)
  {
    CollisionSquare *sq = this->neighbors[j];

    sq->check_frame_clear();

    for(std::vector<CollisionMap*>::iterator i = sq->maps.begin(),
	e = sq->maps.end(); (i != e); ++i)
    {
      if(func(cmap, NEIGHBOR_DIFF[j], *i))
      {
	return *i;
      }
    }
  }

  return NULL;
}

/** Metaprogramming function to check collisions of terrain against targets.
 * @param method CollisonMap method to check the collisions with.
 * @param cmap Target CollisionMap.
 * @return The collision line the object collided with or NULL it not.
 */
CollisionMap::CollisionLine* CollisionSquare::collides_terrain_meta(
    CollisionMap::CollisionLine* (*func)(CollisionMap*, CollisionMap*),
    CollisionMap *cmap)
{
  this->check_frame_clear();
  return func(this->terrain_map, cmap);
}

//############################################################################
// VisualSquare inline #######################################################
//############################################################################

/** Draw this square assuming camera is in given coordinates.
 * @param x X distance from camera (relative).
 * @param y Y distance from camera (relative).
 * @param screen Surface to draw into.
 */
void VisualSquare::draw(float x, float y)
{
  // It is possible to have null meshes.
  if(this->model)
  {
    this->model->set_pos(x, y, 0.0f);
    this->model->tick();
    libfhi::Surface::draw_model(this->model);
  }
}

//############################################################################
// Terrain inline ############################################################
//############################################################################

/** Get the bottom depth.
 * @return Depth as float.
 */
float Terrain::get_bottom_depth() const
{
  return this->bottom_depth;
}

/** Get the collision plane of this terrain.
 * @return Collision plane as integer.
 */
int Terrain::get_collision_plane() const
{
  return this->collision_plane;
}

/** Get the address of a square based on the x and y coordinates of it's lower
 * left coordinate.
 * @param x Coordinate x.
 * @param y Coordinate y.
 * @return Pointer to square or NULL on error.
 */
CollisionSquare* Terrain::get_collision_square(int x, int y)
{
  return this->collision_squares +
    libfhi::congr(y, this->collision_squares_y) * this->collision_squares_x +
    libfhi::congr(x, this->collision_squares_x);
}

/** Return the number of collision squares in x direction.
 * @return Number as int.
 */
int Terrain::get_collision_squares_x() const
{
  return this->collision_squares_x;
}

/** Return the number of collision squares in y direction.
 * @return Number as int.
 */
int Terrain::get_collision_squares_y() const
{
  return this->collision_squares_y;
}

/** Get color from given point.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @return Color in the given spot.
 */
libfhi::Color4 Terrain::get_color(int x, int y) const
{
  if(this->color_map)
  {
    return this->color_map[this->get_height_map_index(x, y)];
  }

  return libfhi::Color4(1.0f, 1.0f, 1.0f, 1.0f);
}

/** Get heightfield height.
 * @return Height as integer.
 */
int Terrain::get_field_h() const
{
  return this->field_h;
}

/** Get heightfield width.
 * @return Width as integer.
 */
int Terrain::get_field_w() const
{
  return this->field_w;
}

/** Get height from given point.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @return Unscaled height or 0 on error.
 */
int Terrain::get_height(int x, int y) const
{
  if(this->height_map)
  {
    return this->height_map[this->get_height_map_index(x, y)];
  }

  return 0;
}

/** Get map index for given coordinates.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @return Index to given point in map (bounded).
 */
int Terrain::get_height_map_index(int x, int y) const
{
  return libfhi::congr(y, this->field_h) * this->field_w +
    libfhi::congr(x, this->field_w);
}

/** Get the x value where collision plane is intersected (in horizontal).
 * @param x1 First x coordinate.
 * @param x2 Second x coordinate.
 * @param y Y coordinate.
 * @return Floating point value in x (field space).
 */
float Terrain::get_height_section_horiz(int x1, int x2, int y) const
{
  static const float PLANEF = static_cast<float>(this->collision_plane);

  float h1 = static_cast<float>(this->get_height(x1, y)),
	h2 = static_cast<float>(this->get_height(x2, y));

  return (PLANEF - h1) / (h2 - h1) * static_cast<float>(x2 - x1) +
    static_cast<float>(x1);
}

/** Get the x value where collision plane is intersected (in vertical).
 * @param y1 First y coordinate.
 * @param y2 Second y coordinate.
 * @param x X coordinate.
 * @return Floating point value in x (field space).
 */
float Terrain::get_height_section_vert(int y1, int y2, int x) const
{
  static const float PLANEF = static_cast<float>(this->collision_plane);

  float h1 = static_cast<float>(this->get_height(x, y1)),
	h2 = static_cast<float>(this->get_height(x, y2));

  return (PLANEF - h1) / (h2 - h1) * static_cast<float>(y2 - y1) +
    static_cast<float>(y1);
}

/** Get random position from this map.
 * @return Random position vector.
 */
libfhi::Vector2 Terrain::get_random_position() const
{
  return libfhi::Vector2(
      static_cast<float>(rand() % 1000) * 0.001f * this->size_x,
      static_cast<float>(rand() % 1000) * 0.001f * this->size_y);
}

/** Get the bounded position required for drawing objects in a relative
 * manner.
 * @param view Viewport position
 * @param src Object position.
 * @return Bounded position vector.
 */
libfhi::Vector2 Terrain::get_relative_bounded_position(
    const libfhi::Vector2 &view, const libfhi::Vector2 &src)
{
  libfhi::Vector2 diff = src - view;

  return libfhi::Vector2(
      (diff.xf < -this->size_x_div2) ? (diff.xf + this->size_x) :
        ((diff.xf > this->size_x_div2) ? (diff.xf - this->size_x) : diff.xf),
      (diff.yf < -this->size_y_div2) ? (diff.yf + this->size_y) :
        ((diff.yf > this->size_y_div2) ? (diff.yf - this->size_y) : diff.yf));
}

/** Get the air density.
 * @return Air density coefficient.
 */
float Terrain::get_rho() const
{
  return this->rho;
}

/** Get the playground size in x dimension.
 * @return Dimension as float.
 */
float Terrain::get_size_x() const
{
  return this->size_x;
}

/** Get the playground size in y dimension.
 * @return Dimension as float.
 */
float Terrain::get_size_y() const
{
  return this->size_y;
}

/** Get the texture name associated with this terrain.
 * @return Texture name as a string.
 */
std::string Terrain::get_texture_filename() const
{
  return this->texture_filename;
}

/** Get a visual square identified by two coordinates.
 * @param x X coordinate.
 * @param y Y coordinate.
 * @return pointer to the square.
 */
VisualSquare* Terrain::get_visual_square(int x, int y)
{
  return this->visual_squares +
    libfhi::congr(y, this->visual_squares_y) * this->visual_squares_x +
    libfhi::congr(x, this->visual_squares_x);
}

/** Transform an integer height value to a floating point height value.
 * @param op Coordinate to transform.
 */
float Terrain::height_transform(int op)
{
  return this->bottom_depth * (-1.0f + static_cast<float>(op) /
      static_cast<float>(this->collision_plane));
}

/** Calculates the heuristic distance of two coordinates (distance along
 * shorter wrap), squared.
 * @param op1 First point.
 * @param op2 Second point.
 * @return Squared distance along shorter wrap.
 */
float Terrain::heuristic_distance(const libfhi::Vector2 &op1,
    const libfhi::Vector2 &op2)
{
  return this->get_relative_bounded_position(op1, op2).length_sqr();
}

//############################################################################
// Terrain static inline #####################################################
//############################################################################

/** Transform game coordinate to visual coordinate.
 * Warning: this is not bounded.
 * @param op Game paly area coordinate.
 * @return Rounded coordinate.
 */
int Terrain::transform_game_visual(float op)
{
  return static_cast<int>(op / TERRAIN_VISUAL_DIMENSION);
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

