#include "terrain.h"

#include "collisionmap.h"
#include "data.h"
#include "entity.h"

#include <sstream>

//############################################################################
// Variables #################################################################
//############################################################################

#ifdef DEBUG
libfhi::PreModel colldisp;
libfhi::PostModel *colldraw;
#endif

//############################################################################
// Helper methods ############################################################
//############################################################################

/** Calculate the distance (in units) from grid point to another grid point.
 * @param gx Grid point x.
 * @param gy Grid point y.
 * @param cx Corner point x.
 * @param cy Corner point y.
 * @return Distance as float.
 */
float distance_2dgrid(int gx, int gy, int cx, int cy)
{
  int dx = cx - gx,
      dy = cy - gy;

  return sqrtf(static_cast<float>(dx * dx) + static_cast<float>(dy * dy));
}

//############################################################################
// CollisionSquare ###########################################################
//############################################################################

/** Default constructor.
 */
CollisionSquare::CollisionSquare()
  : terrain_map(NULL), last_access_frame(0), ai_search_id(0), ai_allowed(true)
{
  // We want the maps to be at least of some size to avoid useless reservation
  // and freeing of data.
  this->maps.reserve(16);
}

/** Default destructor.
 */
CollisionSquare::~CollisionSquare()
{
  delete this->terrain_map;
}

/** Test the collision in here and all the surrounding squares. Return first
 * collision found.
 * @param cmap The map to test.
 * @return First colliding map found.
 */
CollisionMap* CollisionSquare::collides_freeform(CollisionMap *cmap)
{
  return this->collides_meta(CollisionMap::collides_freeform_freeform,
      cmap);
}

/** As above, but assume the tested map is a single dot.
 * @param cmap The map to test.
 * @return First colliding map found.
 */
CollisionMap* CollisionSquare::collides_single_dot(CollisionMap *cmap)
{
  return this->collides_meta(CollisionMap::collides_single_dot_freeform,
      cmap);
}

/** As above, but assume the tested map is a single line.
 * @param cmap The map to test.
 * @return First colliding map found.
 */
CollisionMap* CollisionSquare::collides_single_line(CollisionMap *cmap)
{
  return this->collides_meta(CollisionMap::collides_single_line_freeform,
      cmap);
}

/** Test the collision of this terrain square against target collision map.
 * @param cmap The map to test.
 * @return The collision line the hit happens in.
 */
CollisionMap::CollisionLine*
CollisionSquare::collides_terrain_freeform(CollisionMap *cmap)
{
  return this->collides_terrain_meta(CollisionMap::collides_lines_freeform,
      cmap);
}

/** Test the collision of this terrain square against target collision map,
 * that is assumed to be a single dot.
 * @param cmap The map to test.
 * @return The collision line the hit happens in.
 */
CollisionMap::CollisionLine*
CollisionSquare::collides_terrain_single_dot(CollisionMap *cmap)
{
  return this->collides_terrain_meta(CollisionMap::collides_lines_single_dot,
      cmap);
}

/** Test the collision of this terrain square against target collision map,
 * that is assumed to be a single line.
 * @param cmap The map to test.
 * @return The collision line the hit happens in.
 */
CollisionMap::CollisionLine*
CollisionSquare::collides_terrain_single_line(CollisionMap *cmap)
{
  return this->collides_terrain_meta(CollisionMap::collides_lines_single_line,
      cmap);
}

/** Tell if this square or any of it's neighbors has nearby entities.
 * @param op Entity who's asking.
 * @return True if someone else here, false if not.
 */
bool CollisionSquare::has_nearby_entities(const Entity *op) const
{
  // Get the reference frame, as with the travel safety checker.
  int reference_frame = Game::get_frame_number() - 1;

  // Check here.
  if(!this->is_free_for(reference_frame,
	static_cast<const EntityBase*>(op)))
  {
    return true;
  }

  // Check neighbors.
  int i = 8;
  CollisionSquare* const *array = this->neighbors; // OBFU!
  do {
    CollisionSquare *sq = (*array++);

    if(!sq->is_free_for(reference_frame, op))
    {
      return true;
    }
  } while(--i);

  // Not found
  return false;
}

/** Insert the CollisionMap here.
 * @param op CollisionMap to insert.
 */
void CollisionSquare::insert(CollisionMap *op)
{
  // Check if the vector needs to be emptied first.
  this->check_frame_clear();

  // Push in.
  this->maps.push_back(op);
  op->set_square(this);
}

/** Set the position of this. Params are capped at 23, because they represent
 * the lower left coordinate of this.
 * @param x X coordinate (floating point).
 * @param y Y coordinate (floating point).
 * @param host Host Terrain.
 */
void CollisionSquare::set_position(int x, int y, Terrain *host)
{
  // Constants.
  static const int CGRID = TERRAIN_TILE_COLLISION;

  // Set variables.
  this->pos_x = x;
  this->pos_y = y;
  this->trans.set(static_cast<float>(x), static_cast<float>(y));

  // Set the neighbors. Use the fixed arrangement of neighbors as described
  // earlier.
  this->neighbors[0] = host->get_collision_square(x - 1, y - 1);
  this->neighbors[1] = host->get_collision_square(x, y - 1);
  this->neighbors[2] = host->get_collision_square(x + 1, y - 1);

  this->neighbors[3] = host->get_collision_square(x - 1, y);
  this->neighbors[4] = host->get_collision_square(x + 1, y);

  this->neighbors[5] = host->get_collision_square(x - 1, y + 1);
  this->neighbors[6] = host->get_collision_square(x, y + 1);
  this->neighbors[7] = host->get_collision_square(x + 1, y + 1);

  // Reserve new collision map.
  this->terrain_map = new CollisionMap();

  // Calculate the collision lines for the terrain. They are fairly easy to
  // do, we just take the height at each field point inside the square, then
  // search for upside/downside patterns from them and act.
  //
  // Note that interpolation happens two times the grid size, since each
  // square actually encompasses half of the second.
  //
  // Squares are ordered in this way (y up, x right).
  //
  // A-D
  // | |
  // B-C
  //
  // Just like when constructing the visual data.
  for(int j = 0; (j < CGRID * 2); ++j)
  {
    for(int i = 0; (i < CGRID * 2); ++i)
    {
      // Find absolute coordinates of 
      int cx = i + this->pos_x * CGRID,
	  cy = j + this->pos_y * CGRID,
	  nx = cx + 1,
	  ny = cy + 1;

      // Floating point versions.
      float cxf = static_cast<float>(cx),
	    cyf = static_cast<float>(cy),
	    nxf = static_cast<float>(nx),
	    nyf = static_cast<float>(ny);

      // Find heights in these coords.
      int ha = host->get_height(cx, ny),
	  hb = host->get_height(cx, cy),
	  hc = host->get_height(nx, cy),
	  hd = host->get_height(nx, ny);

      // Over statuses.
      bool oa = (ha >= host->get_collision_plane()),
	   ob = (hb >= host->get_collision_plane()),
	   oc = (hc >= host->get_collision_plane()),
	   od = (hd >= host->get_collision_plane());

      // AI is not allowed in any squares that have heigth.
      if(oa || ob || oc || od)
      {
	this->ai_allowed = false;
      }

      // Create two vectors to hold the points and bool to tell if there
      // really is anything.
      libfhi::Vector2 p1, p2;
      bool foundsection = false;

      // Calculate intersection points along lines.
      libfhi::Vector2
	int_ab(cxf, host->get_height_section_vert(cy, ny, cx)),
	int_bc(host->get_height_section_horiz(cx, nx, cy), cyf),
	int_cd(nxf, host->get_height_section_vert(cy, ny, nx)),
	int_da(host->get_height_section_horiz(cx, nx, ny), nyf);

      // Now begin checking for some friendly patterns, if found, get them.
      // First, vertical walls left.
      if(oa && ob && !oc && !od)
      {
	p1 = int_bc;
	p2 = int_da;
	foundsection = true;
      }
      // Vertical wall right.
      else if(!oa && !ob && oc && od)
      {
	p1 = int_da;
	p2 = int_bc;
	foundsection = true;
      }
      // Horizontal wall up.
      else if(oa && !ob && !oc && od)
      {
	p1 = int_ab;
	p2 = int_cd;
	foundsection = true;
      }
      // Horizontal wall down.
      else if(!oa && ob && oc && !od)
      {
	p1 = int_cd;
	p2 = int_ab;
	foundsection = true;
      }
      // Corner a.
      else if(oa && !ob && !oc && !od)
      {
	p1 = int_ab;
	p2 = int_da;
	foundsection = true;
      }
      // Corner a inverse.
      else if(!oa && ob && oc && od)
      {
	p1 = int_da;
	p2 = int_ab;
	foundsection = true;
      }
      // Corner b.
      else if(!oa && ob && !oc && !od)
      {
	p1 = int_bc;
	p2 = int_ab;
	foundsection = true;
      }
      // Corner b inverse.
      else if(oa && !ob && oc && od)
      {
	p1 = int_ab;
	p2 = int_bc;
	foundsection = true;
      }
      // Corner c.
      else if(!oa && !ob && oc && !od)
      {
	p1 = int_cd;
	p2 = int_bc;
	foundsection = true;
      }
      // Corner c inverse.
      else if(oa && ob && !oc && od)
      {
	p1 = int_bc;
	p2 = int_cd;
	foundsection = true;
      }
      // Corner d.
      else if(!oa && !ob && !oc && od)
      {
	p1 = int_da;
	p2 = int_cd;
	foundsection = true;
      }
      // Corner d inverse.
      else if(oa && ob && oc && !od)
      {
	p1 = int_cd;
	p2 = int_da;
	foundsection = true;
      }

      // If section found, add a line and transform coords while at it.
      if(foundsection)
      {
	p1.xf = p1.xf / static_cast<float>(CGRID) - this->trans.xf;
	p1.yf = p1.yf / static_cast<float>(CGRID) - this->trans.yf;
	p2.xf = p2.xf / static_cast<float>(CGRID) - this->trans.xf;
	p2.yf = p2.yf / static_cast<float>(CGRID) - this->trans.yf;

	this->terrain_map->add_line(p1, p2,
	    TERRAIN_COLLISION_LINE_WIDTH);

#ifdef DEBUG
	libfhi::ComponentVertex
	  *v1 = colldisp.add_vertex(
	      libfhi::Vector3(p1.xf + this->trans.xf, p1.yf + this->trans.yf,
		0.0f),
	      libfhi::Color4(1.0f, 0.0f, 0.0f, 1.0f),
	      libfhi::Vector2(0.0f, 0.0f)),
	  *v2 = colldisp.add_vertex(
	      libfhi::Vector3(p2.xf + this->trans.xf, p2.yf + this->trans.yf,
		0.0f),
	      libfhi::Color4(0.0f, 0.0f, 1.0f, 1.0f),
	      libfhi::Vector2(0.0f, 0.0f));

	colldisp.add_face(v1->get_idx(), v2->get_idx());

	v1 = colldisp.add_vertex(
	    libfhi::Vector3(this->trans.xf, this->trans.yf, 0.0f),
	      libfhi::Color4(0.0f, 1.0f, 1.0f, 1.0f),
	      libfhi::Vector2(0.0f, 0.0f));
	v2 = colldisp.add_vertex(
	    libfhi::Vector3(this->trans.xf + 1.0f, this->trans.yf, 0.0f),
	      libfhi::Color4(0.0f, 1.0f, 1.0f, 1.0f),
	      libfhi::Vector2(0.0f, 0.0f));

	libfhi::ComponentVertex
	  *v3 = colldisp.add_vertex(
	      libfhi::Vector3(this->trans.xf, this->trans.yf + 1.0f, 0.0f),
	      libfhi::Color4(0.0f, 1.0f, 1.0f, 1.0f),
	      libfhi::Vector2(0.0f, 0.0f)),
  	  *v4 = colldisp.add_vertex(
  	      libfhi::Vector3(this->trans.xf + 1.0f, this->trans.yf + 1.0f,
		0.0f),
	      libfhi::Color4(0.0f, 1.0f, 1.0f, 1.0f),
	      libfhi::Vector2(0.0f, 0.0f));

	colldisp.add_face(v1->get_idx(), v2->get_idx());
	colldisp.add_face(v1->get_idx(), v3->get_idx());
	colldisp.add_face(v4->get_idx(), v2->get_idx());
	colldisp.add_face(v4->get_idx(), v3->get_idx());
#endif
      }
    }
  }

  // Terrain is constant, but still needs to have it's data in the same
  // 'transformed' coordinates as everyone else.
  this->terrain_map->transform(libfhi::Vector2(0.0f, 0.0f), 1.0f, 0.0f);
  this->terrain_map->set_faction(FACTION_NONE);
}

//############################################################################
// VisualSquare ##############################################################
//############################################################################

/** Default constructor.
 */
VisualSquare::VisualSquare()
  : model(NULL), mesh(NULL)
{
  // Do nothing.
}

/** Default destructor.
 */
VisualSquare::~VisualSquare()
{
  delete this->model;
  delete this->mesh;
}

/** Default constructor.
 * @param x The x coordinate in square space.
 * @param y The y coordinate in square space.
 * @param host Host terrai object.
 */
void VisualSquare::set_position(int x, int y, Terrain *host)
{
  static const int GRID_SIZE = TERRAIN_TILE_VISUAL + 3;
  static const float GRID_MUL = TERRAIN_VISUAL_DIMENSION /
    static_cast<float>(TERRAIN_TILE_VISUAL);
  size_t main_grid[GRID_SIZE * GRID_SIZE];

  // Set position.
  this->pos_x = x;
  this->pos_y = y;
  
  // Get fields from host.
  float field_w = static_cast<float>(host->get_field_w()),
	field_h = static_cast<float>(host->get_field_h());

  // Create a new PreModel (will be exported into a mesh).
  libfhi::PreModel *pre = new libfhi::PreModel();

  // Iterate through this square. Note that the first coordinates on the
  // 'next' squares need to be taken into account to generate a smooth and
  // nice result. Grid in this is done in the following way:
  //
  //  y+
  //  +-----+
  //  |A   D|
  //  | \ / |
  //  |  E  |
  //  | / \ |
  //  |B   C|
  //  +-----+x+
  //
  // In this case, each loop, grid points are filled on alphabetic order,
  // but only for those grid points that exist (are not 0).
  //
  // Another important note is, that the loop rans one square block larger
  // than it would need, to get the lighting correct.
  
  // Y starts a bit off.
  float curr_yf = -GRID_MUL, next_yf = 0.0f;

  // Begin the actual loop.
  for(int j = 0; (j < TERRAIN_TILE_VISUAL + 2);
      ++j, curr_yf += GRID_MUL, next_yf += GRID_MUL)
  {
    // X starts from a bit off.
    float curr_xf = -GRID_MUL, next_xf = 0.0f;

    for(int i = 0; (i < TERRAIN_TILE_VISUAL + 2);
	++i, curr_xf += GRID_MUL, next_xf += GRID_MUL)
    {
      // Calculate host grid index.
      int real_x = this->pos_x * TERRAIN_TILE_VISUAL + i - 1,
	  real_y = this->pos_y * TERRAIN_TILE_VISUAL + j - 1;

      // Calculate local square index.
      int idx_a = (j + 1) * GRID_SIZE + i,
	  idx_b = j * GRID_SIZE + i,
	  idx_c = j * GRID_SIZE + (i + 1),
	  idx_d = (j + 1) * GRID_SIZE + (i + 1);

      // Get height integers.
      int hai = host->get_height(real_x, real_y + 1),
	  hbi = host->get_height(real_x, real_y),
	  hci = host->get_height(real_x + 1, real_y),
	  hdi = host->get_height(real_x + 1, real_y + 1);

      // Get height values.
      float haf = host->height_transform(hai),
	    hbf = host->height_transform(hbi),
	    hcf = host->height_transform(hci),
	    hdf = host->height_transform(hdi);

      float tex_x1 = static_cast<float>(real_x) / (field_w + 0.0f),
	    tex_y1 = 1.0f - static_cast<float>(real_y) / (field_h + 0.0f),
	    tex_x2 = static_cast<float>(real_x + 1) / (field_w + 0.0f),
	    tex_y2 = 1.0f - static_cast<float>(real_y + 1) / (field_h + 0.0f);

      // Points.
      libfhi::Vector3
	va(curr_xf, next_yf, haf),
	vb(curr_xf, curr_yf, hbf),
	vc(next_xf, curr_yf, hcf),
	vd(next_xf, next_yf, hdf);

      // Texture coordinates.
      libfhi::Vector2
	ta(tex_x1, tex_y2),
	tb(tex_x1, tex_y1),
	tc(tex_x2, tex_y1),
	td(tex_x2, tex_y2);

      // Colors.
      libfhi::Color4
	ca = host->get_color(real_x, real_y + 1),
	cb = host->get_color(real_x, real_y),
	cc = host->get_color(real_x + 1, real_y),
	cd = host->get_color(real_x + 1, real_y + 1);

      // Create lower left corner.
      if((i == 0) && (j == 0))
      {
	main_grid[idx_b] = (hbi) ? pre->add_vertex(vb, cb, tb)->get_idx() : 0;
      }
      // Create lower right corner.
      if(j == 0)
      {
	main_grid[idx_c] = (hci) ? pre->add_vertex(vc, cc, tc)->get_idx() : 0;
      }
      // Create upper left corner.
      if(i == 0)
      {
  	main_grid[idx_a] = (hai) ? pre->add_vertex(va, ca, ta)->get_idx() : 0;
      }
      // Always create upper right corner.
      main_grid[idx_d] = (hdi) ? pre->add_vertex(vd, cd, td)->get_idx() : 0;

      // Calculate number of real coords.
      int num_real = ((hai) ? 1 : 0) + ((hbi) ? 1 : 0) + ((hci) ? 1 : 0) +
	((hdi) ? 1 : 0);

      // If at least 3 corners exist, create the approperiate triangles.
      if(num_real >= 3)
      {
	size_t vidx_a = main_grid[idx_a],
	       vidx_b = main_grid[idx_b],
	       vidx_c = main_grid[idx_c],
	       vidx_d = main_grid[idx_d];

	// Enable forget mode if neccessary.
	if((i == 0) || (j == 0) || (i == TERRAIN_TILE_VISUAL + 1) ||
	    (j == TERRAIN_TILE_VISUAL + 1))
	{
	  pre->set_attr(libfhi::PreModel::ATTR_FORGET);
	}

	// Averages.
	libfhi::Color4 ce;
        ce.average(ca, cb, cc, cd);
	float hef = (haf + hbf + hcf + hdf) * 0.25f;

	// Addition index.
	size_t middle_idx = pre->add_vertex(
	    libfhi::Vector3(
	      (curr_xf + next_xf) * 0.5f,
	      (curr_yf + next_yf) * 0.5f,
	      hef),
	    ce,
	    libfhi::Vector2(
	      (tex_x1 + tex_x2) * 0.5f,
	      (tex_y1 + tex_y2) * 0.5f)
	    )->get_idx();

	// If all valid.
	if(hai && hbi && hci && hdi)
	{
	  pre->add_face(vidx_a, vidx_b, middle_idx);
	  pre->add_face(vidx_b, vidx_c, middle_idx);
	  pre->add_face(vidx_c, vidx_d, middle_idx);
	  pre->add_face(vidx_d, vidx_a, middle_idx);
	}
	else
	{
	  // Down left.
	  if(hai && hbi && hci)
	  {
	    pre->add_face(vidx_a, vidx_b, middle_idx);
	    pre->add_face(vidx_b, vidx_c, middle_idx);
	  }

	  // Down right.
	  if(hbi && hci && hdi)
	  {
	    pre->add_face(vidx_b, vidx_c, middle_idx);
	    pre->add_face(vidx_c, vidx_d, middle_idx);
	  }

	  // Up right
	  if(hci && hdi && hai)
	  {
	    pre->add_face(vidx_c, vidx_d, middle_idx);
	    pre->add_face(vidx_d, vidx_a, middle_idx);
	  }

	  // Up left.
	  if(hdi && hai && hbi)
	  {
	    pre->add_face(vidx_d, vidx_a, middle_idx);
	    pre->add_face(vidx_a, vidx_b, middle_idx);
	  }
	}

	pre->attr_not(libfhi::PreModel::ATTR_FORGET);
      }
    }
  }

  // Create a new model by compiling the background into a mesh with no name.
  pre->set_texture_file(host->get_texture_filename());
  this->mesh = pre->compile(NULL);

  // It is possible that the mesh was empty. In that case, do not bother with
  // the model.
  if(mesh)
  {
    this->model = new libfhi::PostModel(this->mesh);
    this->model->set_rotation(libfhi::Orientation::ROTATION_NO);
  }

#ifdef DEBUG
  std::cout << "Generated square in (" << this->pos_x << "," << this->pos_y
    << "): V(" << pre->get_num_vertex() << ") F(" << pre->get_num_face() <<
    ")\n";
#endif

  // Delete premodel and return.
  delete pre;
}

//############################################################################
// ParallaxScroll ############################################################
//############################################################################

/** Default constructor.
 */
ParallaxScroll::ParallaxScroll()
  : mesh(NULL), model(NULL)
{
  // Do nothing.
}

/** Default destructor.
 */
ParallaxScroll::~ParallaxScroll()
{
  // Delete model and mesh.
  delete this->model;
  delete this->mesh;

  // Clean up the texture file from taking space.
  libfhi::Texture::canon_delete(this->texture_file.c_str());
}

/** Initialization function.
 * @param xscale How many screens does the parallax scroll encompass in x.
 * @param yscale How many screens does the parallax scroll encompass in y.
 * @param zposition The z position of the scroll.
 * @param ph Phase.
 * @param filename Texture file name.
 * @param host Host terrain to use.
 * @return True if successful, false otherwise.
 */
bool ParallaxScroll::init(float xscale, float yscale, float zposition,
    bool ph, const char *filename, Terrain *host)
{
  // Try to load the texture, abort on failure.
  if(Data::load_texture(filename) == NULL)
  {
    return false;
  }

  // Set parallax scales.
  this->scale_x = xscale;
  this->scale_y = yscale;
  this->zpos = zposition;
  this->phase = ph;
  this->texture_file = filename;

  // Reserve generation model.
  libfhi::PreModel *pre = new libfhi::PreModel();
  pre->set_texture_file(this->texture_file);

  // Get the coordinate x addition
  xscale = host->get_size_x() * xscale * 0.5f;
  yscale = host->get_size_y() * yscale * 0.5f;

  // Add vertices (only 25 of them).
  float ystart = -yscale;

  for(int j = 0; (j < 4); ++j, ystart += yscale)
  {
    float xstart = -xscale;

    for(int i = 0; (i < 4); ++i, xstart += xscale)
    {
      size_t
	va = pre->add_vertex(
	    libfhi::Vector3(xstart, ystart + yscale, 0.0f),
	    libfhi::Color4(1.0f, 1.0f, 1.0f, 1.0f),
	    libfhi::Vector2(0.0f, 1.0f))->get_idx(),
	vb = pre->add_vertex(
	    libfhi::Vector3(xstart, ystart, 0.0f),
	    libfhi::Color4(1.0f, 1.0f, 1.0f, 1.0f),
	    libfhi::Vector2(0.0f, 0.0f))->get_idx(),
	vc = pre->add_vertex(
	    libfhi::Vector3(xstart + xscale, ystart, 0.0f),
	    libfhi::Color4(1.0f, 1.0f, 1.0f, 1.0f),
	    libfhi::Vector2(1.0f, 0.0f))->get_idx(),
	vd = pre->add_vertex(
	    libfhi::Vector3(xstart + xscale, ystart + yscale, 0.0f),
	    libfhi::Color4(1.0f, 1.0f, 1.0f, 1.0f),
	    libfhi::Vector2(1.0f, 1.0f))->get_idx();

      pre->add_face(va, vb, vc, vd);
    }
  }

  // Compile.
  this->mesh = pre->compile(NULL);
  this->model = new libfhi::PostModel(this->mesh);

  std::cout << "Parallax scroll created with V(" <<
    static_cast<uint32_t>(pre->get_num_vertex()) << ") F(" <<
    static_cast<uint32_t>(pre->get_num_face()) << ") T(\"" << filename <<
    "\")\n";

  // Delete and return.
  delete pre;
  return true;
}

/** Draw one parallax scroll.
 * @param view View vector.
 * @param ph False former, true latter.
 */
void ParallaxScroll::draw(const libfhi::Vector2 &view, bool ph)
{
  if(ph != this->phase)
  {
    return;
  }

  this->model->set_pos(-view.xf * this->scale_x, -view.yf * this->scale_y,
      this->zpos);
  this->model->tick();
  libfhi::Surface::draw_model(this->model);
}

//############################################################################
// Terrain ###################################################################
//############################################################################

/** Empty constructor. Uses somewhat reasonable values for initialization, I
 * have absolutely no patience of making named constants of them, since they
 * should ALWAYS be defined in the map files.
 */
Terrain::Terrain() :
  collision_plane(TERRAIN_DEFAULT_COLLISION_PLANE),
  bottom_depth(TERRAIN_DEFAULT_BOTTOM_DEPTH),
  rho(PHYSICS_AIR_DENSITY_MULTIPLIER * PHYSICS_DEFAULT_AIR_DENSITY),
  collision_squares(NULL), visual_squares(NULL),
  color_map(NULL), height_map(NULL)
{
  std::cerr << "Warning: Terrain constructor called without arguments.\n";
}

/** Default constructor.
 * @param cplane Collision plane.
 * @param bottom Bottom depth.
 * @param airdensity Air density.
 */
Terrain::Terrain(const char *filename) :
  collision_plane(TERRAIN_DEFAULT_COLLISION_PLANE),
  bottom_depth(TERRAIN_DEFAULT_BOTTOM_DEPTH),
  rho(PHYSICS_AIR_DENSITY_MULTIPLIER * PHYSICS_DEFAULT_AIR_DENSITY),
  collision_squares(NULL), visual_squares(NULL),
  color_map(NULL), height_map(NULL)
{
  libfhi::ConfigFile filu(filename);

  if(!filu.is_ok())
  {
    std::cerr << "Error: Could not open map: \"" << filename << "\"\n";
    return;
  }

  while(filu.advance())
  {
    // Load the terrain file. This will open the different png files etc.
    // Please note that bottom depth and collision plane must be specified
    // before a call to the terrain command or they will have no effect.
    if(filu.has_id_arg("terrain", 1))
    {
      this->load_terrain(filu.get_string(0).c_str());
    }
    else if(filu.has_id_arg("collision_plane", 1))
    {
      this->collision_plane = filu.get_int(0);
    }
    else if(filu.has_id_arg("bottom_depth", 1))
    {
      this->bottom_depth = filu.get_float(0);
    }
    else if(filu.has_id_arg("rho", 1))
    {
      this->rho = filu.get_float(0) * PHYSICS_AIR_DENSITY_MULTIPLIER;
    }
    // Parallax scroll should be invoked after invoking terrain loading.
    else if(filu.has_id_arg("parallax", 5))
    {
      this->add_parallax(filu.get_float(0),
	  filu.get_float(1),
	  filu.get_float(2),
	  static_cast<bool>(filu.get_int(3) != 0),
	  filu.get_string(4).c_str());
    }
    else
    {
      filu.warn_empty();
    }
  }
}

/** Default destructor.
 */
Terrain::~Terrain()
{
  // Delete all squares and maps.
  delete[] this->collision_squares;
  delete[] this->visual_squares;
  delete[] this->color_map;
  delete[] this->height_map;

  // Delete the parallax scrolls.
  for(std::vector<ParallaxScroll*>::iterator i = this->parallax.begin(),
      e = this->parallax.end(); (i != e); ++i)
  {
    delete (*i);
  }

  // Delete texture with our filename.
  libfhi::Texture::canon_delete(this->texture_filename.c_str());
}

/** Generates a new terrain and all the subsquares in it.
 * @param filename The file basename, will be appended with something.
 * @return True on success, false on failure.
 */
bool Terrain::load_terrain(const char *filename)
{
  // We need stringstreams for the filenames.
  std::stringstream sscolor, ssheight, sstexture;

  // Open the color map and heightfield with given name.
  sscolor << filename << "_color.png";
  libfhi::sdlSurface *surf_color = libfhi::sdlsurface_new(
      sscolor.str().c_str());

  ssheight << filename << "_height.png";
  libfhi::sdlSurface *surf_height = libfhi::sdlsurface_new(
      ssheight.str().c_str());

  // Save the filename, squares will take care of loading textures.
  sstexture << filename << "_texture.png";
  this->texture_filename = sstexture.str();

  // Check that files exist.
  if(!surf_color)
  {
    std::cout << "Error: file \"" << sscolor.str() <<
      "\" could not be found.\n";
    delete surf_height;
    return false;
  }
  if(!surf_height)
  {
    std::cout << "Error: file \"" << ssheight.str() <<
      "\" could not be found.\n";
    delete surf_color;
    return false;
  }

  // Check that both are of equal size.
  if((surf_color->get_w() != surf_height->get_w()) ||
      (surf_color->get_h() != surf_height->get_h()))
  {
    std::cout << "Error: images from \"" << filename <<
      "\" are of different size.\n";
    delete surf_color;
    delete surf_height;
    return false;
  }

  // Check bit depths.
  if(surf_color->get_bpp() != 32)
  {
    std::cout << "Error: \"" << sscolor.str() << "\" not in RGBA mode.\n";
    return false;
  }
  if(surf_height->get_bpp() != 8)
  {
    std::cout << "Error: \"" << ssheight.str() <<
      "\" not in grayscale mode.\n";
    return false;
  }

  // Save heights now that they are equal.
  this->field_w = surf_color->get_w();
  this->field_h = surf_color->get_h();

  // Check for divisiblity
  if((this->field_w % TERRAIN_TILE_VISUAL != 0) ||
      (this->field_h % TERRAIN_TILE_VISUAL != 0))
  {
    std::cout << "Error: Image dimensions from \"" << filename <<
      "\" are not divisible by " << TERRAIN_TILE_VISUAL << ".\n";
    delete surf_color;
    delete surf_height;
    return false;
  }

  // Get visual and collision square counts.
  this->visual_squares_x = this->field_w / TERRAIN_TILE_VISUAL;
  this->visual_squares_y = this->field_h / TERRAIN_TILE_VISUAL;
  this->collision_squares_x = this->field_w / TERRAIN_TILE_COLLISION;
  this->collision_squares_y = this->field_h / TERRAIN_TILE_COLLISION;
  this->size_x = static_cast<float>(this->visual_squares_x) *
    TERRAIN_VISUAL_DIMENSION;
  this->size_y = static_cast<float>(this->visual_squares_y) *
    TERRAIN_VISUAL_DIMENSION;
  this->size_x_div2 = this->size_x / 2.0f;
  this->size_y_div2 = this->size_y / 2.0f;

  // Create the colors and heights.
  int field_size = this->field_w * field_h;
  this->height_map = new uint8_t[field_size];
  this->color_map = new libfhi::Color4[field_size];

  // Color iterators
  uint8_t *csrc = static_cast<uint8_t*>(surf_color->get_cbuf()),
	  *hsrc = static_cast<uint8_t*>(surf_height->get_cbuf());

  // Iterate all colors.
  for(int j = 0; (j < this->field_h); ++j)
  {
    int inverse_row = (this->field_h - 1 - j) * this->field_w;

    uint8_t *hdst = this->height_map + inverse_row;
    libfhi::Color4 *cdst = this->color_map + inverse_row;

    for(int i = 0; (i < this->field_w); ++i)
    {
      float rf = static_cast<float>(csrc[0]) / 255.0f,
  	    gf = static_cast<float>(csrc[1]) / 255.0f,
  	    bf = static_cast<float>(csrc[2]) / 255.0f,
  	    af = static_cast<float>(csrc[3]) / 255.0f;

      (*cdst) = libfhi::Color4(rf, gf, bf, af);
      (*hdst) = (*hsrc);

      ++cdst;
      ++hdst;

      // These increase all the time and it does not matter.
      csrc += 4;
      ++hsrc;
    }
  }

  // Destroy used surfaces, they are no longer needed.
  delete surf_color;
  delete surf_height;

  // Now, create all the subsquares.
  int collision_square_count =
        this->collision_squares_x * this->collision_squares_y,
      visual_square_count =
        this->visual_squares_x * this->visual_squares_y;
  this->collision_squares = new CollisionSquare[collision_square_count];
  this->visual_squares = new VisualSquare[visual_square_count];

  // Initialize collision squares.
  for(int j = 0; (j < this->collision_squares_y); ++j)
  {
    for(int i = 0; (i < this->collision_squares_x); ++i)
    {
      this->get_collision_square(i, j)->set_position(i, j, this);
    }
  }

  // Initialize visual squares.
  for(int j = 0; (j < this->visual_squares_y); ++j)
  {
    for(int i = 0; (i < this->visual_squares_x); ++i)
    {
      this->get_visual_square(i, j)->set_position(i, j, this);
    }
  }

  // We are ready, safely delete fields.
  libfhi::zdel_array(this->height_map);
  libfhi::zdel_array(this->color_map);

#ifdef DEBUG
  colldraw = new libfhi::PostModel(colldisp.compile(NULL));
#endif

  std::cout << "Loaded terrain \"" << filename << "\" F(" <<
    this->field_w << "x" << this->field_h << ") V(" << this->visual_squares_x
    << "x" << this->visual_squares_y << ") C(" << this->collision_squares_x <<
    "x" << this->collision_squares_y << ")\n";
  return true;
}

//############################################################################
// Terrain methods ###########################################################
//############################################################################

/** Draw this terrain, the camera is in the given coordinates, draw
 * surrounding squares. Note, that when drawing terrain the camera itself does
 * not move, there would be no point, since the world wraps around. The
 * surrounding squares are drawn relative to it.
 * @param view Viewport.
 * @param screen Screen to draw into.
 */
void Terrain::draw(const libfhi::Vector2 &view)
{
  // Just a shorthand.
  static const float
    DIM = TERRAIN_VISUAL_DIMENSION,
    INVDIM = 1.0f / DIM;

  // Draw pre-parallax scrolls.
  for(std::vector<ParallaxScroll*>::iterator i = this->parallax.begin(),
      e = this->parallax.end(); (i != e); ++i)
  {
    (*i)->draw(view, false);
  }

  // Precalculated value.
  float axisdiff = (this->bottom_depth + VIEW_CAMERA_PLANE) * VIEW_RATIO;

  // Begin 
  float mx = -libfhi::congr(view.xf, DIM),
      	my = -libfhi::congr(view.yf, DIM);
  int xminus = -static_cast<int>(ceilf((axisdiff + mx) * INVDIM)),
      yminus = -static_cast<int>(ceilf((axisdiff + my) * INVDIM)),
      xplus = static_cast<int>(ceilf((axisdiff - (DIM + mx)) * INVDIM)),
      yplus = static_cast<int>(ceilf((axisdiff - (DIM + my)) * INVDIM));
  int sx = Terrain::transform_game_visual(view.xf),
      sy = Terrain::transform_game_visual(view.yf);

  mx += xminus * DIM;
  my += yminus * DIM;

  // Draw All relevant surrounding squares.
  for(int j = yminus; (j <= yplus); ++j, my += DIM)
  {
    float fx = mx;

    for(int i = xminus; (i <= xplus); ++i, fx += DIM)
    {
      this->get_visual_square(sx + i, sy + j)->draw(fx, my);
    }
  }

  // Draw parallax scrollers, bottom-up.
  for(std::vector<ParallaxScroll*>::iterator i = this->parallax.begin(),
      e = this->parallax.end(); (i != e); ++i)
  {
    (*i)->draw(view, true);
  }

#ifdef DEBUG
  colldraw->set_pos(-view.xf, -view.yf, 0.0f);
  colldraw->tick();
  //screen->set_boundary(0, 0, screen->get_h() -1, screen->get_h() - 1, Game::VIEW_RATIO, 15.0f, 300.0f);
  libfhi::Surface::draw_model(colldraw);
#endif
}

/** Gets the subsquare this collisionmap is located in (all collision maps are
 * always completely located inside one square). This method WILL modify the
 * map that is inserted (and set the host square of the map to the found
 * square), but will NOT actually insert it. To act identically, but perform
 * the insertion, call the method insert.
 * @param op The collision map to test.
 * @return A pointer to the list of potential collision targets.
 */
CollisionSquare* Terrain::get_collision_square(CollisionMap *op)
{
  // Get the absolute lower left coordinates from collision object.
  libfhi::Vector2 ll = op->get_lowerleft();

  // Get the square coordinates (same as collision coordinates!).
  int xi = static_cast<int>(floorf(ll.xf)),
      yi = static_cast<int>(floorf(ll.yf)),
      congrx = libfhi::congr(xi, this->collision_squares_x),
      congry = libfhi::congr(yi, this->collision_squares_y);

  // There is a possibility, that the congruential coordinates do not equal
  // the actual coordinates. If this is the case, there needs to be heavy
  // translation.
  libfhi::Vector2 trans(
      (congrx == xi) ? 0.0f : static_cast<float>(congrx - xi),
      (congry == yi) ? 0.0f : static_cast<float>(congry - yi));

  // Get the square corresponding to given coords.
  CollisionSquare *ret = this->collision_squares +
    congry * this->collision_squares_x + congrx;

  /*std::cout << "LL: " << ll << "\n";
  std::cout << "Xi/Yi: " << xi << ", " << yi << "\n";
  std::cout << "X/Y: " << congrx << ", " << congry << "\n";*/

  // Translate the collision map by the negative of square index to get it
  // into absolute square coords.
  op->translate(trans - ret->get_trans());

  // Return the square it is in.
  return ret;
}

/** Add one parallax scroll.
 * @param xscale X scale of the parallax scroll.
 * @param yscale X scale of the parallax scroll.
 * @param zpos Z position.
 * @param phase Draw phase.
 * @param filename Texture filename.
 */
void Terrain::add_parallax(float xscale, float yscale, float zpos,
    bool phase, const char *filename)
{
  ParallaxScroll *scroll = new ParallaxScroll();

  if(scroll->init(xscale, yscale, zpos, phase, filename, this))
  {
    this->parallax.push_back(scroll);
  }
  else
  {
    std::cerr << "Warning: initialization of parallax scroll \"" <<
      filename << "\" failed.\n";

    delete scroll;
  }
}

//############################################################################
// End #######################################################################
//############################################################################

