#ifndef bullet_h
#define bullet_h

//############################################################################
// Include ###################################################################
//############################################################################

#include "libfhi.h"

//############################################################################
// Include ###################################################################
//############################################################################

/** This is the 'superclass' for all munitions, even though there is no real
 * virtuality at all. The weapon structures will call correct methods that
 * will in turn call correct methods to handle the munitions.
 */
class Munition
{
  protected:
    /** Faction insignia. */
    int faction;

    /** Position. */
    libfhi::Vector2 pos;

    /** Movement. */
    libfhi::Vector2 mov;

    /** Acceleration. */
    libfhi::Vector2 acc;

    /** Translated position. */
    libfhi::Vector2 tpos;

    /** Direction as unit vector. */
    libfhi::Vector2 dir_unit;

    union
    {
      /** Power value of this. */
      int power;

      /** Target of this. */
      Entity *ai_target;
    };

    /** Damage of this. */
    int damage;

    /** Lifetime, all bullets have finite lifetime (this cannot be < 0). */
    int lifetime;

  public:
    inline Munition();
    inline ~Munition();

    inline void init_bullet(int, const libfhi::Vector2&,
	const libfhi::Vector2&, float, float, int, int, int, float);
    inline void init_homing(int, const libfhi::Vector2&,
	const libfhi::Vector2&, float, float, int, int, float);
    inline void init_instant(int, const libfhi::Vector2&, float, float, int,
	int);

    // Inline.
  public:
    inline int get_damage() const;
    inline const libfhi::Vector2& get_dir() const;
    inline int get_faction() const;
    inline const libfhi::Vector2& get_pos() const;
    inline void homing_select_target();
    inline bool inside(const libfhi::Vector2&, float);
    inline bool tick_bullet();
    inline bool tick_homing(float, float, float, float, float, int);
    inline bool tick_instant();
    inline void transform(libfhi::Vector3*, libfhi::Vector3*, size_t) const;
};

//############################################################################
// Munition construction #####################################################
//############################################################################

/** Empty constructor, does nothing.
 */
Munition::Munition()
{
  // Do nothing.
}

/** Default destructor.
 */
Munition::~Munition()
{
  // Do nothing.
}

/** Initialization for bullets.
 * @param f Faction.
 * @param hpos Host position.
 * @param hmov Host movement vector.
 * @param op_cr Cosine of direction rotation.
 * @param op_sr Sine of direction rotation.
 * @param pow Power of this.
 * @param dmg Damage of this.
 * @param lif Lifetime.
 * @param speed Muzzle speed of bullets (of this type).
 */
void Munition::init_bullet(int f, const libfhi::Vector2 &hpos,
    const libfhi::Vector2 &hmov, float op_cr, float op_sr, int pow, int dmg,
    int lif, float speed)
{
  this->faction = f;
  this->pos = hpos;
  this->dir_unit.set(op_cr, op_sr);
  this->power = pow;
  this->damage = dmg;
  this->lifetime = lif;

  // Calculate movement components.
  this->mov = this->dir_unit * speed + hmov;
}

/** Initialization for homing weapons.
 * @param f Faction.
 * @param hpos Host position.
 * @param hmov Host movement.
 * @param op_cr Cosine of direction rotation.
 * @param op_sr Sine of direction rotation.
 * @param dmg Damage of this.
 * @param lif Lifetime.
 */
void Munition::init_homing(int f, const libfhi::Vector2 &hpos,
    const libfhi::Vector2 &hmov, float op_cr, float op_sr, int dmg, int lif,
    float speed)
{
  this->faction = f;
  this->pos = hpos;
  this->dir_unit.set(op_cr, op_sr);
  this->damage = dmg;
  this->lifetime = lif;

  // Missiles select target upon shooting.
  this->mov = this->dir_unit * speed + hmov;
  this->acc.set(0.0f, 0.0f);
  this->homing_select_target();
}

/** Initialization for instants.
 * @param f Faction.
 * @param hpos Host position.
 * @param op_cr Cosine of direction rotation.
 * @param op_sr Sine of direction rotation.
 * @param dmg Damage of this.
 * @param lif Lifetime.
 */
void Munition::init_instant(int f, const libfhi::Vector2 &hpos, float op_cr,
    float op_sr, int dmg, int lif)
{
  this->faction = f;
  this->pos = hpos;
  this->dir_unit.set(op_cr, op_sr);
  this->damage = dmg;
  this->lifetime = lif;
}

//############################################################################
// Munition inline ###########################################################
//############################################################################

/** Get damage caused by this.
 * @return Damage as integer.
 */
int Munition::get_damage() const
{
  return this->damage;
}

/** Get direction.
 * @return Const reference to the vector.
 */
const libfhi::Vector2& Munition::get_dir() const
{
  return this->dir_unit;
}

/** Get the faction of this.
 * @return Faction enum.
 */
int Munition::get_faction() const
{
  return this->faction;
}

/** Get position of this.
 * @return Position as 2d vector.
 */
const libfhi::Vector2& Munition::get_pos() const
{
  return this->pos;
}

/** Select a new target for a homing weapon.
 */
void Munition::homing_select_target()
{
  libfhi::Vector2 select_pos = this->pos + this->dir_unit *
    WEAPON_HOMING_TARGET_ADVANCE;

  // Using absorb fire range for calculations.
  std::vector<std::pair<float, Entity*> > *targets =
    Entity::ai_get_proximity_targets(this->faction, WEAPON_HOMING_TARGETS,
	select_pos, WEAPON_HOMING_TARGET_RADIUS_SQR);

  size_t target_size = targets->size();
  if(target_size > 0)
  {
    this->ai_target = (*targets)[rand() % target_size].second;
  }
  else
  {
    this->ai_target = NULL;
  }

  // Not needed anymore.
  delete targets;
}

/** Tell if this bullet needs drawing. Also translate it's middlepoint in
 * order for it to be rendered correctly (relative to viewport).
 * @param view Camera position.
 * @param maxdist Maximum size distance in this model.
 */
bool Munition::inside(const libfhi::Vector2 &view, float maxdist)
{
  this->tpos = Terrain::instance->get_relative_bounded_position(view,
      this->pos);

  //std::cout << "Tpos is " << tpos << "\n";

  return Game::inside_view_area(this->tpos, maxdist);

  return true;
}

/** Move forward and decrement life.
 * @return True if the bullet still exists, false if not.
 */
bool Munition::tick_bullet()
{
  if(this->lifetime-- <= 0)
  {
    return false;
  }

  Terrain *terrain = Terrain::instance;

  this->pos.xf = libfhi::congr(this->pos.xf + this->mov.xf,
      terrain->get_size_x());
  this->pos.yf = libfhi::congr(this->pos.yf + this->mov.yf,
      terrain->get_size_y());

  return true;
}

/** Move forward and decrement life.
 * @return True if the bullet still exists, false if not.
 */
bool Munition::tick_homing(float thurst, float drag_v, float drag_h,
    float mass, float maxv, int rotspeed)
{
  if(this->lifetime-- <= 0)
  {
    return false;
  }

  // If there is no target, try to reacquire.
  if((this->ai_target) && !Game::instance->entity_exists(this->ai_target))
  {
    this->homing_select_target();
  }

  // Try to rotate only if this has a target.
  if(this->ai_target)
  {
    // Get destination orientation.
    libfhi::Vector2 ddir = Terrain::instance->get_relative_bounded_position(
	this->pos, this->ai_target->get_pos());

    // Get lenght and potentially rotate.
    float len = ddir.length();
    int cang = libfhi::pi2int(atan2f(this->dir_unit.yf, this->dir_unit.xf)),
    	dang;

    if(len > 0.0f)
    {
      ddir /= len;
      dang = libfhi::pi2int(atan2f(ddir.yf, ddir.xf));
    }
    else
    {
      dang = cang;
    }

    // Get new angle.
    dang = Entity::ai_try_rotate(cang, dang, rotspeed);

    // Set new angle.
    float pi = libfhi::uint2pi(static_cast<uint16_t>(dang));
    this->dir_unit.set(cosf(pi), sinf(pi));
  }

  // Get thurst (always forward)
  libfhi::Vector2 tv = this->dir_unit * thurst;

  // And apply physics so this moves forward.
  Entity::apply_simple_physics(this->pos, this->mov, this->acc, tv,
      this->dir_unit.xf, this->dir_unit.yf, drag_v, drag_h, mass, maxv);

  return true;
}

/** Move forward and decrement life. This should return false most of the
 * time, since instants do not live for long. In theory, the life time of
 * all instant weapons should be 1.
 * @return True if the bullet still exists, false if not.
 */
bool Munition::tick_instant()
{
  return ((this->lifetime--) > 0);
}

/** Transform this by it's given cr and sr and translate it. This translation
 * is done only over the x/y plane and rotated over the z axis.
 * @param src Source vectors (position).
 * @param dst Destination vectors (position).
 * @param cnt Count of vectors inside this.
 */
void Munition::transform(libfhi::Vector3 *src, libfhi::Vector3 *dst,
    size_t cnt) const
{
  do {
    dst->xf = src->xf * this->dir_unit.xf - src->yf * this->dir_unit.yf +
      this->tpos.xf;
    dst->yf = src->xf * this->dir_unit.yf + src->yf * this->dir_unit.xf +
      this->tpos.yf;
    ++src;
    ++dst;
  } while(--cnt);
}

//############################################################################
// End #######################################################################
//############################################################################

#endif

